
############################ GENERAL UTILITY FUNCTIONS ##############################################

# Using a zoo object of ratios (X_t/X_(t-1)), compute the level of an index that starts at 1
# and then grows according to the relatives
compute_index_level <- function(relatives) {
  cumulative <- cumprod(na.trim(relatives, sides = "both"))
  if(length(cumulative)==0) {return(NA)}
  if(class(index(cumulative))=="yearmon") {
    zooreg(c(1, cumulative), order.by = c(first(index(cumulative))-1/12, index(cumulative)))
  } else if (class(index(cumulative))=="Date") {
    zooreg(c(1, cumulative), order.by = c(first(index(cumulative))-1, index(cumulative)))
  }
}

############################# COMPUTE FIXED-WEIGHT EER RELATIVES FOR A REPORTER ################################

# Compute the fixed EER relatives using a chosen vintage of weights for a chosen set of dates
compute_fixed_EER_relatives <- function(weights_for_reporter, relatives_dates, reporter, log_fx) {
  ### SPECIFY DATES
  # We want to compute an EER relative on each date in the weight reference period, t = 1,..., T
  # To do this, we need EER levels on each date in the weight reference period and one day earlier, t = 0, 1,..., T
  if(class(relatives_dates$start)=="Date") {
    levels_dates <- list(start = relatives_dates$start - 1, end = relatives_dates$end)
  } else if (class(relatives_dates$start)=="yearmon") {
    levels_dates <- list(start = relatives_dates$start - 1/12, end = relatives_dates$end)
  } else if (class(relatives_dates$start)=="yearqtr") {
    levels_dates <- list(start = relatives_dates$start - 1/4, end = relatives_dates$end)
  }
  
  ### EXTRACT DATA
  # Extract FX data for the period of interest only. Pad with NAs if required
  log_fx_window <- window(do.call(merge.zoo, log_fx), start = levels_dates$start, end = levels_dates$end)
  log_fx_padded <- as.list(merge.zoo(log_fx_window, zooreg(NA, start = levels_dates$start, end = levels_dates$end))[ , names(log_fx_window)])
  
  ### COMPUTE NUMERATOR
  fixed_EER_numerator_zoo <- exp(log_fx_padded[[reporter]])
  
  ### COMPUTE DENOMINATOR
  # To compute the denominator of the fixed EERs, we start by listing our chosen partners
  # We use all countries that have log exchange rates on all required dates AND have weights
  partners_with_fx <- names(Filter(function(x) all(!is.na(x)), log_fx_padded[names(log_fx_padded)!=reporter]))
  partners_with_weights <- weights_for_reporter$Partner.Country.Name
  partners <- intersect(partners_with_fx, partners_with_weights)
  
  # Extract the weights for the partners specifically
  weights_df <- weights_for_reporter[weights_for_reporter$Partner.Country.Name %in% partners, c("1970M1", "Partner.Country.Name")]
  weights_vec <- weights_df$`1970M1`
  names(weights_vec) <- weights_df$Partner.Country.Name
  
  # If some countries that had weights were missing FX data, then the weights will sum to less than 1
  # If only a minority of partners by weight are available, we can't compute a reasonably accurate EER, so should return NULL
  # Otherwise, re-scale the weights so they sum to 1
  if(sum(weights_vec) < 50) {return(NULL)}
  scaled_weights_vec <- weights_vec/sum(weights_vec)
  
  # Scale each country's FX by their weight
  log_fx_weighted <- do.call(merge.zoo, map2(.x = log_fx_padded[partners], .y = scaled_weights_vec, function(x, y) x*y))
  
  # The denominator is the sum across countries of the scaled FX
  fixed_EER_denominator_vec <- exp(rowSums(log_fx_weighted))
  fixed_EER_denominator_zoo <- zooreg(fixed_EER_denominator_vec, order.by = index(log_fx_weighted))
  
  ### COMPUTE EER RELATIVES
  
  # Compute the fixed EER levels as numerator (i.e. exp(log(fx)) over denominator
  fixed_EER_levels_zoo <- fixed_EER_numerator_zoo/fixed_EER_denominator_zoo
  
  # Compute the fixed EER relatives on all dates in the weight reference period
  return(fixed_EER_levels_zoo/zoo:::lag.zoo(fixed_EER_levels_zoo, k = -1))
}

################################# COMPUTE CHAINED EER FOR A REPORTER ###########################################

compute_chained_eer_for_reporter <- function(reporter, log_fx, known_weights_vintages) {
  ### Extract the reporter's weights. If there are none, we can't compute an EER so return NULL
  weights_vintages_reporter <- Filter(function(x) nrow(x) > 0, map(known_weights_vintages, function(x) x[x$Reporting.Country.Name == reporter, ]))
  if(length(weights_vintages_reporter)==0) {return(NULL)}

  # We will compute the chained EER based on chained EER relatives
  # We compute the chained EER relatives by concatenating fixed-weight EER relatives for different periods
  # The fixed weight EER relative on a date t will be computed with the weights for the weight reference period that includes t
  # If the date t is after the latest weight reference period, just use the latest weight reference period
  index_class <- class(index(log_fx[[reporter]]))
  extract_start_end <- function(weights_vintage_name) {
    start_year <- substr(weights_vintage_name, start = 1, stop = 4)
    end_year <- substr(weights_vintage_name, start = 6, stop = 9)
      if(index_class == "Date") {
      start <- as.Date(paste0(start_year, "-01-01", format = "%Y-%m-%d"))
      end <- as.Date(paste0(end_year, "-12-31", format = "%Y-%m-%d"))
    } else if (index_class == "yearmon") {
      start <- as.yearmon(paste0("Jan ", start_year))
      end <- as.yearmon(paste0("Dec ", end_year))
    } else if (index_class == "yearqtr") {
      start <- as.yearqtr(paste0(start_year, " Q1"))
      end <- as.yearqtr(paste0(end_year, " Q4"))
    }
    list(start = start, end = end)
  }
  relatives_dates_list <- map(names(weights_vintages_reporter), extract_start_end)
  names(relatives_dates_list) <- names(weights_vintages_reporter)
  relatives_dates_list[[length(relatives_dates_list)]]$end <- last(index(log_fx[[reporter]]))
  
  # For the 1990 to 1995 weights, I will separately compute relatives for '1 Jan 1990 to 1 Jan 1993' and for '2 Jan 1993 to 31 Dec 1995'.
  # This implies that EERs during either of these periods will be based on all partners with exchange rate data during that period,
  # rather than only those partners with exchange rate data for all of 1990 to 1995
  # This is useful, as the number of countries with NER data is greater after 1 Jan 1993 (when I use IMF data) than before (when I use Eikon NER data)
  if ("1990_1995" %in% names(weights_vintages_reporter)) {
    position <- which("1990_1995" == names(weights_vintages_reporter))
    weights_vintages_reporter <- append(weights_vintages_reporter, list(weights_vintages_reporter$`1990_1995`), after = position)
    names(weights_vintages_reporter)[position + 1] <- "1990_1995_part_two"
    relatives_dates_list <- append(relatives_dates_list, list(relatives_dates_list$`1990_1995`), after = position)
    names(relatives_dates_list)[position + 1] <- "1990_1995_part_two"
    if(index_class == "Date") {
      relatives_dates_list$`1990_1995`$end <- as.Date("1993-01-01")
      relatives_dates_list$`1990_1995_part_two`$start <- as.Date("1993-01-02")
    } else if (index_class == "yearmon") {
      relatives_dates_list$`1990_1995`$end <- as.yearmon("Jan 1993")
      relatives_dates_list$`1990_1995_part_two`$start <- as.yearmon("Feb 1993")
    } else if (index_class == "yearqtr") {
      relatives_dates_list$`1990_1995`$end <- as.yearqtr("1993 Q1")
      relatives_dates_list$`1990_1995_part_two`$start <- as.yearqtr("1993 Q2")
    }
  }
  
  # Compute fixed EER relatives on the specified dates
  fixed_EER_relatives <- map2(.x = weights_vintages_reporter,
                              .y = relatives_dates_list,
                              .f = function(x, y) compute_fixed_EER_relatives(weights_for_reporter = x, relatives_dates = y, reporter = reporter, log_fx = log_fx))
  names(fixed_EER_relatives) <- names(weights_vintages_reporter)
  
  # We want our EER levels to be based on a contiguous period of EER relatives with no gaps
  null_weight_periods <- which(unlist(map(fixed_EER_relatives, is.null)))
  if(length(null_weight_periods)==0) {
    # If none of the fixed EERs are NULL, use all of them
    fixed_EER_relatives_filter <- fixed_EER_relatives
  } else {
    # If some of the fixed EERs are NULL, use the fixed EERs after the last NULL
    if(max(null_weight_periods) == length(fixed_EER_relatives)) {return(NULL)}
    fixed_EER_relatives_filter <- fixed_EER_relatives[(max(null_weight_periods) + 1):length(fixed_EER_relatives)]
  }
  
  # Concatenate the fixed EER relatives for different periods, except those that are NULL
  chained_EER_relatives <- do.call(c, fixed_EER_relatives_filter)

  # Use the EER relatives to compute the EER level
  compute_index_level(chained_EER_relatives)
}

############################## COMPUTE CHAINED EER FOR ALL REPORTERS ###########################################

compute_chained_eer_for_vintage <- function(vintage_name, weights_vintages, weights_lag_in_months, read_fx_directory) {
  # Import EDNA CPI for the vintage, and then take logs (# Check Guinea)
  fx <- readRDS(paste0(read_fx_directory, vintage_name, ".rds"))
  log_fx <- map(fx, log)
  
  # Determine which weights were known at the time
  # If the vintage is before the publication of the earliest weights (which are 1979 to 1989), we will be unable to compute EERs with real-time information, so we return NULL
  # For a named set of weights, compute the years to which the weights relate
  publication_months <- map(names(weights_vintages), function(x) as.yearmon(paste0("Dec ", substr(x, start = 6, stop = 9))) + weights_lag_in_months/12)
  known_weights_vintages <- weights_vintages[as.yearmon(vintage_name) >= publication_months]
  if(length(known_weights_vintages)==0) {return(NULL)}
  
  # Compute a chained EER for the reporter
  reporters <- names(fx)
  EERs <- map(reporters, compute_chained_eer_for_reporter, log_fx = log_fx, known_weights = known_weights_vintages)
  names(EERs) <- reporters
  return(remove_null(EERs))
}
