
rm(list = ls())

library(tidyverse)
library(zoo)
library(openxlsx)

source('Code/Import_functions/import_country_facts.R')

source('Code/Tidy_functions/remove_null.R')
source('Code/Tidy_functions/change_nesting_order.R')

######################### TIDY EXCHANGE RATES ########################################################

enforce_min_years <- function(z, min_years) {
  if(class(index(z))=="Date") {
    years <- (last(index(z)) - first(index(z)) + 1)/365.25
  } else if (class(index(z))=="yearmon" | class(index(z))=="yearqtr") {
    years <- last(index(z)) - first(index(z))
  }
  if(years < min_years) {return(NULL)} else {return(z)}
}
map_enforce_min_years <- function(z_list, min_years) {
  if(any(!unlist(map(z_list, is.zoo)))) {stop("Every element of z_list must be a zoo object")}
  remove_null(map(z_list, enforce_min_years, min_years = min_years))
}
tidy_fx <- function(read_directory) {
  # Read in the vintages of EERs
  vintages_country <- map(list.files(read_directory), function(x) remove_null(readRDS(paste0(read_directory, x))))
  names(vintages_country) <- gsub(".rds", "", list.files(read_directory))
  
  # For EERs that will be used as inputs into forecasting models, remove any series that are less than 2 months long
  # For EERs that will be used as actuals, we do not remove any series
  if(!grepl("actual", read_directory)) {vintages_country <- remove_null(map(vintages_country, function(x) map_enforce_min_years(x, min_years = 1)))}
  
  # Change nesting order, then save as RDS files for Martin's code
  country_vintages <- convert_to_country_vintages(vintages_country)
  write_directory <- gsub("RDS_by_vintage", "RDS_by_country", read_directory)
  walk(names(country_vintages), function(x) saveRDS(country_vintages[[x]], paste0(write_directory, x, ".rds")))
}

### Bilateral RER
tidy_fx(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_vintage/Month_avg_Bilateral_RER/")
tidy_fx(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_vintage/Month_end_Bilateral_RER/")
tidy_fx(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_vintage/Daily_Bilateral_Linear_RER/")
tidy_fx(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_vintage/Daily_Bilateral_Step_RER/")

### Bilateral NER
tidy_fx(read_directory = "Mid_data/FX/Bilateral_NER/RDS_by_vintage/Month_avg_Bilateral_NER/")
tidy_fx(read_directory = "Mid_data/FX/Bilateral_NER/RDS_by_vintage/Month_end_Bilateral_NER/")
tidy_fx(read_directory = "Mid_data/FX/Bilateral_NER/RDS_by_vintage/Daily_Bilateral_NER/")

### REERs
tidy_fx(read_directory = "Mid_data/FX/Step_REER/RDS_by_vintage/Daily_Step_REER/")
tidy_fx(read_directory = "Mid_data/FX/Step_REER/RDS_by_vintage/Month_avg_of_Daily_Step_REER/")
tidy_fx(read_directory = "Mid_data/FX/Step_REER/RDS_by_vintage/Month_end_of_Daily_Step_REER/")

tidy_fx(read_directory = "Mid_data/FX/Step_REER/RDS_by_vintage/Daily_Step_REER_actuals/")
tidy_fx(read_directory = "Mid_data/FX/Step_REER/RDS_by_vintage/Month_avg_of_Daily_Step_REER_actuals/")

### NEERs
tidy_fx(read_directory = "Mid_data/FX/NEER/RDS_by_vintage/Daily_NEER/")
tidy_fx(read_directory = "Mid_data/FX/NEER/RDS_by_vintage/Month_avg_of_Daily_NEER/")
tidy_fx(read_directory = "Mid_data/FX/NEER/RDS_by_vintage/Month_end_of_Daily_NEER/")

tidy_fx(read_directory = "Mid_data/FX/NEER/RDS_by_vintage/Daily_NEER_actuals/")
tidy_fx(read_directory = "Mid_data/FX/NEER/RDS_by_vintage/Month_avg_of_Daily_NEER_actuals/")

######################### CONVERT TO CSV FILES FOR STEPHEN ########################################

country_facts <- import_country_facts("Raw_data/Country_facts/country_facts.xlsx")
convert_to_iso <- function(country_name) {
  row <- country_facts[country_facts$country == country_name, ]
  if(nrow(row)==1) {return(row$iso)} else {stop(paste0(country_name, " appears ", nrow(row), " times in country_facts"))}
}
convert_to_csv <- function(read_directory) {
  country_vintages <- map(list.files(read_directory), function(x) readRDS(paste0(read_directory, x)))
  names(country_vintages) <- map(gsub(".rds", "", list.files(read_directory)), convert_to_iso)
  save_to_CSV_for_iso <- function(country_iso) {
    vintages_list <- country_vintages[[country_iso]]
    vintages_zoo <- do.call(merge.zoo, vintages_list[order(as.yearmon(names(vintages_list)))])
    vintages_df <- data.frame(Date = index(vintages_zoo), vintages_zoo)
    write_directory <- gsub("RDS_by_country", "CSV_by_country", read_directory)
    write.csv(vintages_df, file = paste0(write_directory, country_iso, ".csv"), row.names = F)
  }
  walk(names(country_vintages), save_to_CSV_for_iso)
}

### Bilateral RER
convert_to_csv(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_country/Month_avg_Bilateral_RER/")
convert_to_csv(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_country/Month_end_Bilateral_RER/")
convert_to_csv(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_country/Daily_Bilateral_Linear_RER/")
convert_to_csv(read_directory = "Mid_data/FX/Bilateral_RER/RDS_by_country/Daily_Bilateral_Step_RER/")

### Bilateral NER
convert_to_csv(read_directory = "Mid_data/FX/Bilateral_NER/RDS_by_country/Month_avg_Bilateral_NER/")
convert_to_csv(read_directory = "Mid_data/FX/Bilateral_NER/RDS_by_country/Month_end_Bilateral_NER/")
convert_to_csv(read_directory = "Mid_data/FX/Bilateral_NER/RDS_by_country/Daily_Bilateral_NER/")

### REER
convert_to_csv(read_directory = "Mid_data/FX/Step_REER/RDS_by_country/Daily_Step_REER/")
convert_to_csv(read_directory = "Mid_data/FX/Step_REER/RDS_by_country/Month_avg_of_Daily_Step_REER/")
convert_to_csv(read_directory = "Mid_data/FX/Step_REER/RDS_by_country/Month_end_of_Daily_Step_REER/")

convert_to_csv(read_directory = "Mid_data/FX/Step_REER/RDS_by_country/Daily_Step_REER_actuals/")
convert_to_csv(read_directory = "Mid_data/FX/Step_REER/RDS_by_country/Month_avg_of_Daily_Step_REER_actuals/")

### NEER
convert_to_csv(read_directory = "Mid_data/FX/NEER/RDS_by_country/Daily_NEER/")
convert_to_csv(read_directory = "Mid_data/FX/NEER/RDS_by_country/Month_avg_of_Daily_NEER/")
convert_to_csv(read_directory = "Mid_data/FX/NEER/RDS_by_country/Month_end_of_Daily_NEER/")

convert_to_csv(read_directory = "Mid_data/FX/NEER/RDS_by_country/Daily_NEER_actuals/")
convert_to_csv(read_directory = "Mid_data/FX/NEER/RDS_by_country/Month_avg_of_Daily_NEER_actuals/")
