
rm(list = ls())

library(tidyverse)
library(zoo)
library(openxlsx)
library(lubridate)
library(stargazer)

source('Code/Import_functions/import_country_facts.R')
source('Code/Import_functions/import_weo.R')
source('Code/Import_functions/import_imf_ner.R')
source('Code/Import_functions/import_eikon_ner.R')
source('Code/Import_functions/import_iso.R')
source('Code/Import_functions/import_EAER.R')

source('Code/Tidy_fx_functions/convert_EAER_ccy_names_to_iso.R')
source('Code/Tidy_fx_functions/transform_ner.R')
source('Code/Tidy_fx_functions/trim_partial_period.R')
source('Code/Tidy_fx_functions/splice_imf_eikon_ner.R')

source('Code/plotting_functions/ggsave_functions.R')

################################### IMPORT NOMINAL EXCHANGE RATES ###############################################

### Import country facts
# In this script, I often need to refer to countries by name.
country_facts <- import_country_facts("Raw_data/Country_facts/country_facts.xlsx")

### Import bilateral NERs from the IMF
imf_ner <- fill_gaps(import_imf_ner("Raw_data/GDS_data/ENDA_EER/", country_facts = country_facts))

### Import bilateral NERs from Eikon. I may switch to using these later
eikon_ner <- fill_gaps(import_eikon_ner("Raw_data/Eikon_data/", make_colnames_ccynames = F))

################################### IMPORT CURRENCY METADATA ###############################################

### Import ISO data
# ISO-4217 is a standard that associates a 3-digit alphabetic code with each country
# List one contains current currencies, while list three contains discontinued currencies
# The tidy function make the country names consistent with those in country_facts, and in the case of iso_three, make the withdrawal dates more precise
iso_one <- tidy_iso_one(read.xlsx(xlsxFile = "Raw_data/ISO_data/ISO-4217-list-one.xlsx", startRow = 4, colNames = T, rowNames = F))
iso_three <- tidy_iso_three(read.xlsx(xlsxFile = "Raw_data/ISO_data/ISO-4217-list-three.xlsx", startRow = 4, colNames = T, rowNames = F))

### Import IMF exchange arrangements data on which currency the country used in each year
# Countries are identified by the country_names in country facts, while currencies are identified with their ISO codes
EAER_currency <- import_EAER(path = "Raw_data/EAER_data/EAER_currency.xlsx", country_facts = country_facts, variable = "currency")
EAER_currency_iso <- convert_to_iso(EAER_currency = EAER_currency, iso_one = iso_one, iso_three = iso_three)

################################### PERFORM SPLICING ###############################################

### Guess currencies in use before the 1993 start of IMF data
pre_1993_currencies <- compute_pre_1993_currencies(iso_three = iso_three, EAER_currency_iso = EAER_currency_iso)

### Splice IMF and Eikon data for each country, where possible
spliced_ner <- map(names(imf_ner), splice_imf_eikon, pre_1993_currencies = pre_1993_currencies, imf_ner = imf_ner, eikon_ner = eikon_ner)
names(spliced_ner) <- names(imf_ner)

### Use spliced series if available and IMF series otherwise
imf_or_spliced_ner_d <- map2(spliced_ner, imf_ner, function(x, y) if(is.zoo(x)) {return(x)} else {return(y)})

### Extract daily NERs up to end Sep 2022, which is the last month with IMF NERs for the whole month

# For subsequent months, we'd only have Eikon NERs. If we retained these, we'd end up computing EERs for Oct 2022 onwards
# that are based on Eikon NERs but no IMF NERs. I don't want to do this, because it'd be silly to compute EERs for these later months
# using an unrepresentative set of countries when the data for other countries exists, it is just absent from our
# particular dataset due to having downloaded IMF NERs in mid-Oct 2022.
ner_d_up_to_Sep2022 <- map(imf_or_spliced_ner_d, function(x) window(x, start = NULL, end = as.Date("2022-09-30", format = "%Y-%m-%d")))

### Save this daily spliced series, along with month-average and quarter-averages of the daily series
ner_transformed <- transform_ner(ner_d_up_to_Sep2022)
saveRDS(ner_transformed, "Mid_data/FX/NER/ner_transformed.rds")

################################# COUNT COUNTRIES WHERE SPLICING WAS POSSIBLE ###############################################

possibilities <- c("No splice as pre-1993 currency not guessed",
                   "No splice as Eikon lacks data on pre-1993 currency",
                   "No splice as Eikon NERs start no earlier than IMF NERs",
                   "No splice as Eikon and IMF differ on overlapping days",
                   "Splice made")
countries_by_possibility <- map(possibilities[1:4], function(x) names(spliced_ner)[spliced_ner == x])
names(countries_by_possibility) <- possibilities[1:4]
countries_by_possibility[[possibilities[5]]] <- names(spliced_ner)[unlist(map(spliced_ner, is.zoo))]

df <- data.frame(Situation = possibilities, `Number of Countries` = unlist(map(countries_by_possibility, length)))
stargazer(df, summary = F, rownames = F)

##################################### PLOT NERs WITH DISCREPANCIES ########################################################

plot_merged <- function(country_name) {
  merged <- merge_imf_eikon(country_name = country_name, pre_1993_currencies = pre_1993_currencies, imf_ner = imf_ner, eikon_ner = eikon_ner)
  if(is.zoo(merged)) {plot_imf_eikon(merged = merged, country_name = country_name)} else {return(merged)}
}
plot_list <- map(countries_by_possibility$`No splice as Eikon and IMF differ on overlapping days`, plot_merged)
names(plot_list) <- countries_by_possibility$`No splice as Eikon and IMF differ on overlapping days`
ggsave_pdf(plot_list$`Gambia, The` + labs(title = "Nominal Exchange Rate of The Gambia"),
           plot_name = "Gambia", directory = "Graphs/FX/NERs/", height = 7, width  = 10)
