
rm(list = ls())

library(zoo)
library(tidyverse)
library(openxlsx)
library(lubridate)

library(parallel)
library(doParallel)

source('Code/Import_functions/import_country_facts.R')
source('Code/Import_functions/import_ins_weights.R')

source('Code/Tidy_functions/change_nesting_order.R')
source('Code/Tidy_functions/remove_null.R')
source('Code/Tidy_fx_functions/compute_EER_parallel.R')
source('Code/Tidy_fx_functions/trim_partial_period.R')
source('Code/Tidy_fx_functions/compute_period_avg_period_end.R')

####################################### IMPORT WEIGHTS ########################################################

country_facts <- import_country_facts("Raw_data/Country_facts/country_facts.xlsx")
weights_vintages <- prepare_ins_weights(country_facts = country_facts)

# We make vintages from Jan 1995 (the month when the 1979-1989 weights become available, given a 5 year lag)
# to Sep 2022 (the last month where IMF has daily NERs for the whole month)
vintage_names <- as.character(seq(from = as.yearmon("Jan 1995"), to = as.yearmon("Sep 2022"), by = 1/12))

######################### COMPUTE CHAINED EERs IN PARALLEL ########################################################

### Initiate the cluster, which is a set of copies of R running in parallel
# Then tell the foreach package that I want to use the parallel back-end with the cluster I just created
cl <- makeCluster(8)
registerDoParallel(cl)

### Compute daily EER inputs
parLapply(cl = cl,
          X = vintage_names,
          fun = compute_chained_eer_for_vintage_wrapper,
          weights_vintages = weights_vintages, weights_lag_in_months = 61,
          read_fx_directory = "Mid_data/FX/Bilateral_NER_CPI/Daily_Bilateral_NER_Daily_Step_CPI/",
          write_EER_directory = "Mid_data/FX/REER/RDS_by_vintage/Daily_Step_REER/")
parLapply(cl = cl,
          X = vintage_names,
          fun = compute_chained_eer_for_vintage_wrapper,
          weights_vintages = weights_vintages, weights_lag_in_months = 61,
          read_fx_directory = "Mid_data/FX/Bilateral_NER/RDS_by_vintage/Daily_Bilateral_NER/",
          write_EER_directory = "Mid_data/FX/NEER/RDS_by_vintage/Daily_NEER/")

### Compute daily EER actuals
parLapply(cl = cl,
          X = names(weights_vintages),
          fun = compute_eer_actuals_wrapper,
          weights_vintages = weights_vintages,
          read_fx_directory = "Mid_data/FX/Bilateral_NER_CPI/Daily_Bilateral_NER_Daily_Step_CPI/",
          write_EER_directory = "Mid_data/FX/REER/RDS_by_vintage/Daily_Step_REER_actuals/")
parLapply(cl = cl,
          X = names(weights_vintages),
          fun = compute_eer_actuals_wrapper,
          weights_vintages = weights_vintages,
          read_fx_directory = "Mid_data/FX/Bilateral_NER/RDS_by_vintage/Daily_Bilateral_NER/",
          write_EER_directory = "Mid_data/FX/NEER/RDS_by_vintage/Daily_NEER_actuals/")

stopCluster(cl)

### Compute month-averages of daily EERs
compute_monthly_EER <- function(read_EER_directory, transform) {
  vintage_names <- gsub(".rds", "", list.files(read_EER_directory))
  transform_vintage <- function(vintage_name) {
    # Extract daily EERs, and compute month-average or month-end series
    daily_EER <- remove_null(readRDS(paste0(read_EER_directory, vintage_name, ".rds")))
    if (transform == "Month_avg") {
      monthly_EER <- map(daily_EER, compute_m_avg)
    } else if (transform == "Month_end") {
      monthly_EER <- extract_end_period_values(daily_EER, desired_index_class = "yearmon")
    }
    
    # Write to disk
    write_EER_directory <- paste0(substr(read_EER_directory, start = 1, stop = regexpr("Daily_", read_EER_directory)-1),
                                  transform,
                                  "_of_",
                                  substr(read_EER_directory, start = regexpr("Daily_", read_EER_directory), stop = 999))
    saveRDS(monthly_EER, paste0(write_EER_directory, vintage_name, ".rds"))
  }
  walk(vintage_names, transform_vintage)
}
compute_monthly_EER(read_EER_directory = "Mid_data/FX/REER/RDS_by_vintage/Daily_Step_REER/", transform = "Month_end")
compute_monthly_EER(read_EER_directory = "Mid_data/FX/REER/RDS_by_vintage/Daily_Step_REER/", transform = "Month_avg")
compute_monthly_EER(read_EER_directory = "Mid_data/FX/REER/RDS_by_vintage/Daily_Step_REER_actuals/",transform = "Month_avg")

compute_monthly_EER(read_EER_directory = "Mid_data/FX/NEER/RDS_by_vintage/Daily_NEER/", transform = "Month_end")
compute_monthly_EER(read_EER_directory = "Mid_data/FX/NEER/RDS_by_vintage/Daily_NEER/", transform = "Month_avg")
compute_monthly_EER(read_EER_directory = "Mid_data/FX/NEER/RDS_by_vintage/Daily_NEER_actuals/", transform = "Month_avg")
