
assign_m_to_d <- function(zoo_m, method) {
  # Construct a blank daily series
  date_range <- seq.Date(from = as.Date(first(index(zoo_m))), to = ceiling_date(as.Date(last(index(zoo_m))), unit = "months") - 1, by = 1)
  NA_d <- zooreg(rep(NA, times = length(date_range)), order.by = date_range)
  
  # Construct a series where each monthly value is assigned to the last day of the month
  if (method == "Step") {
    return(zooreg(unlist(map(as.yearmon(index(NA_d)), function(x) window(zoo_m, start = x, end = x))),
                  order.by = index(NA_d)))
  } else if (method == "EOM") {
    assigned_d <- zoo_m
    convert_yearmon_to_eom <- function(yearmon) {ceiling_date(as.Date(yearmon), unit = "months") - 1}
    index(assigned_d) <- do.call(c, map(index(assigned_d), convert_yearmon_to_eom))
    merged_d <- merge.zoo(NA_d, assigned_d, all = T)
  }
  return(merged_d$assigned_d)
}
interpolate_eom_to_d <- function(zoo_m, method) {
  if (method == "Step") {
    return(assign_m_to_d(zoo_m, method = "Step"))
  } else if (method == "Linear") {
    return(na.approx(assign_m_to_d(zoo_m, method = "EOM")))
  } else if (method == "Spline") {
    return(na.spline(assign_m_to_d(zoo_m, method = "EOM")))
  }
}
interpolate_vintage <- function(vintage_name, read_directory, write_directory, method) {
  # Prepare environment
  library(tidyverse)
  library(zoo)
  library(lubridate)
  source('Code/Tidy_functions/transform_zoo_functions.R')
  source('Code/Interpolation_fx_functions/interpolate_fx.R')
  
  # Interpolate vintage
  cpi_m_log_for_vintage <- readRDS(paste0(read_directory, vintage_name, ".rds"))
  cpi_eom_to_d_log_for_vintage <- map(cpi_m_log_for_vintage, interpolate_eom_to_d, method = method)
  saveRDS(cpi_eom_to_d_log_for_vintage, paste0(write_directory, vintage_name, ".rds"))
}
interpolate_vintages <- function(read_directory, method) {
  write_directory <- paste0("Mid_data/FX/CPI/Daily_CPI/cpi_eom_to_d_log_", method, "/")
  vintage_names <- setdiff(gsub(".rds", "", list.files(read_directory)), gsub(".rds", "", list.files(write_directory)))
  sorted_vintage_names <- vintage_names[order(as.yearmon(vintage_names))]
  
  # Interpolate in parallel
  cl <- makeCluster(detectCores(logical = T)-1)
  registerDoParallel(cl)
  parLapply(cl = cl,
            X = sorted_vintage_names,
            fun = interpolate_vintage,
            read_directory = read_directory, write_directory = write_directory, method = method)
  stopCluster(cl)
}
visualise_interpolate_eom_to_d <- function(vintage_name, country_name, method, cpi_m) {
  zoo_m <- cpi_m[[vintage_name]][[country_name]]
  assigned_d <- assign_m_to_d(zoo_m, method = method)
  interpolated_d <- interpolate_eom_to_d(zoo_m = zoo_m, method = method)
  merged <- merge.zoo(Monthly = assigned_d, Interpolated = interpolated_d)
  window <- window(merged, start = as.Date(as.yearmon(vintage_name) - 1), end = NULL)
  wide <- data.frame(Date = index(window), window)
  long <- pivot_longer(wide, cols = -Date)
  ggplot(wide) +
    geom_line(aes(x = Date, y = Interpolated)) +
    geom_point(aes(x = Date, y = Monthly)) +
    labs(title = paste0(method, " Interpolation of ", vintage_name, " Vintage of Log CPI of ", country_name), y = "")
}
