
import_eikon_ner_sheet <- function(sheet_name, directory) {
  # Import Eikon NER
  NER <- read.xlsx(paste0(directory, "Eikon_NER.xlsx"), colNames = F, rowNames = F, startRow = 2, sheet = sheet_name)
  
  # Convert NERs to a zoo object
  coredata_chr <- NER[3:nrow(NER), 2:ncol(NER)]
  coredata <- do.call(cbind, map(as.list(coredata_chr), as.numeric))
  dates <- as.Date(as.integer(NER[3:nrow(NER), 1]), origin = as.Date("1899-12-30"))
  NER_zoo <- zooreg(coredata, order.by = dates)
  colnames(NER_zoo) <- gsub("=", "", NER[1, 2:ncol(NER)])
  return(NER_zoo)
}
import_eikon_ner <- function(directory, make_colnames_ccynames) {
  ### Import NERs
  sheet_names <- getSheetNames(paste0(directory, "Eikon_NER.xlsx"))
  NER_zoo_by_sheet <- map(sheet_names, import_eikon_ner_sheet, directory = directory)
  merged <- do.call(merge.zoo, NER_zoo_by_sheet)
  
  ### Import NER metadata
  metadata <- read.xlsx(paste0(directory, "Eikon_NER_metadata.xlsx"))
  metadata$RIC <- gsub("=", "", metadata$RIC)
  
  ### Tidy metadata
  
  # Using the metadata, I classify each instrument as foreign currency valued in USD,
  # USD valued in foreign currency, or something else
  clasify_instrument <- function(currency, second_currency) {
    if(is.na(currency) | is.na(second_currency)) {return("Series is not an exchange rate with USD")}
    if(currency == "US Dollar") {return("Value of USD in foreign currency")}
    if(second_currency == "US Dollar") {return("Value of foreign currency in USD")}
    if(currency != "US Dollar" & second_currency != "US Dollar") {return("Series is not an exchange rate with USD")}
  }
  metadata$Classification <- unlist(map2(metadata$Currency, metadata$Second.Currency, clasify_instrument))
  
  # Extract the name of the foreign currency
  extract_currency <- function(metadata_row) {
    if(metadata_row$Classification == "Value of USD in foreign currency") {return(metadata_row$Second.Currency)}
    if(metadata_row$Classification == "Value of foreign currency in USD") {return(metadata_row$Currency)}
    return(NA)
  }
  metadata$Foreign.Currency <- unlist(map(split(metadata, 1:nrow(metadata)), extract_currency))

  # There are some Reuters Identification Codes (RICs) for which the Eikon metadata lists the same currency name
  # This arises due to Eikon using vague or incorrect currency names for countries that changed their currencies, so I correct the currency names
  rename_currency <- function(metadata, RIC, my_currency_name) {
    metadata[metadata$RIC==RIC, "Foreign.Currency"] <- my_currency_name
    return(metadata)
  }
  metadata <- rename_currency(metadata, "TRL", "Old Turkish Lira") %>%
    rename_currency("TRY", "New Turkish Lira") %>%
    rename_currency("SRG", "Surinamese Guilder") %>%
    rename_currency("GHC", "Old Ghanaian Cedi") %>%
    rename_currency("VEB", "Venezuelan Bolivar") %>%
    rename_currency("ZWD", "First Zimbabwean Dollar") %>%
    rename_currency("ZWR", "Third Zimbabwean Dollar") %>%
    rename_currency("ZWL", "Fourth Zimbabwean Dollar")
  
  ### Tidy data
  
  # Compute reciprocal if needed
  compute_reciprocal <- function(z) {zooreg(unlist(map(as.numeric(z), function(x) 1/x)), order.by = index(z))}
  compute_reciprocal_if_needed <- function(RIC) {
    classification <- metadata$Classification[metadata$RIC == RIC]
    if(classification == "Value of foreign currency in USD") {return(merged[ , RIC])}
    if(classification == "Value of USD in foreign currency") {return(compute_reciprocal(merged[ , RIC]))}
    if(classification == "Series is no an exchange rate with USD") {return(NULL)}
  }
  reciprocal_list <- map(colnames(merged), compute_reciprocal_if_needed)
  names(reciprocal_list) <- colnames(merged)
  reciprocal_zoo <- do.call(merge.zoo, Filter(function(x) !is.null(x), reciprocal_list))
  
  # By default, the column names are the Reuters Identification Codes (RICs) of the instruments, which seem to correspond to the ISO codes of the currencies
  # However, the user has the option of making the column names the currency names used by Eikon
  if(make_colnames_ccynames) {
    replace_RIC_with_currency_name <- function(RIC) {metadata[metadata$RIC == RIC, "Foreign.Currency"]}
    colnames(reciprocal_zoo) <- unlist(map(colnames(reciprocal_zoo), replace_RIC_with_currency_name))
  }
  return(reciprocal_zoo)
}
