
# I need to know what currency countries used up to 1993 to be able to link Eikon and IMF NERs
# The earliest year for which the IMF AREAER dataset provides the currency a country is using is 1999
# Hence, I look at the ISO data on discontinued currencies to identify countries that
# appear to have used the same currency in all years up to 1999
# Count how many times a country withdrew a currency before the end of 2002
count_ccys_withdrawn_pre_2004 <- function(country_name, iso_three) {
  # Extract any withdrawn currencies of the country
  withdrawn <- iso_three[(iso_three$ENTITY == country_name), ]
  
  # Extract any currencies withdrawn before the last day of 1999
  withdrawn_pre_2004 <- withdrawn[withdrawn$Withdrawal.Date <= as.Date("2004-12-31"), ]
  return(nrow(withdrawn_pre_2004))
}

compute_pre_1993_currencies <- function(iso_three, EAER_currency_iso) {
  # Count how many times a country withdrew a currency before the end of 2002
  ccys_withdrawn_pre_2004 <- unlist(map(colnames(EAER_currency_iso), count_ccys_withdrawn_pre_2004, iso_three = iso_three))
  names(ccys_withdrawn_pre_2004) <- colnames(EAER_currency_iso)
  
  # Compute the currency used by the country before 1999
  compute_pre_1993_ccy <- function(country_name) {
    # When a country changes currency, it often uses the old and new currencies in parallel
    # for a few years before withdrawing the old currency
    # Hence, for any country that withdrew a currency on or before 2004, there is a possibility
    # they introduced a new currency before 1999
    
    # When a currency may have introduced a currency before 1999, I view its currency as unknown
    # In future work I could try to determine these pre-1999 currencies, but it is no easy to do so
    # based on ISO list three, as some discontinued currencies were not in regular use (e.g. Unidad de Valor Constante)
    if(ccys_withdrawn_pre_2004[[country_name]] > 0) {return(NA)}
    
    # Otherwise, it's safe to assume that before 1999 it used whichever currency the IMF said it was using in 1999
    if(ccys_withdrawn_pre_2004[[country_name]] == 0) {
      return(as.character(window(EAER_currency_iso[ , country_name], start = 1999, end = 1999)))
    }
  }
  pre_1993_currencies <- map(colnames(EAER_currency_iso), compute_pre_1993_ccy)
  names(pre_1993_currencies) <- colnames(EAER_currency_iso)
  return(pre_1993_currencies)
}

### Compare IMF and Eikon data
merge_imf_eikon <- function(country_name, pre_1993_currencies, imf_ner, eikon_ner) {
  # If the country's currency before 1999 is unknown, then we can't splice the Eikon and IMF series
  if(is.na(pre_1993_currencies[[country_name]])) {return("No splice as pre-1993 currency not guessed")}

  # If Eikon lacks data on the country's currency, return NULL
  if(!(pre_1993_currencies[[country_name]] %in% names(eikon_ner))) {return("No splice as Eikon lacks data on pre-1993 currency")}
  
  # Import the IMF and Eikon NERs
  merged <- merge.zoo(IMF = imf_ner[[country_name]], Eikon = eikon_ner[[pre_1993_currencies[[country_name]]]])
  
  # If Eikon has data on the country's currency, but it starts no earlier than IMF data, we don't perform a splice
  first_IMF <- first(index(na.trim(merged$IMF, sides = "left")))
  first_Eikon <- first(index(na.trim(merged$Eikon, sides = "left")))
  if(first_IMF <= first_Eikon) {return("No splice as Eikon NERs start no earlier than IMF NERs")}
  
  return(merged)
}
splice_imf_eikon <- function(country_name, pre_1993_currencies, imf_ner, eikon_ner) {
  # Extract IMF and Eikon data for the chosen country, and scale Eikon data to match IMF mean in overlapping period
  merged <- merge_imf_eikon(country_name = country_name, pre_1993_currencies = pre_1993_currencies, imf_ner = imf_ner, eikon_ner = eikon_ner)
  if(!is.zoo(merged)) {return(merged)}
  
  # Scale the Eikon NER to have a similar mean to the IMF NER
  overlapping <- na.trim(merged, sides = "both", is.na = "any")
  scaling_factor <- mean(overlapping$IMF, na.rm = T)/mean(overlapping$Eikon, na.rm = T)
  merged$Eikon <- merged$Eikon*scaling_factor
  
  # Check that the series are similar
  merged$Discrepancy <- abs((merged$IMF - merged$Eikon)/merged$IMF)
  if(mean(merged$Discrepancy, na.rm = T) >= 0.01) {return("No splice as Eikon and IMF differ on overlapping days")}
  
  # Splice the series
  splice_zoo <- function(zoo_preferred, zoo_backup) {
    map2(zoo_preferred, zoo_backup, function(x, y) if(!is.na(x)) {return(x)} else {return(y)})
  }
  merged$Spliced <- zooreg(unlist(splice_zoo(merged$IMF, merged$Eikon)), order.by = index(merged))
  return(merged$Spliced)
}

plot_imf_eikon <- function(merged, country_name) {
  if(is.null(merged)) {return(NULL)}
  merged$Discrepancy <- NULL
  wide_df <- data.frame(Date = index(merged), merged)
  long_df <- pivot_longer(wide_df, cols = -Date)
  long_df$value <- as.numeric(long_df$value)
  ggplot(long_df) + geom_line(aes(x = Date, y = value, col = name)) +
    labs(title = paste0("Nominal Exchange Rate of ", country_name), y = "Value in USD") +
    guides(col = guide_legend(title = "Source"))
}
