# EAER_currency uses its own names for each currency
# I map these to ISO-3 currency codes so that they can be linked to other data sources that use the ISO-3 codes
convert_ccy_name_to_iso_code <- function(currency_name, iso_one, iso_three) {
  ### Extract the ISO codes of any active or discontinued currencies with the specified name
  iso_one_results <- iso_one[iso_one$Currency == currency_name, "Alphabetic.Code"]
  iso_three_results <- iso_three[iso_three$Historic.currency == currency_name, "Alphabetic.Code"]
  possible_iso_codes <- unique(c(iso_one_results, iso_three_results))
  
  ### Return output
  # If there is exactly one possible ISO code, then this must be the ISO code
  if(length(possible_iso_codes)==1) {return(possible_iso_codes)}
  
  # If there are no codes, tell the user
  if(length(possible_iso_codes)==0) {return("No matching currency names")}
  
  # If there are multiple codes, tell the user the whole set of codes so they can investigate further
  if(length(possible_iso_codes)!=1) {return(do.call(paste, as.list(possible_iso_codes)))}
}

convert_iso_codes_for_country <- function(country_name, EAER_currency, iso_one, iso_three) {
  # If the currency name is in the list of official ISO currency names, use the ISO currency code
  z <- zooreg(unlist(map(EAER_currency[ , country_name], convert_ccy_name_to_iso_code, iso_one = iso_one, iso_three = iso_three)),
              order.by = index(EAER_currency[ , country_name]))
  
  # In some countries, a currency name matches multiple ISO codes
  # This occurs where a country replaces their currency with another higher-valued one (e.g. Zimbabwe in respones to inflation),
  # resulting in a new ISO code but no change in currency name
  # I should handle these cases manually
  if(country_name == "Afghanistan") {
    window(z, start = 1999, end = 2002) <- rep("AFA", 4)
    window(z, start = 2003, end = 2020) <- rep("AFN", 18)
  }
  if(country_name == "Azerbaijan") {
    window(z, start = 1999, end = 2006) <- rep("AYM", 8)
    window(z, start = 2007, end = 2020) <- rep("AZN", 14)
  }
  if(country_name == "Mexico") {
    window(z, start = 1999, end = 2020) <- rep("MXN", 22)
  }
  if(country_name == "Russia") {
    window(z, start = 1999, end = 2020) <- rep("RUB", 22)
  }
  if(country_name == "Turkey") {
    window(z, start = 1999, end = 2005) <- rep("TRL", 7)
    window(z, start = 2006, end = 2020) <- rep("TRY", 15)
  }
  if(country_name == "Turkmenistan") {
    window(z, start = 1999, end = 2008) <- rep("TMM", 10)
    window(z, start = 2009, end = 2020) <- rep("TMT", 12)
  }
  if(country_name == "Zambia") {
    window(z, start = 1999, end = 2012) <- rep("ZMK", 14)
    window(z, start = 2013, end = 2020) <- rep("ZMW", 8)
  }
  if(country_name == "Zimbabwe") {
    window(z, start = 1999, end = 2005) <- rep("ZWD", 7)
    window(z, start = 2006, end = 2020) <- rep("Complex", 15)
  }
  return(z)
}
convert_to_iso <- function(EAER_currency, iso_one, iso_three) {
  converted_list <- map(names(EAER_currency), convert_iso_codes_for_country, EAER_currency = EAER_currency, iso_one = iso_one, iso_three = iso_three)
  names(converted_list) <- names(EAER_currency)
  do.call(merge.zoo, Filter(function(x) !is.null(x), converted_list))
}


