
##################################### CONVERT LEVELS TO CHANGES ############################################

# Calculate percentage changes from levels
# I use stats lag to avoid using dplyr's lag. https://stackoverflow.com/questions/65824081
compute_pch <- function(zoo, k = -1) {(zoo/stats::lag(zoo, k) - 1)*100}
compute_pch_list <- function(zoo_list, k = -1) {Filter(function(x) length(x) > 0, map(zoo_list, compute_pch, k = k))}

# Calculate log changes from levels
compute_logch <- function(zoo) {(log(zoo, base = exp(1)) - log(stats::lag(zoo, - 1), base = exp(1)))*100}

##################################### CONVERT CHANGES TO LEVELS ############################################

convert_pch_to_level_univariate <- function(pch) {
  pricerelatives <- 1 + pch/100
  
  # Calculate cumulated price relatives
  pricerelatives_trim <- na.trim(pricerelatives, sides = "both")
  if(all(is.na(pricerelatives_trim))) {return(NULL)}
  cumulate <- function(period) {prod(window(pricerelatives_trim, start = NULL, end = period), na.rm = F)    }
  cumulated <- unlist(map(index(pricerelatives_trim), cumulate))
  
  # Define an index that starts as 1 and is then extrapolated by cumulated growth rates
  level <- c(1, cumulated)
  
  # Convert to a zoo object
  is.wholenumber <- function(x)  {all(abs(x - round(x)) < .Machine$double.eps^0.5)}
  if(is.wholenumber(index(pch))) {
    return(zooreg(level, order.by = c(first(index(pricerelatives_trim)) - 1, index(pricerelatives_trim))))
  } else if (class(index(pch)) == "yearqtr") {
    return(zooreg(level, order.by = c(first(index(pricerelatives_trim)) - 1/4, index(pricerelatives_trim))))
  } else if (class(index(pch)) == "yearmon") {
    return(zooreg(level, order.by = c(first(index(pricerelatives_trim)) - 1/12, index(pricerelatives_trim))))
  } else {
    stop("convert_pch_to_level_univariate only supports annual, yearqtr and yearmon indexes at present")
  }
}
convert_pch_to_level <- function(pch_list) {map(pch_list, convert_pch_to_level_univariate)}

############################# CONVERTING QUARTERLY LEVELS TO ANNUAL LEVELS ###########################################

# Convert a yearqtr index into a year index
convert_yearqtr_to_year <- function(yearqtr) {as.numeric(substr(as.yearqtr(yearqtr), start = 1, stop = 4))}

convert_q_level_to_yavg_level <- function(q_level) {
  # Remove data for any years that lack all four quarters of data
  is_full_year <- table(convert_yearqtr_to_year(index(q_level)))==4
  full_years <- as.integer(names(is_full_year)[is_full_year])
  trimmed_q_level <- q_level[convert_yearqtr_to_year(index(q_level)) %in% full_years, ]
  
  # Compute yearaverage levels
  aggregate(trimmed_q_level, by = convert_yearqtr_to_year, FUN = mean)
}
convert_q_level_list_to_yavg_level_list <- function(q_level_list) {map(q_level_list, convert_q_level_to_yavg_level)}

convert_q_level_to_a_ye_level <- function(q_level) {
  yearended_level <- q_level[quarter(index(q_level))==4, ]
  index(yearended_level) <- year(index(yearended_level))
  return(yearended_level)
}

############################# CONVERTING MONTHLY LEVELS TO ANNUAL LEVELS ###########################################

convert_m_level_to_q_level <- function(m_level) {
  # Remove data for any quarters that lack all three quarters of data
  is_full_qtr <- table(as.yearqtr(index(m_level)))==3
  full_qtrs <- names(is_full_qtr)[is_full_qtr]
  trimmed_m_level <- m_level[as.yearqtr(index(m_level)) %in% as.yearqtr(full_qtrs)]
  
  # Compute quarter-average levels
  aggregate(trimmed_m_level, by = as.yearqtr, FUN = mean)
}

############################# CONVERTING TWICE-YEARLY LEVELS TO ANNUAL LEVELS ###########################################

convert_bi_level_to_yavg_level <- function(bi_level) {
  is_full_year <- table(convert_yearqtr_to_year(index(bi_level)))==2
  full_years <- as.integer(names(is_full_year)[is_full_year])
  trimmed_bi_level <- bi_level[convert_yearqtr_to_year(index(bi_level)) %in% full_years, ]
  aggregate(trimmed_bi_level, by = convert_yearqtr_to_year, FUN = mean)
}

#################################### CALCULATING RATIOS #################################################

### Calculate ratios of zoo objects
compute_ratio_zoo <- function(numerator_zoo, denominator_zoo, make_percent) {
  countries <- intersect(colnames(numerator_zoo), colnames(denominator_zoo))
  if(make_percent) {
    ratio <- map(countries, function(x) 100*(numerator_zoo[ , x]/denominator_zoo[ , x]))
  } else {
    ratio <- map(countries, function(x) numerator_zoo[ , x]/denominator_zoo[ , x])
  }
  names(ratio) <- countries
  ratio_filtered <- Filter(function(x) !all(is.na(x)), ratio)
  return(na.trim(do.call(merge, ratio_filtered), sides = "both", is.na = "all"))
}

### Calculate, for each vintage, the ratio of two zoo objects of that vintage
compute_ratio_vintages <- function(numerator_vintages, denominator_vintages, make_percent) {
  # I can compute percentages for each vintage that has a numerator and denominator available
  vintage_names <- intersect(names(numerator_vintages), names(denominator_vintages))
  compute_ratio_zoo_for_vintage <- function(x) {compute_ratio_zoo(numerator_vintages[[x]], denominator_vintages[[x]], make_percent)}
  ratio_vintages <- map(vintage_names, compute_ratio_zoo_for_vintage)
  names(ratio_vintages) <- vintage_names
  return(ratio_vintages)
}
