
# Given a zoo object whose columns are countries, make a list of one zoo object for each country
# The zoo objects column names will be countries. To ensure this, I will sometimes include columns labelled "_duplicated"
# These columns can be ignored if I am making the zoo object a list
# Similarly, given a zoo object whose columns are vintages, make a list of one zoo object for each vintages
convert_zoo_to_list <- function(z) {if(is.zoo(z)) {return(as.list(z[ , !grepl("_duplicated", names(z))]))} else {return(z)}}

############################### From vintages_country to country_vintages #################################3

# Make a zoo object with all vintages of forecasts & actuals for that country
# The index has a year and month. Each value refers to the 12 months ending in the specified year and month
# E.g. If UK has a balance of -2.5 in Apr-2016, this means its balance was -2.5 over the 12 months to Apr 2016
collect_vintages_for_country <- function(vintages_country, country_name) {
  ### Extract the vintages and put them in a single zoo object
  extract_country <- function(vintage_list) {
    if(country_name %in% names(vintage_list)) {
      vintage_list[[country_name]]
    } else{
      NA
    }
  }
  vintages_for_country <- map(vintages_country, extract_country)
  
  # I limit myself to vintages that have at least one non-NA value for the country of interest
  vintages_for_country_non_NA <- vintages_for_country[unlist(map(vintages_for_country, function(x) any(!is.na(x))))]
  
  # Some countries may have no non-NA vintages for the variable of interest (e.g. Somalia), in which case we return NULL
  if(length(vintages_for_country_non_NA)==0) {return(NULL)}
  
  # Otherwise, return the list of vintages
  return(vintages_for_country_non_NA)
}

convert_to_country_vintages <- function(vintages_country) {
  # If vintages_country is a list of zoo objects, convert it into a list of lists, as expected by subsequent functions
  vintages_country <- map(vintages_country, convert_zoo_to_list)
  
  # For each country, extract all vintages
  country_names <- Reduce(union, map(vintages_country, names))
  sorted_country_names <- country_names[order(country_names)]
  country_vintages <- map(sorted_country_names, collect_vintages_for_country, vintages_country = vintages_country)
  names(country_vintages) <- sorted_country_names
  return(Filter(function(x) !is.null(x), country_vintages))
}

############################### From country_vintages to vintages_country ###########################################

collect_countries_for_vintage <- function(country_vintages, vintage_name) {
  ### Extract countries and put them in a single zoo object
  extract_vintage <- function(country_list) {
    if(vintage_name %in% names(country_list)) {
      country_list[[vintage_name]]
    } else {
      NA
    }
  }
  countries_for_vintage <- map(country_vintages, extract_vintage)
  
  # I limit myself to those countries that have at least one non-NA value for the vintage of interest
  countries_for_vintage_non_NA <- countries_for_vintage[unlist(map(countries_for_vintage, function(x) any(!is.na(x))))]
  
  # Some vintages may have non-NA countries, in which case we return NULL
  if(length(countries_for_vintage_non_NA)==0) {return(NULL)}
  
  # Otherwise, return the list of countries
  return(countries_for_vintage_non_NA)
}

convert_to_vintages_country <- function(country_vintages) {
  # If country_vintages is a list of zoo objects, convert it into a list of lists, as expected by subsequent functions
  country_vintages <- map(country_vintages, convert_zoo_to_list)
  
  # For each vintage, extract all countries
  vintage_names <- Reduce(union, map(country_vintages, names))
  sorted_vintage_names <- vintage_names[order(as.yearmon(vintage_names))]
  vintages_country <- map(sorted_vintage_names, collect_countries_for_vintage, country_vintages = country_vintages)
  names(vintages_country) <- sorted_vintage_names
  return(vintages_country)
}
