
### ARGUMENTS
# partial_zoo.  A zoo object containing one series. E.g. Mar 1990 to Oct 1995
# first_period  A period to which we want the series to be extrapolated back. Must be of the same class as the index of partial_zoo
# last_period   A period to which we want the series to be extrapolated forward. Must be of the same class as the index of partial_zoo

### VALUE
# A zoo object with the desired index, formed by extrapolating partial_zoo backwards and forwards as needed to fill missing observations

extrapolate_linearly_forward_back <- function(partial_zoo, first_period, last_period) {
  # Remove any leading NAs or trailing NAs, as these interefere with interpolation
  partial_zoo <- na.trim(partial_zoo, sides = "both", is.na = "all")
  
  # If the user doesn't specify first_period, don't extrapolate back. If they don't specify a last_period, don't extrapolate forward.
  if(is.null(first_period)) {first_period <- first(index(partial_zoo))}
  if(is.null(last_period)) {last_period <- last(index(partial_zoo))}
  
  # Generate a zoo object with NAs for missing periods
  if(class(index(partial_zoo))=="Date") {period_length <- 1} else if (class(index(partial_zoo))=="yearmon") {period_length <- 1/12} else if (class(index(partial_zoo))=="yearqtr") {period_length <- 1/4}
  desired_index <- seq(from = first_period, to = last_period, by = period_length)
  full_zoo <- merge.zoo(partial_zoo, zooreg(rep(NA, length(desired_index)), order.by = desired_index))[ , 1]
  
  # Extrapolate forward to the latest desired period
  if(last_period > last(index(partial_zoo))) {
    last_level <- partial_zoo[length(partial_zoo)]
    penultimate_level <- partial_zoo[length(partial_zoo)-1]
    slope <- (as.numeric(last_level) - as.numeric(penultimate_level))/period_length
    extrapolate_forward <- function(t) {as.numeric(last_level) + (t - index(last_level))*slope}
    late_periods <- seq(from = last(index(partial_zoo))+period_length, to = last(desired_index), by = period_length)
    window(full_zoo, start = first(late_periods), end = last(late_periods)) <- unlist(map(late_periods, extrapolate_forward))
  }

  # Extrapolate backwards to the earliest desired period
  if(first_period < first(index(partial_zoo))) {
    first_level <- partial_zoo[1]
    second_level <- partial_zoo[2]
    early_slope <- (as.numeric(second_level) - as.numeric(first_level))/period_length
    extrapolate_backward <- function(t) {as.numeric(first_level) + early_slope*(t - index(first_level))}
    early_periods <- seq(from = first(desired_index), to = first(index(partial_zoo))-period_length, by = period_length)
    window(full_zoo, start = first(early_periods), end = last(early_periods)) <- unlist(map(early_periods, extrapolate_backward))
  }
  return(full_zoo)
}