% Figure 6:  Aggregate Markup with Translog Demand
%  
%   left panel plots closed-form solution for aggregate markup Mu,
%   equation (79) in the text, as a function of sigma*N for various 
%   levels of xi
%   
%   right panel plots locus of xi and sigma*N such that cutoff condition 
%   from equation (78) in the text holds exactly     


clearvars
close all
clc

set(groot, 'DefaultAxesLineWidth', 1.5);
set(groot, 'DefaultLineLineWidth', 3);
set(groot, 'DefaultAxesTickLabelInterpreter','latex'); 
set(groot, 'DefaultLegendInterpreter','latex');
set(groot, 'DefaultAxesFontSize', 22);


%%%%% translog analytics

sigmas       = (0.1:0.1:30)';
xis          = [1/0.25,1/0.15,1/0.10]';

N            = 1;         % mass of firms

Mu           = zeros(numel(sigmas),numel(xis));
sigmastar    = zeros(numel(xis),1); % cutoff sigma for z*=1

for i = 1:numel(xis)

p.xi         = xis(i);    % Pareto tail productivity

%%%%% generalized exponential integral terms

E1 = exp(p.xi)*ExpInt(p.xi,p.xi);
E2 = p.xi*exp(p.xi)*ExpInt(p.xi+1,p.xi);
E3 = p.xi*exp(p.xi)*ExpInt(p.xi+2,p.xi);

Delta = (E2+E3);

test = E1/(1-Delta);   % should equal 1+1/xi

p.sigma      = sigmas; % demand elasticity: match level of markups 

%fprintf('Mu (by exp integrals, conjecture) = %9.4f %9.4f\n',  [1+1/p.xi, test]);

zstar = 1 + (1/p.xi)*log(p.sigma*N)+ (1/p.xi)*log(ExpInt(p.xi,p.xi));
zstar = exp(zstar);

zstar = max(1,zstar);       % cutoff productivity
Gstar = 1 - zstar.^(-p.xi); % mass of firms at cutoff


Mu(:,i)    = 1./(p.sigma*N.*(1-Gstar)*(1-Delta));

sigmastar(i) = 1/E1/N;

end

figure(6)
subplot(1,2,1)
hold on
plot(sigmas,Mu(:,1),'color',rgb('Indigo'))
plot(sigmas,Mu(:,2),'color',rgb('DarkMagenta'))
plot(sigmas,Mu(:,3),'color',rgb('BlueViolet'))
plot(sigmas,1+1./sigmas,'k--','LineWidth',1)
plot(sigmas,(1+1/xis(1))*ones(size(sigmas)),'k--','LineWidth',1)
plot(sigmas,(1+1/xis(2))*ones(size(sigmas)),'k--','LineWidth',1)
plot(sigmas,(1+1/xis(3))*ones(size(sigmas)),'k--','LineWidth',1)
plot(sigmastar(1)*ones(10,1),linspace(0,2,10),'k--','LineWidth',1)
hold off
grid on
box on
axis([0 30 1 1.5])
xlabel('translog $\bar{\sigma} N$','Interpreter','latex')
ylabel('aggregate markup, $\mathcal{M}$','Interpreter','latex')
legend('$1/\xi = 0.25$','$1/\xi = 0.15$','$1/\xi = 0.10$','Interpreter','latex')



xis = (1.01:0.1:20)';
E1s = exp(xis).*ExpInt(xis,xis);

subplot(1,2,2)
hold on
plot(1./E1s,1./xis,'color',rgb('Gray'))
hold off
grid on
box on
axis([0 30 0 0.5])
ylabel('dispersion, $1/\xi$','Interpreter','latex')
xlabel('translog $\bar{\sigma} N$','Interpreter','latex')
text(6,0.45,'choke price binds, $z^*>1$','interpreter','latex','FontSize', 18)
text(2,0.05,'choke price slack, $z^*=1$','interpreter','latex','FontSize', 18)

print -dpng figures/Figure6






