function rnd = slicesampleadj(initial,nsamples,logpdf,width,thin,burnin,widthprob,widthinf)
% This is a modified version of the slice sampling algorithm provided in
% the Statistics toolbox (slicesample.m). The main modification is to
% introduce a 'contaminated' proposal, where the width of the slice is
% occasionally larger than the baseline width. This should improve mixing
% properties when the identified set possesses disconnected regions.
% This version also removes parts of the code that are redundant in the
% current application (e.g. 'stepping out' when sampling from a univariate 
% density).

% Inputs:
% - initial: row vector containing initial value of random sample sequence
% - nsamples: number of samples to be generated
% - logpdf: function handle created using @, taking one argument as input
% and which has the same type and size as initial; defines function that is
% equal to the log target density (up to a constant).
% - width: 'typical' width of slice, either scalar or vector; if scalar,
% all dimensions assumed to have same typical width; if vector, each
% element of vector is typical width of marginal target distribution along
% that dimension.
% - thin: `thinning' parameter; omit every thin-1 out of thin values in the
% generated sequence.
% - burnin: size of `burn-in' sample; omit random samples before this point 
% in sequence.
% - widthprob: probability of using inflated width of hyper-rectangle
% - widthinf: width inflation factor

initial = initial(:)';
 
if (logpdf(initial)==-Inf)
    error(message('stats:slicesample:BadInitial'))
end
%error checks for burnin and thin
if (burnin<0) || burnin~=round(burnin)
    error(message('stats:slicesample:BadBurnin'));
end
if (thin<=0)|| thin~=round(thin)
    error(message('stats:slicesample:BadThin'));
end
if (~isscalar(width) && ~isscalar(initial) && ~isequal(size(width),size(initial))) ...
    || any(width<eps(initial))
    error(message('stats:slicesample:BadWidth'));
end

maxiter = 200; % Limit for number of iterations in 'shrinking in' step
dim = size(initial,2); % Dimension of the distribution
outclass = superiorfloat(initial); % Single or double
rnd = zeros(nsamples,dim,outclass); % Storage for random sample

% Generate some random values up front
e = exprnd(1,nsamples*thin+burnin,1); % Vertical position of slice
RW = rand(nsamples*thin+burnin,dim); % Position of hyper-rectangle
RD = rand(nsamples*thin+burnin,dim); % Point within slice
RWIDTH = rand(nsamples*thin+burnin,1); % Proposal width

x0 = initial; % current value 

% Bool function indicating whether the point is inside the slice.
inside = @(x,th) (logpdf(x) > th); 

width1 = width; % Baseline width
width2 = widthinf*width; % Inflated width

% Update using shrink-in procedure.
for i = 1-burnin:nsamples*thin
    % A vertical level is drawn uniformly from (0,f(x0)) and used to define
    % the horizontal `slice'.
    z = logpdf(x0) - e(i+burnin);

    % A hyper-recatangle with vertices defined by the pairs [xl, xr] and 
    % with side length ('widths') w is randomly positioned around x0.   
    % With probability widthprob, use 'baseline' width, otherwise use
    % inflated width.
    if RWIDTH(i+burnin) > widthprob
        width = width1;
    else
        width = width2;
    end
    r = width.*RW(i+burnin,:); % Randomly position hyper-rectangle
    xl = x0 - r; 
    xr = xl + width; 
    
    % Find new point by sampling uniformly from inside hyper-rectangle.
    xp = RD(i+burnin,:).*(xr-xl) + xl; % Proposal
    
    % Shrink hyper-rectangle if point lies outside slice.
    iter = 0;  
    while(~inside(xp,z))&& iter<maxiter 
        rshrink = (xp>x0);
        xr(rshrink) = xp(rshrink);
        lshrink = ~rshrink;
        xl(lshrink) = xp(lshrink);
        xp = rand(1,dim).*(xr-xl) + xl; % Draw again
        iter = iter+1;
    end
    if iter>=maxiter % It takes too many iterations to shrink in.
             error(message('stats:slicesample:ErrShrinkin')) 
    end
    x0 = xp; % Update current value 
    if i>0 && mod(i,thin)==0 % Burnin and thin
        rnd(i/thin,:) = x0;
    end  
end

end
