function lf = logf(z,restr,phi,Delta)
% Compute log target density for slice sampler.
% Inputs:
% - z: n^2-dimensional vector
% - restr: structure containing information about restrictions
% - phi: structure containing reduced-form parameters
% - Delta: regularisation parameter

vma = phi.vma;
Sigmatrinv = phi.Sigmatrinv;
U = Sigmatrinv*phi.U; % Multiply by Sigmatrinv to avoid doing this repeatedly
sphi = restr.sphi;

hdSign = restr.hdSign;
mHD = size(restr.hdSign,1); % No. of restrictions on historical decomp.

lf = -0.5*(z*z');
n = sqrt(length(z));
z = reshape(z',[n,n]);
[Q,R] = qr(z);
Q = diag(sign(diag(R)))*Q;

% Compute penalisation term corresponding to linear restrictions on Q
I1 = sphi*Q(:);

% Compute penalisation terms corresponding to nonlinear restrictions (i.e.
% on historical decomposition). For example, when restriction on most
% important contributor, this could be 
% |H_{i,j,t,t}| - max_{l \neq j}|H_{i,l,t,t|% 
I2 = zeros(mHD,1);

% Pre-compute q_j*q_j' for j=1,...,n.
QQ = pagemtimes(reshape(Q,[n,1,n]),reshape(Q,[1,n,n]));

for ii = 1:mHD

    % Extract row of hdSign corresponding to ith restriction.
    hdRestr = hdSign(ii,:);

    % Compute contribution of each shock.
    HH = zeros(1,1,n);    

    for hh = 1:hdRestr(4)+1 % Sum contributions over horizons

        HH = HH + pagemtimes(pagemtimes(vma(hdRestr(1),:,hh),QQ),...
            U(:,hdRestr(3)+hh-1));

    end    

    if hdRestr(5) == 1 && hdRestr(6) == 1

        % Type A - most important contributor
        I2(ii) = abs(HH(hdRestr(2))) - max(abs(HH((1:n) ~= hdRestr(2))));

    elseif hdRestr(5) == 1 && hdRestr(6) == -1

        % Type A - least important contributor
        I2(ii) = min(abs(HH((1:n) ~= hdRestr(2)))) - abs(HH(hdRestr(2)));

    elseif hdRestr(5) == 2 && hdRestr(6) == 1

        % Type B - overwhelming contributor
        I2(ii) = abs(HH(hdRestr(2))) - sum(abs(HH((1:n) ~= hdRestr(2))));                

    elseif hdRestr(5) == 2 && hdRestr(6) == -1

        % Type B - negligible contributor
        I2(ii) = sum(abs(HH((1:n) ~= hdRestr(2)))) - abs(HH(hdRestr(2)));

    end

end

I = [I1; I2];

% Evaluate log target density
lf = lf - sum(log_one_plus_e(-I./Delta));

end