function [Q,empty,ess,Qs] = drawQs_ss_chebycheck(restr,phi,opt,options)
% Function draws Q from target density induced by 'soft' sign restrictions.
% This version includes a 'pre-check' for whether the identified set is
% empty given a relaxed set of restrictions.
% Allows for sign, elasticity and narrative restrictions.
% Inputs:
% - restr: structure containing information about restrictions
% - phi: structure containing reduced-form VAR parameters
% - opt: structure containing model information and options

tradSign = restr.tradSign;
elast = restr.elast;
shockSign = restr.shockSign;
Sigmatrinv = phi.Sigmatrinv;
vma = phi.vma;
U = Sigmatrinv*phi.U; % Multiply by Sigmatrinv to avoid doing this repeatedly

Delta = opt.Delta;
qDraws = opt.qDraws;

n = size(Sigmatrinv,1); % Number of variables in VAR
mS = size(tradSign,1); % No. of traditional sign restrictions
mE = size(elast,1); % No. of elasticity restrictions
mSS = size(shockSign,1); % No. of shock-sign restrictions

%% Construct matrix representing linear restrictions on Q.
% Construct matrix representing traditional sign restrictions.
% Restrictions are represented as S(phi)*vec(Q) >= 0.

S = zeros(mS,n^2);

for ii = 1:mS % For each restriction
    
    if tradSign(ii,5) == 1 % Sign restriction on impulse response
    
        S(ii,(tradSign(ii,2)-1)*n+1:tradSign(ii,2)*n) = ...
            vma(tradSign(ii,1),:,tradSign(ii,3)+1)*tradSign(ii,4);
    
    elseif tradSign(ii,5) == 2 % Sign restriction on A0
        
        S(ii,(tradSign(ii,2)-1)*n+1:tradSign(ii,2)*n) = ...
            Sigmatrinv(:,tradSign(ii,2))'*tradSign(ii,4);
        
    end
    
end

% Append sign normalisations (on diagonal elements of A_0).
SN = zeros(n,n^2);
for ii = 1:n
    SN(ii,(ii-1)*n+1:ii*n) = Sigmatrinv(:,ii)';
end
S = [S; SN];

% Construct matrix representing elasticity restrictons.
% Restrictions are represented as ES(phi,m)*vec(Q) >= 0, where m is a 
% vector containing the bounds. Do this by writing restrictions as 
% m*B(k,l) - s*B(i,j) >= 0.
ES = zeros(mE,n^2);

for ii = 1:mE

    ES(ii,(elast(ii,4)-1)*n+1:elast(ii,4)*n) = ...
        elast(ii,6)*vma(elast(ii,3),:,1); % m*B(k,l)
    ES(ii,(elast(ii,2)-1)*n+1:elast(ii,2)*n) = ES(ii,(elast(ii,2)-1)*n+1:elast(ii,2)*n) ...
        -elast(ii,5)*vma(elast(ii,1),:,1); % -s*B(i,j) - first term allows for possibility that j = l

end

% Construct matrix representing shock-sign restrictions.
% Restrictions are represented as SS(phi,U)*vec(Q) >= 0.
SS = zeros(mSS,n^2);

for ii = 1:mSS % For each restriction
    
    SS(ii,(shockSign(ii,1)-1)*n+1:shockSign(ii,1)*n) = ...
        U(:,shockSign(ii,3))'*shockSign(ii,2);
    
end

% Combine coefficient matrices from restrictions.
restr.sphi = [S; ES; SS];
s = size(restr.sphi,1); % Number of linear sign restrictions

%% Check sufficient condition for identified set to be nonempty.
% Based on Chebyshev criterion from Amir-Ahmadi and Drautzburg (2021).
% If there does not exist x > 0 such that S(phi)*x >=0, then identified 
% set must be empty.

% Options for linear programming routine.
lpoptimOptions = optimoptions('linprog');
lpoptimOptions.Display = 'off';

% Find the centre and radius of the largest ball that can be inscribed
% within the intersection of the half-spaces generated by the
% restrictions represented in S(phi) and the unit hypercube.
A = zeros(s+1,n^2+1);
A(1:end-1,2:end) = -restr.sphi;

for ii = 1:(s+1)

    A(ii,1) = norm(A(ii,2:end));

end

A(end,1) = -1; % Add constraint that radius is positive

% Add additional restrictions that ball lies inside unit hypercube (so 
% problem is well defined).
A = [A; [ones(n^2,1), eye(n^2)]; [ones(n^2,1), -eye(n^2)]];

% Problem is to maximise a*x s.t. A*x <= b, with x = (R,c')'.
a = [-1, zeros(1,n^2)]; % 
b = [zeros(s+1,1); ones(2*n^2,1)];
x = linprog(a,A,b,[],[],[],[],lpoptimOptions);

% Check if radius is positive (i.e. set is nonempty).
if x(1) == 0
     
    Q = nan(n);
    Qs = nan(n);
    ess = [];
    empty = 1;
    %fprintf('\n Sufficient condition for nonempty identified set not satisfied\n');
    return % Terminate function
    
end

%% Use slice sampler to draw from target density.
% Find initial value for slice sampler by maximising log target density
% under relatively large value of Delta.
z0 = fminsearch(@(x) -logf(x,restr,phi,opt.Delta_optim),...
    randn([n^2,1])',options);

% Use slice sampler to obtain qDraws draws of Q from target  density.
z = slicesampleadj(z0,qDraws,@(x)logf(x,restr,phi,Delta),opt.width,...
    opt.thin,opt.burn,opt.widthprob,opt.widthinf);
z = reshape(z',[n,n,qDraws]);

%% Use importance sampling to obtain draws from uniform distribution.
% Compute importance weights.
Qs = zeros(n,n,qDraws);
weights = zeros(qDraws,1);
for kk = 1:qDraws
    [Q,R] = qr(z(:,:,kk));
    Qs(:,:,kk) = diag(sign(diag(R)))*Q;
    weights(kk) = iw(Qs(:,:,kk),restr,phi,Delta);
end 

if any(weights > 0) 
    % Resample draws using importance sampling
    inds = randsample(qDraws,qDraws,1,weights);
    Q = Qs(:,:,inds);
    % Compute effective sample size
    ess = 100*((sum(weights)^2)/sum(weights.^2))/qDraws;
    empty = 0;
else
    Q = [];
    ess = [];
    empty = 1;
end

end