% Explore importance of narrative restrictions in driving inferences about
% the effects of shocks in the oil market.
% Impose only restriction on August 1990 event.

clear variables
close all
clc

addpath('auxFunctions');

%% Import data.
% Data are obtained from the replication files to Antolin-Diaz and
% Rubio-Ramirez (2018), available on Juan Antolin-Diaz's website:
% https://sites.google.com/site/juanantolindiaz/
% The variables are:
% 1) Growth in oil production (q); 2) economic activity index (a); and
% 3) log real oil price (p)
% Data run from January 1971 to December 2015
load('Kilian_Data_Updated.mat');

data = data(:,[2, 1, 3]); % Re-order variables

varNames = {'Activity','Production','Price'};
shockNames = {'Demand','Oil-specific','Supply'};

% Declare any exogenous variables (other than the constant; leave empty 
% if none).
exog = [];

%% Options.
opt.p = 24; % Number of lags in VAR
opt.const = 1; % const = 1 if constant in VAR, = 0 otherwise
opt.cumIR = [2]; % Indices of variables for which cumulative impulse response should be used
opt.H = 20; % Terminal horizon for impulse responses
opt.nonEmpty = 1000; % No. of draws from posterior of phi with non-empty identified set
opt.aalpha = 0.68; % Credibility level for credible intervals
opt.dispIter = 10; % Print number of reduced-form draws remaining every dispIter draws

% Set values for Delta in smoothed target density - smaller values = larger
% penalty for violating restrictions
opt.Delta = 1e-5;
opt.Delta_optim = 0.1; % Delta to use when finding initial value with numerical optimisation
opt.width = 1; % Tuning parameter for slice sampler (width of hypercube)
opt.thin = 1; % Thinning parameter - keep every thin-th draw
opt.burn = 0; % Burn-in sample size - discard first burn draws
opt.widthprob = 0.05; % Probability of drawing proposal with inflated width
opt.widthinf = 3; % Inflation factor for proposal

rng(19061987); % Set seed for random number generator

%% Specify restrictions from Antolin-Diaz and Rubio-Ramirez (2018).
% Each row of tradSign contains a vector (i,j,h,s,t) representing a
% particular sign restriction, where t is the type of restriction:
% t = 1: the horizon-h impulse response of the ith to the jth shock 
% is nonnegative (s = 1) or nonpositive (s = -1).
% t = 2: the (ij)th element of A0 is nonnegative (s = 1) or nonpositive 
% (s = -1).
% tradSign should NOT include sign normalisations on A0. These are 
% imposed by default.
restr.tradSign = ...
    [1, 1, 0, 1, 1; % Impact response of activity to demand shock is nonnegative
     2, 1, 0, 1, 1; % Impact response of production to demand shock is nonnegative
     3, 1, 0, 1, 1; % Impact response of oil price to demand shock is nonnegative
     1, 2, 0, -1, 1; % Impact response of activity to oil-specific demand shock is nonpositive
     2, 2, 0, 1, 1; % Impact response of production to oil-specific demand shock is nonnegative
     3, 2, 0, 1, 1; % Impact response of oil price to oil-specific demand shock is nonnegative
     1, 3, 0, -1, 1; % Impact response of activity to supply shock is nonpositive
     2, 3, 0, -1, 1; % Impact response of production to supply shock is nonpositive
     3, 3, 0, 1, 1]; % Impact response of oil price to supply shock is nonnegative

% Each row of elast contains a vector (i,j,k,l,s,m) representing a bound on a
% ratio of impulse responses (an 'elasticity'): s*B(i,j)/B(k,l) <= m,
% where B is the matrix of impact impulse responses, s = 1 or -1 and m is scalar.
% restr.elast = []; % No elasticity restrictions
restr.elast = [2, 1, 3, 1, 1, 0.0258; % Ratio of impact responses of production and oil price following aggregate demand shock <= 0.0258
               2, 2, 3, 2, 1, 0.0258]; % Ratio of impact responses of production and oil price following oil-specific demand shock <= 0.0258

% Each row of shockSign contains a vector (i,s,t) representing the
% shock-sign restriction that the ith shock in period t is >= 0 (s = 1) or
% <= 0 (s = -1). t is equal to the index of the observation.
restr.shockSign = []; % No shock-sign restrictions

% Each row of hdSign contains a vector (i,j,t,h,k,s) representing a
% narrative restriction on the historical decomposition. 
% k represents the class of restriction (Type A or Type B) and s represents
% the type of restriction within the class.
% k = 1: Type A restriction
%  - s = 1:  The jth shock is the 'most important contributor' to the 
%   change in the ith variable between periods t and  t+h.
%  - s = -1: The jth shock is the 'least important contributor' to the
%   change in the ith variable between periods t and t+h.
% k = 2: Type B restriction
%  - s = 1: The jth shock is the 'overwhelming contributor' to the change
%   in the ith variable between periods t and t+h.
% - s = -1: The jth shock is a 'negligible contributor' to the change in
%   the ith variable between periods t and t+h.
% restr.hdSign = []; % No restrictions on historical decomposition
% Consider only restriction that in August 1990 (outbreak of the Persian
% Gulf War), aggregate demand shocks are the least important contributor 
% to the observed unexpected movements in the real price of oil. In other 
% words, the absolute value of the contribution of aggregate demand shocks 
% is smaller than the absolute contribution of any other structural shock.
restr.hdSign = [3, 1, find(dates == datenum(1990,08,01)), 0, 1, -1];

%% Implement robust Bayesian inference as in Giacomini and Kitagawa (2021)
d = 3*3*(opt.H+1); % Dimension of identified set (number of impulse responses)
% Misclassification parameters - misclassification error of at most epsilon
% per cent with probability at least 1-delta per cent.
epsilon = 0.05; 
delta = 0.05;

% Compute lower and upper bounds on number of required draws
ub = min(2*d*log(2*d/delta),exp(1)*(2*d+log(1/delta)))/epsilon;
% Adjust for average effective sample size based on the standard Bayesian
% exercise with 1000 draws of Q
opt.qDraws = ceil(ub/0.82); 
fprintf('Using %d draws to approximate identified sets \n',opt.qDraws);

% Soft sign restrictions (implemented via slice sampling)
svar_ss_rob; 

cd Results
save('OilMarket_rob_restr.mat');
cd ..
