% Illustrate sampling approach using bivariate model.

clear variables
close all
clc

addpath('auxFunctions');

rng(23032021); % Set seed for random number generator

draws = 1e6; % Number of draws of Q
Deltas = [100,0.1,0.01,1e-4]';

opt.thin = 1; % No thinning
opt.burn = 10; % Drop first 10 draws
opt.width = 1; % Width of proposal interval
opt.widthprob = 0.05; % Probability of drawing from proposal with inflated width
opt.widthinf = 3; % Scale inflation factor

% Specify reduced-form parameters (bivariate model with no dynamics)
Sigmatr = [1, 0; -0.5 1]; % Lower triangular Cholesky factor of variance-covariance matrix
Sigmatrinv = eye(2)/Sigmatr;
n = size(Sigmatr,1);

% Specify sign restrictions as S*vec(Q) >= 0
% B(1,1) >= 0, B(2,1) <= 0, B(1,2) >= 0, B(2,2) >=0 where y_t = B*eps_t.
S = zeros(5,4);
S(1:2,1:2) = [Sigmatr(1,:);
              -Sigmatr(2,:)];
S(3:4,3:4) = [Sigmatr(1,:);
              Sigmatr(2,:)];

% Augment sign restrictions with bound on price elasticity of supply
omega_bar = 1; % Bound on price elasticity of supply
S(5,1:2) = -(Sigmatrinv(:,2)'*omega_bar + Sigmatrinv(:,1)');

% Compute identified set for theta
theta_lb = atan(Sigmatr(2,2)/Sigmatr(2,1));
theta_ub = acot(Sigmatr(2,1)/Sigmatr(2,2) - omega_bar*Sigmatr(1,1)/Sigmatr(2,2));

%% Obtain draws of Q satisfying sign restrictions via accept-reject.
Qdraws = zeros([2,2,draws]);

tic
for kk = 1:draws

    flag = 0;
    while flag == 0

        [Q0,~] = qr(randn(n));

        % Impose normalisation of diagonal elements of A_0^{-1} to improve
        % efficiency of sampler.
        Q0 = sign(diag(Sigmatr*Q0))'.*Q0;

        % Check if proposed draw satisfies sign restrictions.
        if all(S*Q0(:) >= 0)
            
            Qdraws(:,:,kk) = Q0;
            flag = 1;

        end

    end

end
runTime = toc;

% Compute theta from draws of Q
thetas_ar = thetaval(Qdraws);

fh = figure;
ah = axes(fh);
histogram(thetas_ar,'Normalization','pdf','EdgeAlpha',0,'FaceColor','blue');
ylims = ylim;
line([theta_lb,theta_lb],ylims,'color','black');
line([theta_ub,theta_ub],ylims,'color','black');
xlabel('$$\theta$$','interpreter','latex');
ylabel('Density');
title('Accept-reject');
xlim([-pi/2,0]);
xticks([-pi/2,-pi/4,0]);
xticklabels({'-\pi/2','-\pi/4','0'});
ah.TickLength = [0.04, 0.04];
ah.YAxis.TickLength = [0, 0];
ah.Box = 'off';
xline(ah,ah.XLim(2));
yline(ah,ah.YLim(2));
set(findall(gcf,'-property','FontSize'),'FontSize',24);

%% Soft sign restrictions approach

runTime_ss = zeros(length(Deltas),1);
thetas_ss = zeros(draws,length(Deltas));
thetas_rs = zeros(draws,length(Deltas));
ess = zeros(length(Deltas),1);

for ii = 1:length(Deltas)

Delta = Deltas(ii);

tic
z0 = randn([4,1]);
z = slicesampleadj(z0,draws,@(x)logtd(x,S,Delta),opt.width,opt.thin,...
    opt.burn,opt.widthprob,opt.widthinf);
z = reshape(z',[2,2,draws]);
% Use importance sampling to obtain draws from uniform distribution
Qdraws_ss = zeros([n,n,draws]);
weights = zeros(draws,1);
for kk = 1:draws
    [R,~] = chol(z(:,:,kk)'*z(:,:,kk));
    Qdraws_ss(:,:,kk) = z(:,:,kk)/R; 
    qq = Qdraws_ss(:,:,kk);
    I = S*qq(:);
    if any(I < 0)
        weights(kk) = 0;
    else
        weights(kk) = 1./exp(-sum(log_one_plus_e(-I./Delta)));
    end
end 

% Resample draws using importance sampling.
inds = randsample(draws,draws,1,weights);
Qdraws_rs = Qdraws_ss(:,:,inds);
runTime_ss(ii) = toc;

% Compute theta from draws of Q
thetas_ss(:,ii) = thetaval(Qdraws_ss); % Slice sampled
thetas_rs(:,ii) = thetaval(Qdraws_rs); % Importance sampled

% Compute effective sample size
ess(ii) = 100*((sum(weights)^2)/sum(weights.^2))/draws;

end

%% Figures
binEdges = (-1.58:0.01:0)';
blue_col = [0 0 255]./255;
red_col = [255 153 153]./255;

% Plot distribution of theta under different values of Delta (before and
% after importance sampling)
cd Figures
for ii = 1:length(Deltas)
    fh = figure;
    ah = axes(fh);
    h1 = histogram(thetas_ss(:,ii),'BinEdges',binEdges,...
        'Normalization','pdf','EdgeAlpha',0,...
        'FaceColor',blue_col);
    hold on;
    h2 = histogram(thetas_rs(:,ii),'BinEdges',binEdges,...
        'Normalization','pdf','EdgeAlpha',0,...
        'FaceColor',red_col);   
    if ii == 1
        ylims = ylim;
    end
    ylim(ylims);
    line([theta_lb,theta_lb],ylims,'color','black');
    line([theta_ub,theta_ub],ylims,'color','black');
    xlim([-pi/2,0]);
    xticks([-pi/2,-pi/4,0]);
    xticklabels({'-\pi/2','-\pi/4','0'});
    ah.TickLength = [0.04, 0.04];
    ah.YAxis.TickLength = [0.04, 0.04];
    title(['$$\Delta =',num2str(Deltas(ii)),'$$'],'interpreter','latex');
    xlabel('$$\theta$$','interpreter','latex');
    ylabel('Density');
    ah.Box = 'off';
    xline(ah,ah.XLim(2));
    yline(ah,ah.YLim(2));
    if ii == 2
    legend([h1, h2],{'Slice','RS'},'Location','NorthEast');
    legend boxoff;
    end
    set(findall(gcf,'-property','FontSize'),'FontSize',24);
    print(['Histogram_',num2str(ii)'],'-depsc');  
end
cd ..

cd Results
save('bivariate.mat');
cd ..

%% Compute log target density for slice sampler
function lf = logtd(z,S,Delta)
    lf = -0.5*(z*z');
    n = sqrt(length(z));
    z = reshape(z',[n,n]);
    [R,~] = chol(z'*z);
    Q = z/R; 
    I = S*Q(:); 
    lf = lf - sum(log_one_plus_e(-I./Delta));
end

%% Function returning value of theta from elements of Q
function theta = thetaval(Q)
% Input can be 2x2xn array
% Function uses that first column of Q is [cos(theta), sin(theta)]';

q1 = squeeze(Q(1,1,:));
q2 = squeeze(Q(2,1,:));
theta = acos(q1).*sign(q2);

end