function [Q,sphi,empty] = drawQ0_ar_chebycheck(restr,phi,opt)
% Function attempts to draw Q from the space of orthonormal matrices 
% satisfying sign, elasticity and narrative restrictions.
% This version includes a 'pre-check' for whether the identified set is
% empty given a relaxed set of restrictions.
% Inputs:
% - restr: structure containing information about restrictions
% - phi: structure containing reduced-form VAR parameters
% - opt: structure containing model information and options

tradSign = restr.tradSign;
elast = restr.elast;
shockSign = restr.shockSign;
hdSign = restr.hdSign;
Sigmatr = phi.Sigmatr;
Sigmatrinv = phi.Sigmatrinv;
vma = phi.vma;
U = Sigmatrinv*phi.U; % Multiply by Sigmatrinv to avoid doing this repeatedly
Sigmatrinvp = Sigmatrinv'; % Transpose to avoid doing this repeatedly

n = size(Sigmatrinv,1); % Number of variables in VAR
mS = size(tradSign,1); % No. of traditional sign restrictions
mE = size(elast,1); % No. of elasticity restrictions
mSS = size(shockSign,1); % No. of shock-sign restrictions
mHD = size(hdSign,1); % No. of restrictions on historical decomp.

%% Construct matrix representing traditional sign restrictions.
% Restrictions are represented as S(phi)*vec(Q) >= 0.

S = zeros(mS,n^2);

for ii = 1:mS % For each restriction
    
    if tradSign(ii,5) == 1 % Sign restriction on impulse response
    
        S(ii,(tradSign(ii,2)-1)*n+1:tradSign(ii,2)*n) = ...
            vma(tradSign(ii,1),:,tradSign(ii,3)+1)*tradSign(ii,4);
    
    elseif tradSign(ii,5) == 2 % Sign restriction on A0
        
        S(ii,(tradSign(ii,2)-1)*n+1:tradSign(ii,2)*n) = ...
            Sigmatrinv(:,tradSign(ii,2))'*tradSign(ii,4);
        
    end
    
end

%% Construct matrix representing elasticity restrictons.
% Restrictions are represented as ES(phi,m)*vec(Q) >= 0, where m is a 
% vector containing the bounds. Do this by writing restrictions as 
% m*B(k,l) - s*B(i,j) >= 0.
ES = zeros(mE,n^2);

for ii = 1:mE

    ES(ii,(elast(ii,4)-1)*n+1:elast(ii,4)*n) = ...
        elast(ii,6)*Sigmatr(elast(ii,3),:); % m*B(k,l)
    ES(ii,(elast(ii,2)-1)*n+1:elast(ii,2)*n) = ES(ii,(elast(ii,2)-1)*n+1:elast(ii,2)*n) ...
        -elast(ii,5)*Sigmatr(elast(ii,1),:); % -s*B(i,j) - first term allows for possibility that j = l

end

%% Construct matrix representing shock-sign restrictions.
% Restrictions are represented as SS(phi,U)*vec(Q) >= 0.
SS = zeros(mSS,n^2);

for ii = 1:mSS % For each restriction
    
    SS(ii,(shockSign(ii,1)-1)*n+1:shockSign(ii,1)*n) = ...
        U(:,shockSign(ii,3))'*shockSign(ii,2);
    
end

%% Add sign normalisation restrictions (diagonal elements of A_0)
SN = zeros(n,n^2);
for ii = 1:n

    SN(ii,(ii-1)*n+1:ii*n) = Sigmatrinv(:,ii)';

end

% Combine coefficient matrices from restrictions.
sphi = [S; ES; SS; SN];
s = size(sphi,1); % Number of linear sign restrictions

%% Check sufficient condition for identified set to be nonempty.
% Based on Chebyshev criterion from Amir-Ahmadi and Drautzburg (2021).
% If there does not exist x > 0 such that S(phi)*x >=0, then identified 
% set must be empty.

% Options for linear programming routine.
lpoptimOptions = optimoptions('linprog');
lpoptimOptions.Display = 'off';

% Find the centre and radius of the largest ball that can be inscribed
% within the intersection of the half-spaces generated by the
% restrictions represented in S(phi) and the unit hypercube.
A = zeros(s+1,n^2+1);
A(1:end-1,2:end) = -sphi;

for ii = 1:(s+1)

    A(ii,1) = norm(A(ii,2:end));

end

A(end,1) = -1; % Add constraint that radius is positive

% Add additional restrictions that ball lies inside unit hypercube (so 
% problem is well defined).
A = [A; [ones(n^2,1), eye(n^2)]; [ones(n^2,1), -eye(n^2)]];

% Problem is to maximise a*x s.t. A*x <= b, with x = (R,c')'.
a = [-1, zeros(1,n^2)]; % 
b = [zeros(s+1,1); ones(2*n^2,1)];
x = linprog(a,A,b,[],[],[],[],lpoptimOptions);

% Check if radius is positive (i.e. set is nonempty).
if x(1) == 0
     
    Q = nan(n);
    empty = 1;
    %fprintf('\n Sufficient condition for nonempty identified set not satisfied\n');
    return % Terminate function
    
end

%% If radius is positive, attempt to draw Q satisfying restrictions.

% Remove sign normalisation restrictions (imposed during sampling).
sphi = sphi(1:end-n,:);

iter = 0;
empty = 1;

while iter <= opt.qMax && empty == 1

    iter = iter + 1;
    
    % Draw Q from space of orthonormal matrices.
    z = randn(n);
    [Qtilde,~] = qr(z);
    
    % Normalise diagonal elements of A0 to be positive. Note that Matlab 
    % is implicitly expanding arrays to be compatible with elementwise 
    % array operations.
    Q = ((sign(diag(Sigmatrinvp*Qtilde))').*Qtilde)./vecnorm(Qtilde);
    
    % Check whether proposed draw satisfies sign, elasticity and shock-sign
    % restrictions.
    sSat = all(sphi*Q(:) >= 0);
       
    if sSat == 0  
        % If restrictions not satisfied, return to beginning of while loop.
        continue;          
    end

    % Check whether proposed draw additionally satisfies restrictions on 
    % historical decomposition.
    hdCheck = zeros(mHD,1); 

    % Pre-compute q_j*q_j' for j=1,...,n.
    QQ = pagemtimes(reshape(Q,[n,1,n]),reshape(Q,[1,n,n]));

    for ii = 1:mHD % For each restriction  

        % Extract row of hdSign corresponding to ith restriction.
        hdRestr = hdSign(ii,:);

        % Compute contribution of each shock.
        HH = zeros(1,1,n);    

        for hh = 1:hdSign(ii,4)+1 % Sum contribution over horizons

            HH = HH + pagemtimes(pagemtimes(vma(hdRestr(1),:,hh),QQ),...
                U(:,hdRestr(3)+hh-1));

        end

        if hdRestr(5) == 1 && hdRestr(6) == 1

            % Type A - most important contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) == max(abs(HH));

        elseif hdRestr(5) == 1 && hdRestr(6) == -1

            % Type A - least important contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) == min(abs(HH));

        elseif hdRestr(5) == 2 && hdRestr(6) == 1

            % Type B - overwhelming contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) - ...
                sum(abs(HH((1:n) ~= hdRestr(2)))) >= 0;                

        elseif hdRestr(5) == 2 && hdRestr(6) == -1

            % Type B - negligible contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) - ...
                sum(abs(HH((1:n) ~= hdRestr(2)))) <= 0;

        end
        
        if hdCheck(ii) == 0              
            % If ith restriction on historical decomposition not 
            % satisfied, exit loop (saves computing contributions 
            % relating to subsequent restrictions).
            continue; % Break out of for loop over each HD restriction
        end

    end 
 
    % If all restrictions satisfied, terminate while loop.
    empty = 1 - all(hdCheck);
        
end

end