function [Q,sphi,empty] = drawQ0_ar(restr,phi,opt)
% Function attempts to draw Q from the space of orthonormal matrices 
% satisfying sign, elasticity and narrative restrictions.
% Inputs:
% - restr: structure containing information about restrictions
% - phi: structure containing reduced-form VAR parameters
% - opt: structure containing model information and options

tradSign = restr.tradSign;
elast = restr.elast;
shockSign = restr.shockSign;
hdSign = restr.hdSign;
Sigmatr = phi.Sigmatr;
Sigmatrinv = phi.Sigmatrinv;
vma = phi.vma;
U = Sigmatrinv*phi.U; % Multiply by Sigmatrinv to avoid doing this repeatedly
Sigmatrinvp = Sigmatrinv'; % Transpose to avoid doing this repeatedly

n = size(Sigmatrinv,1); % No. of variables in VAR
mS = size(tradSign,1); % No. of traditional sign restrictions
mE = size(elast,1); % No. of elasticity restrictions
mSS = size(shockSign,1); % No. of shock-sign restrictions
mHD = size(hdSign,1); % No. of restrictions on historical decomposition

%% Construct matrix representing traditional sign restrictions.
% Restrictions are represented as S(phi)*vec(Q) >= 0.

S = zeros(mS,n^2);

for ii = 1:mS % For each restriction
    
    if tradSign(ii,5) == 1 % Sign restriction on impulse response
    
        S(ii,(tradSign(ii,2)-1)*n+1:tradSign(ii,2)*n) = ...
            vma(tradSign(ii,1),:,tradSign(ii,3)+1)*tradSign(ii,4);
    
    elseif tradSign(ii,5) == 2 % Sign restriction on A0
        
        S(ii,(tradSign(ii,2)-1)*n+1:tradSign(ii,2)*n) = ...
            Sigmatrinv(:,tradSign(ii,2))'*tradSign(ii,4);
        
    end
    
end

%% Construct matrix representing elasticity restrictons.
% Restrictions are represented as ES(phi,m)*vec(Q) >= 0, where m is a 
% vector containing the bounds. Do this by writing restrictions as 
% m*B(k,l) - s*B(i,j) >= 0.
ES = zeros(mE,n^2);

for ii = 1:mE

    ES(ii,(elast(ii,4)-1)*n+1:elast(ii,4)*n) = ...
        elast(ii,6)*Sigmatr(elast(ii,3),:); % m*B(k,l)
    ES(ii,(elast(ii,2)-1)*n+1:elast(ii,2)*n) = ES(ii,(elast(ii,2)-1)*n+1:elast(ii,2)*n) ...
        -elast(ii,5)*Sigmatr(elast(ii,1),:); % -s*B(i,j) - first term allows for possibility that j = l

end

%% Construct matrix representing shock-sign restrictions.
% Restrictions are represented as SS(phi,U)*vec(Q) >= 0.
SS = zeros(mSS,n^2);

for ii = 1:mSS % For each restriction
    
    SS(ii,(shockSign(ii,1)-1)*n+1:shockSign(ii,1)*n) = ...
        U(:,shockSign(ii,3))'*shockSign(ii,2);
    
end

% Combine coefficient matrices from restrictions.
sphi = [S; ES; SS];

%% Attempt to draw Q satisfying restrictions.

iter = 0;
empty = 1;

while iter <= opt.qMax && empty == 1

    iter = iter + 1;
    
    % Draw Q from space of orthonormal matrices.
    z = randn(n);
    [Qtilde,~] = qr(z);
    
    % Normalise diagonal elements of A0 to be positive. Note that Matlab 
    % is implicitly expanding arrays to be compatible with elementwise 
    % array operations.
    Q = ((sign(diag(Sigmatrinvp*Qtilde))').*Qtilde)./vecnorm(Qtilde);
    
    % Check whether proposed draw satisfies sign, elasticity and shock-sign
    % restrictions.
    sSat = all(sphi*Q(:) >= 0);
       
    if sSat == 0  
        % If restrictions not satisfied, return to beginning of while loop.
        continue;          
    end

    % Check whether proposed draw additionally satisfies restrictions on 
    % historical decomposition.
    % Pre-compute q_j*q_j' for j=1,...,n.
    QQ = pagemtimes(reshape(Q,[n,1,n]),reshape(Q,[1,n,n]));
    hdCheck = zeros(mHD,1); 

    for ii = 1:mHD % For each restriction  
    
        % Extract row of hdSign corresponding to ith restriction.
        hdRestr = hdSign(ii,:);

        % Compute contribution of each shock.
        HH = zeros(1,1,n);    

        for hh = 1:hdSign(ii,4)+1 % Sum contribution over horizons

            HH = HH + pagemtimes(pagemtimes(vma(hdRestr(1),:,hh),QQ),...
                U(:,hdRestr(3)+hh-1));

        end

        if hdRestr(5) == 1 && hdRestr(6) == 1

            % Type A - most important contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) == max(abs(HH));

        elseif hdRestr(5) == 1 && hdRestr(6) == -1

            % Type A - least important contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) == min(abs(HH));

        elseif hdRestr(5) == 2 && hdRestr(6) == 1

            % Type B - overwhelming contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) - ...
                sum(abs(HH((1:n) ~= hdRestr(2)))) >= 0;                

        elseif hdRestr(5) == 2 && hdRestr(6) == -1

            % Type B - negligible contributor
            hdCheck(ii) = abs(HH(hdRestr(2))) - ...
                sum(abs(HH((1:n) ~= hdRestr(2)))) <= 0;

        end
        
        if hdCheck(ii) == 0              
            % If ith restriction on historical decomposition not 
            % satisfied, exit loop (saves computing contributions 
            % relating to subsequent restrictions).
            continue; % Break out of for loop over each HD restriction
        end

    end
 
    % If all restrictions satisfied, terminate while loop.
    empty = 1 - all(hdCheck);
        
end

end