% Compare performance of accept-reject and soft sign restrictions in
% empirical setting. Model is oil market VAR in Antolin-Diaz and
% Rubio-Ramirez (2018). Three-variable VAR with sign restrictions on all
% shocks, bounds on 'elasticity' and narrative restrictions.

% This version uses alternative method to check whether identified set is
% empty.

clear variables
close all
clc

addpath('auxFunctions');

%% Import data.
% Data are obtained from the replication files to Antolin-Diaz and
% Rubio-Ramirez (2018), available on Juan Antolin-Diaz's website:
% https://sites.google.com/site/juanantolindiaz/
% The variables are:
% 1) Growth in oil production; 2) economic activity index; and
% 3) log real oil price
% Data run from January 1971 to December 2015
load('Kilian_Data_Updated.mat');

data = data(:,[2, 1, 3]); % Re-order variables

varNames = {'Activity','Production','Price'};
shockNames = {'Demand','Oil-specific','Supply'};

% Declare any exogenous variables (other than the constant; leave empty 
% if none).
exog = [];

%% Options.
opt.p = 24; % Number of lags in VAR
opt.const = 1; % const = 1 if constant in VAR, = 0 otherwise
opt.cumIR = [2]; % Indices of variables for which cumulative impulse response should be used
opt.H = 20; % Terminal horizon for impulse responses
opt.nonEmpty = 1000; % No. of draws from posterior of phi with non-empty identified set
opt.qDraws = 1e3; % Number of draws of Q
opt.aalpha = 0.68; % Credibility level for credible intervals
opt.dispIter = 10; % Print number of reduced-form draws remaining every dispIter draws

% Set values for Delta in smoothed target density - smaller values = larger
% penalty for violating restrictions
opt.Delta = 1e-5;
opt.Delta_optim = 0.1; % Delta to use when finding initial value with numerical optimisation
opt.width = 1; % Tuning parameter for slice sampler (width of hypercube)
opt.thin = 1; % Thinning parameter - keep every thin-th draw
opt.burn = 0; % Burn-in sample size - discard first burn draws
opt.widthprob = 0.05; % Probability of drawing proposal with inflated width
opt.widthinf = 3; % Inflation factor for proposal

rng(19061987); % Set seed for random number generator

%% Specify restrictions from Antolin-Diaz and Rubio-Ramirez (2018).
% Each row of tradSign contains a vector (i,j,h,s,t) representing a
% particular sign restriction, where t is the type of restriction:
% t = 1: the horizon-h impulse response of the ith to the jth shock 
% is nonnegative (s = 1) or nonpositive (s = -1).
% t = 2: the (ij)th element of A0 is nonnegative (s = 1) or nonpositive 
% (s = -1).
% tradSign should NOT include sign normalisations on A0. These are 
% imposed by default.
restr.tradSign = ...
    [1, 1, 0, 1, 1; % Impact response of activity to demand shock is nonnegative
     2, 1, 0, 1, 1; % Impact response of production to demand shock is nonnegative
     3, 1, 0, 1, 1; % Impact response of oil price to demand shock is nonnegative
     1, 2, 0, -1, 1; % Impact response of activity to oil-specific demand shock is nonpositive
     2, 2, 0, 1, 1; % Impact response of production to oil-specific demand shock is nonnegative
     3, 2, 0, 1, 1; % Impact response of oil price to oil-specific demand shock is nonnegative
     1, 3, 0, -1, 1; % Impact response of activity to supply shock is nonpositive
     2, 3, 0, -1, 1; % Impact response of production to supply shock is nonpositive
     3, 3, 0, 1, 1]; % Impact response of oil price to supply shock is nonnegative

% Each row of elast contains a vector (i,j,k,l,s,m) representing a bound on a
% ratio of impulse responses (an 'elasticity'): s*B(i,j)/B(k,l) <= m,
% where B is the matrix of impact impulse responses, s = 1 or -1 and m is scalar.
% restr.elast = []; % No elasticity restrictions
restr.elast = [2, 1, 3, 1, 1, 0.0258; % Ratio of impact responses of production and oil price following aggregate demand shock <= 0.0258
               2, 2, 3, 2, 1, 0.0258]; % Ratio of impact responses of production and oil price following oil-specific demand shock <= 0.0258

% Each row of shockSign contains a vector (i,s,t) representing the
% shock-sign restriction that the ith shock in period t is >= 0 (s = 1) or
% <= 0 (s = -1). t is equal to the index of the observation.
% restr.shockSign = []; % No shock-sign restrictions
restr.shockSign = [3, 1, find(dates == datenum(1978,12,01)); % Supply shock in December 1978 >= 0
                   3, 1, find(dates == datenum(1979,01,01)); % Supply shock in January 1979 >= 0
                   3, 1, find(dates == datenum(1980,09,01)); % Supply shock in September 1980 >= 0
                   3, 1, find(dates == datenum(1980,10,01)); % Supply shock in October 1980 >= 0
                   3, 1, find(dates == datenum(1990,08,01)); % Supply shock in August 1990 >= 0
                   3, 1, find(dates == datenum(2002,12,01)); % Supply shock in December 2002 >= 0
                   3, 1, find(dates == datenum(2003,03,01)); % Supply shock in March 2003 >= 0
                   3, 1, find(dates == datenum(2011,02,01))]; % Supply shock in February 2011 >= 0

% Each row of hdSign contains a vector (i,j,t,h,k,s) representing a
% narrative restriction on the historical decomposition. 
% k represents the class of restriction (Type A or Type B) and s represents
% the type of restriction within the class.
% k = 1: Type A restriction
%  - s = 1:  The jth shock is the 'most important contributor' to the 
%   change in the ith variable between periods t and  t+h.
%  - s = -1: The jth shock is the 'least important contributor' to the
%   change in the ith variable between periods t and t+h.
% k = 2: Type B restriction
%  - s = 1: The jth shock is the 'overwhelming contributor' to the change
%   in the ith variable between periods t and t+h.
% - s = -1: The jth shock is a 'negligible contributor' to the change in
%   the ith variable between periods t and t+h.
% restr.hdSign = []; % No restrictions on historical decomposition
% Oil supply shocks are the most important contributor to the observed
% unexpected movements in oil production growth. In other words, the
% absolute value of the contribution of oil supply shocks is larger than
% the absolute value of the contribution of any other shock.
restr.hdSign = zeros(size(restr.shockSign,1),6);
restr.hdSign(:,1) = 2; % Variable of interest is oil production (second variable)
restr.hdSign(:,2) = 3; % Shock of interest is oil supply shock (third shock)
restr.hdSign(:,3) = restr.shockSign(:,3); % Dates are same as for shock sign restrictions
restr.hdSign(:,4) = 0; % Contributions are with respect to one-step-ahead forecast errors
restr.hdSign(:,5) = 1; % k = 1, so Type A restriction
restr.hdSign(:,6) = 1; % s = 1, so 'most important contributor'
% Add additional restrictions on historical decomposition: for the periods
% corresponding to September-October 1980 (outbreak of the Iran-Iraq War)
% and August 1990 (outbreak of the Persian Gulf War), aggregate demand
% shocks are the least important contributor to the observed unexpected
% movements in the real price of oil. In other words, the absolute value of
% the contribution of aggregate demand shocks is smaller than the absolute
% contribution of any other structural shock.
restr.hdSign = [restr.hdSign;
                3, 1, find(dates == datenum(1980,09,01)), 0, 1, -1;
                3, 1, find(dates == datenum(1980,10,01)), 0, 1, -1; 
                3, 1, find(dates == datenum(1990,08,01)), 0, 1, -1];

%% Soft sign restrictions (implemented via slice sampling)
svar_ss_chebycheck; 

% Compute number of effective draws per hour
effDrawph = mean(ess/100)*opt.qDraws*opt.nonEmpty/(runTime/(60*60));
sprintf('Number of effective draws per hour: %f',effDrawph)

cd Results
save('OilMarket_ss_chebycheck.mat');
cd ..
