% Explore speed and efficiency of sampling algorithms in bivariate model.

clear variables
close all
clc

rng(23032022); % Set seed for random number generator

draws = 1e4; % Number of draws of Q
reps = 100; % Monte Carlo replications
Deltas = [1e-1,1e-2,1e-3,1e-4]';

opt.thin = 1; % No thinning
opt.burn = 10; % Drop first 10 draws
opt.width = 1; % Width of proposal interval
opt.widthprob = 0.05; % Probability of drawing from proposal with inflated width
opt.widthinf = 3; % Scale inflation factor

% Specify reduced-form parameters (bivariate model with no dynamics).
Sigmatr = [1, 0; -0.5 1];
Sigmatrinv = eye(2)/Sigmatr;
n = size(Sigmatr,1);

% Specify sign restrictions as S*vec(Q) >= 0
% B(1,1) >= 0, B(2,1) <= 0, B(1,2) >= 0, B(2,2) >=0 where y_t = B*eps_t.
S = zeros(5,4);
S(1:2,1:2) = [Sigmatr(1,:);
              -Sigmatr(2,:)];
S(3:4,3:4) = [Sigmatr(1,:);
              Sigmatr(2,:)];

% Set up storage arrays
omegas = [1, 0.1, 0.01]';
runTime_ar = zeros(length(omegas),reps);
runTime_ss = zeros(length(omegas),length(Deltas),reps);
ess = zeros(length(omegas),length(Deltas),reps);

for rep = 1:reps

    rep

for ii = 1:length(omegas)

omega_bar = omegas(ii); % Bound on price elasticity of supply

% Augment sign restrictions with bound on price elasticity of supply
S(end,1:2) = -(Sigmatrinv(:,2)'*omega_bar + Sigmatrinv(:,1)');

% Compute identified set for theta
theta_lb = atan(Sigmatr(2,2)/Sigmatr(2,1));
theta_ub = acot(Sigmatr(2,1)/Sigmatr(2,2) - omega_bar*Sigmatr(1,1)/Sigmatr(2,2));

% Obtain draws of Q satisfying sign restrictions via accept-reject.
Qdraws_ar = zeros([2,2,draws]);

tic
for kk = 1:draws

    flag = 0;
    while flag == 0

        [Q0,~] = qr(randn(n));

        % Impose normalisation of diagonal elements of A_0^{-1} to improve
        % efficiency of sampler.
        Q0 = sign(diag(Sigmatr*Q0))'.*Q0;

        % Check if proposed draw satisfies sign restrictions.
        if all(S*Q0(:) >= 0)
            
            Qdraws_ar(:,:,kk) = Q0;
            flag = 1;

        end

    end

end
runTime_ar(ii,rep) = toc;

% Soft sign restrictions approach

for jj = 1:length(Deltas)

Delta = Deltas(jj);

tic
z0 = randn([4,1]);
% Run slice sampler
z = slicesampleadj(z0,draws,@(x)logtd(x,S,Delta),opt.width,opt.thin,...
    opt.burn,opt.widthprob,opt.widthinf);
z = reshape(z',[2,2,draws]);
% Use importance sampling to obtain draws from uniform distribution
Qdraws_ss = zeros([n,n,draws]);
weights = zeros(draws,1);
for kk = 1:draws
    [R,~] = chol(z(:,:,kk)'*z(:,:,kk));
    Qdraws_ss(:,:,kk) = z(:,:,kk)/R; 
    qq = Qdraws_ss(:,:,kk);
    I = S*qq(:);
    if any(I < 0)
        weights(kk) = 0;
    else
        weights(kk) = 1./exp(-sum(log_one_plus_e(-I./Delta)));
    end
end 

% Resample draws using importance sampling
inds = randsample(draws,draws,1,weights);
Qdraws_rs = Qdraws_ss(:,:,inds);

% Compute effective sample size
ess(ii,jj,rep) = 100*((sum(weights)^2)/sum(weights.^2))/draws;

runTime_ss(ii,jj,rep) = toc;

end

end

end

cd Results
save('MonteCarlo_results.mat','-mat');
cd ..

%% Construct tables for exporting results to LaTeX

cd Results

speed_tab = zeros([1+length(Deltas),9]);
for ii = 1:length(omegas)
    speed_tab(1,((ii-1)*3+1):ii*3) = ...
        [min(runTime_ar(ii,:),[],2), mean(runTime_ar(ii,:),2),max(runTime_ar(ii,:),[],2)];
    for jj = 1:length(Deltas)
    speed_tab(jj+1,((ii-1)*3+1):ii*3) = ...
        [min(runTime_ss(ii,jj,:),[],3), mean(runTime_ss(ii,jj,:),3),max(runTime_ss(ii,jj,:),[],3)];
    end
end

ess_tab = zeros([length(Deltas),9]);
for ii = 1:length(omegas)
    for jj = 1:length(Deltas)
    ess_tab(jj,((ii-1)*3+1):ii*3) = ...
        [min(ess(ii,jj,:),[],3), mean(ess(ii,jj,:),3),max(ess(ii,jj,:),[],3)];
    end
end

TT_speed = cell([1+length(Deltas),20]);
TT_speed(:,2:2:18) = {'&'};
TT_speed(:,end) = {'\\'};
TT_speed(1,1) = {'Accept-reject'};
for jj = 1:length(Deltas)
    TT_speed(jj+1,1) = {['$\Delta = ',Deltas(jj),'$']};
end
TT_speed(:,3:2:19) = num2cell(speed_tab);
writecell(TT_speed,'MonteCarlo_table.xlsx','Sheet','speed');

TT_ess = cell([length(Deltas),20]);
TT_ess(:,2:2:18) = {'&'};
TT_ess(:,end) = {'\\'};
for jj = 1:length(Deltas)
    TT_ess(jj,1) = {['$Delta = ',Deltas(jj),'$']};
end
TT_ess(:,3:2:19) = num2cell(ess_tab);
writecell(TT_ess,'MonteCarlo_table.xlsx','Sheet','ess');

% Simple version - means only
means = zeros([1+length(Deltas),6]);
means(1,:) = [mean(runTime_ar,2)', 100*ones(1,3)];
for jj = 1:length(Deltas)
means(jj+1,:) = [mean(runTime_ss(:,jj,:),3)', mean(ess(:,jj,:),3)'];
end

TT = cell([1+length(Deltas),14]);
TT(:,2:2:12) = {'&'};
TT(:,end) = {'\\'};
TT(1,1) = {'Accept-reject'};
for jj = 1:length(Deltas)
    TT(jj+1,1) = {['$\Delta = ',Deltas(jj),'$']};
end
TT(:,3:2:13) = num2cell(means);
writecell(TT,'MonteCarlo_table.xlsx','Sheet','means');

cd ..

%% Compute log target density for slice sampler
function lf = logtd(z,S,Delta)
    lf = -0.5*(z*z');
    n = sqrt(length(z));
    z = reshape(z',[n,n]);
    [R,~] = chol(z'*z);
    Q = z/R; 
    I = S*Q(:); 
    lf = lf - sum(log_one_plus_e(-I./Delta));
end