% Compare performance of accept-reject and soft sign restrictions in
% empirical setting. Model is monetary policy SVAR in Antolin-Diaz and
% Rubio-Ramirez (2018). Six-variable VAR with sign restrictions on impulse
% responses to monetary policy shock and narrative restrictions.

clear variables
close all
clc

addpath('auxFunctions');

%% Import data.
% Data are obtained from the replication files to Antolin-Diaz and
% Rubio-Ramirez (2018), available on Juan Antolin-Diaz's website:
% https://sites.google.com/site/juanantolindiaz/
% The variables are:
% 1) Real GDP 2) GDP deflator 3) Commodity price index 4) Total reserves
% 5) Nonborrowed reserves 6) Federal funds rate
% Data run from January 1965 to November 2007
load('Uhlig_Data_Updated.mat');

% Re-order variables so that the shock of interest is ordered first. 
data = 100*data(:,[end 1:end-1]); % Federal funds rate ordered first

varNames = {'Federal Funds Rate','Real GDP','GDP Deflator',...
    'Commodity Price Index','Total Reserves','Nonborrowed Reserves'};

% Declare any exogenous variables (other than the constant; leave empty 
% if none).
exog = [];

%% Options.
opt.p = 12; % Number of lags in VAR
opt.const = 1; % const = 1 if constant in VAR, = 0 otherwise
opt.cumIR = []; % Indices of variables for which cumulative impulse response should be used
opt.H = 60; % Terminal horizon for impulse responses
opt.nonEmpty = 1000; % No. of draws from posterior of phi with non-empty identified set
opt.qDraws = 1e3; % Number of draws of Q
opt.aalpha = 0.68; % Credibility level for credible intervals
opt.dispIter = 10; % Print number of reduced-form draws remaining every dispIter draws

% Set values for Delta in smoothed target density - smaller values = larger
% penalty for violating restrictions
opt.Delta = 1e-5;
opt.Delta_optim = 0.1; % Delta to use when finding initial value with numerical optimisation
opt.width = 1; % Tuning parameter for slice sampler (width of hypercube)
opt.thin = 1; % Thinning parameter - keep every thin-th draw
opt.burn = 0; % Burn-in sample size - discard first burn draws
opt.widthprob = 0.05; % Probability of drawing proposal with inflated width
opt.widthinf = 3; % Inflation factor for proposal

rng(19061987); % Set seed for random number generator

%% Specify restrictions from Antolin-Diaz and Rubio-Ramirez (2018).
% Each row of tradSign contains a vector (i,j,h,s,t) representing a
% particular sign restriction, where t is the type of restriction:
% t = 1: the horizon-h impulse response of the ith to the jth shock 
% is nonnegative (s = 1) or nonpositive (s = -1).
% t = 2: the (ij)th element of A0 is nonnegative (s = 1) or nonpositive 
% (s = -1).
% Warning: tradSign should NOT include sign normalisations on elements of 
% A0. These are imposed by default.
restr.tradSign = ...
      [1 1 0 1 1; % Response of FFR to monetary policy shock on impact is nonnegative
       1 1 1 1 1; % As above after one month
       1 1 2 1 1; % As above after two months
       1 1 3 1 1; % As above after three months
       1 1 4 1 1; % As above after four months
       1 1 5 1 1; % As above after five months
       3 1 0 -1 1; % Response of GDPDEF to monetary policy shock is nonpositive
       3 1 1 -1 1; % As above after one month, etc
       3 1 2 -1 1;
       3 1 3 -1 1;
       3 1 4 -1 1;
       3 1 5 -1 1;
       4 1 0 -1 1; % Response of COMM to monetary policy shock is nonpositive
       4 1 1 -1 1; % As above after one month, etc
       4 1 2 -1 1;
       4 1 3 -1 1;
       4 1 4 -1 1;
       4 1 5 -1 1;       
       6 1 0 -1 1; % Response of NBR to monetary policy shock is nonpositive
       6 1 1 -1 1; % As above after one month, etc
       6 1 2 -1 1;
       6 1 3 -1 1;
       6 1 4 -1 1;
       6 1 5 -1 1]; 

% Each row of elast contains a vector (i,j,k,l,s,m) representing a bound on a
% ratio of impulse responses (an 'elasticity'): s*B(i,j)/B(k,l) <= m,
% where B is the matrix of impact impulse responses, s = 1 or -1 and m is scalar.
restr.elast = []; % No elasticity restrictions

% Each row of shockSign contains a vector (i,s,t) representing the
% shock-sign restriction that the ith shock in period t is >= 0 (s = 1) or
% <= 0 (s = -1). t is equal to the index of the observation.
% restr.shockSign = []; % No shock-sign restrictions
restr.shockSign = ...
    [1, 1, find(dates == datenum(1974,4,01));
     1, 1, find(dates == datenum(1979,10,01));
     1, 1, find(dates == datenum(1988,12,01));
     1, 1, find(dates == datenum(1994,2,01));
     1, -1, find(dates == datenum(1990,12,01));
     1, -1, find(dates == datenum(1998,10,01));
     1, -1, find(dates == datenum(2001,4,01));
     1, -1, find(dates == datenum(2002,11,01))];

% Each row of hdSign contains a vector (i,j,t,h,k,s) representing a
% narrative restriction on the historical decomposition. 
% k represents the class of restriction (Type A or Type B) and s represents
% the type of restriction within the class.
% k = 1: Type A restriction
%  - s = 1:  The jth shock is the 'most important contributor' to the 
%   change in the ith variable between periods t and  t+h.
%  - s = -1: The jth shock is the 'least important contributor' to the
%   change in the ith variable between periods t and t+h.
% k = 2: Type B restriction
%  - s = 1: The jth shock is the 'overwhelming contributor' to the change
%   in the ith variable between periods t and t+h.
% - s = -1: The jth shock is a 'negligible contributor' to the change in
%   the ith variable between periods t and t+h.
% restr.hdSign = []; % No restrictions on historical decomposition
restr.hdSign = ...
    [1, 1, find(dates == datenum(1974,4,01)), 0, 1, 1;
     1, 1, find(dates == datenum(1979,10,01)), 0, 1, 1;
     1, 1, find(dates == datenum(1988,12,01)), 0, 1, 1;
     1, 1, find(dates == datenum(1994,2,01)), 0, 1, 1;
     1, 1, find(dates == datenum(1990,12,01)), 0, 1, 1;
     1, 1, find(dates == datenum(1998,10,01)), 0, 1, 1;
     1, 1, find(dates == datenum(2001,4,01)), 0, 1, 1;
     1, 1, find(dates == datenum(2002,11,01)), 0, 1, 1];
% First shock in these periods was most important contributor to change in
% first variable (federal funds rate).

% Save prelims (restrictions etc)
cd Results
save('MonetaryPolicy_setup.mat');
cd ..

%% Soft sign restrictions (implemented via slice sampling)
svar_ss; 

% Compute number of effective draws per hour
effDrawph = mean(ess/100)*opt.qDraws*opt.nonEmpty/(runTime/(60*60));
sprintf('Number of effective draws per hour: %f',effDrawph)

cd Results
save('MonetaryPolicy_ss.mat');
cd ..
