% Explore what features of data allow us to learn about the contributions
% of demand and supply shocks via simulation.

clear variables
% close all
% clc

addpath('auxFunctions');

rng(23032021);

T = 100;
K = 10000;

% Specify DGP
alpha = 20;
A0 = [1, -1; 1, 1];
A0(2,:) = alpha*A0(2,:);
B = inv(A0);
Sigma = B*B';
Sigmatr = chol(Sigma,"lower");

ut = zeros(T,2);

for tt = 1:T

    ut(tt,:) = B*normrnd(0,1,[2,1]);

end

ut = [ut; -0.5, -0.5]; % Append additional forecast error

% Compute lower and upper bound of identified set for theta
if Sigmatr(2,1) < 0
    thetalb = atan(Sigmatr(2,2)/Sigmatr(2,1));
    thetaub = 0;
else
    thetalb = -pi/2;
    thetaub = atan(-Sigmatr(2,1)/Sigmatr(2,2));
end

% Draw random values of theta from a uniform distribution over
% identified set.
thetarnd = thetalb + (thetaub-thetalb)*rand(K,1);

% Approximate identified sets for slopes of demand and supply curves.
[demandrnd,supplyrnd] = slopes(thetarnd,Sigmatr);
demandSlope = [min(demandrnd), max(demandrnd)];
supplySlope = [min(supplyrnd),max(supplyrnd)];

% cd Figures

figure;
scatter(ut(:,2),ut(:,1));
hold on;
plot(linspace(-1,1,100)',linspace(1,-1,100)','color','red','LineWidth',2);
plot(linspace(-1,1,100)',linspace(1,-1,100)'-1,'color','red','LineWidth',2,'LineStyle','--');
plot(linspace(-1,1,100)',linspace(-1,1,100)','color','green','LineWidth',2);

figure;
scatter(ut(:,2),ut(:,1));
hold on;
h1 = plot(linspace(-1,1,100)',linspace(1,-1,100)','color','red','LineWidth',2);
plot(linspace(-1,1,100)',linspace(1,-1,100)'-1,'color','red','LineWidth',2,'LineStyle','--');
h2 = plot(linspace(0,0,100)',linspace(-1.5,1,100)','color','green','LineWidth',2);
plot(linspace(-0.5,-0.5,100)',linspace(-1.5,1,100)','color','green','LineWidth',2,'LineStyle','--');
ylims = ylim;
xlims = xlim;
line(xlims,[0 0],'color','black','LineStyle',':');
line([0 0],ylims,'color','black','LineStyle',':');
ylabel('$$p_t$$','Interpreter','latex');
xlabel('$$q_t$$','Interpreter','latex');
title('Vertical Supply Curve');
legend([h1 h2],{'Demand curve','Supply curve'},'Location','NorthEast');
legend boxoff;
% print('illustration1','-dsvg');

figure;
scatter(ut(:,2),ut(:,1));
hold on;
h1 = plot(linspace(-1,1,100)',linspace(1,-1,100)','color','red','LineWidth',2);
plot(linspace(-1,1,100)',linspace(1,-1,100)'-1,'color','red','LineWidth',2,'LineStyle','--');
h2 = plot(linspace(-1,1,100)',linspace(0,0,100)','color','green','LineWidth',2);
plot(linspace(-1,1,100)',linspace(-0.5,-0.5,100)','color','green','LineWidth',2,'LineStyle','--');
ylims = ylim;
xlims = xlim;
line(xlims,[0 0],'color','black','LineStyle',':');
line([0 0],ylims,'color','black','LineStyle',':');
ylabel('$$p_t$$','Interpreter','latex');
xlabel('$$q_t$$','Interpreter','latex');
title('Horizontal Supply Curve');
legend([h1 h2],{'Demand curve','Supply curve'},'Location','NorthEast');
legend boxoff;
% print('illustration2','-dsvg');

% cd ..

function [demand,supply] = slopes(theta,Sigmatr)
% Compute slope of demand and supply curves p_t = a(theta)*q_t + eps for
% values of theta given reduced-form parameters in Sigmatr
sig11 = Sigmatr(1,1);
sig21 = Sigmatr(2,1);
sig22 = Sigmatr(2,2);

supply = -sig11*sin(theta)./(sig22*cos(theta)-sig21*sin(theta));
demand = sig11*cos(theta)./(sig21*cos(theta)+sig22*sin(theta));

end