% Using disaggregated data on US personal consumption expenditure, estimate
% sign-restricted SVARs and compute historical decompositions of price 
% changes in each expenditure category.

clear variables
% close all
% clc

addpath('auxFunctions');

%% Import data
% Data are downloaded from the Bureau of Economic Analysis website,
% Underlying Detail tables. 
cd Data
cats = readcell('SelectionDummies.xlsx','Sheet','Dummies','Range','A2:A409');
dummies = readmatrix('SelectionDummies.xlsx','Sheet','Dummies','Range','B2:B409');
QtyData = readmatrix('T_2_4_3U.xlsx','Range','C8:PO416');
PriceData = readmatrix('T_2_4_4U.xlsx','Range','C8:PO416');
WeightsData = readmatrix('T_2_4_5U.xlsx','Range','C8:PO416');
cd ..

% Keep rows corresponding to selected expenditure categories
catNames = cats(dummies==1);
qty = QtyData(dummies==1,:)';
price = PriceData(dummies==1,:)';
wts = WeightsData(dummies==1,:)';
wts = wts./sum(wts,2); % Expenditure weights - ensure sum to one

start_date = datetime(1988,01,01);
dates = dateshift(start_date,'start','month',0:size(QtyData,2)-1)';

% Transform into quarterly log growth
price = 100*(log(price(2:end,:))-log(price(1:end-1,:)));
qty = 100*(log(qty(2:end,:))-log(qty(1:end-1,:)));
dates = dates(2:end);

N = size(qty,2);

%% Options for VAR
opt.p = 12; % Lag order of VAR
opt.const = 1; % =1 to include constant, = 0 to exclude
opt.trend = 0; % =1 to include linear trend, = 0 to exclude
opt.freq = "M"; % Q = quarterly, M = monthly - used when calculating contribs to year-ended growth

if opt.freq == 'Q'
    freq_adj = 4-1;
elseif opt.freq == 'M'
    freq_adj = 12-1;
end

hor = 48; % Horizon for FEVDs

%% Estimate VAR for each expenditure category
T = size(qty,1) - opt.p; % Number of observations used in estimation
df = T-(2*opt.p+opt.const+opt.trend); % Degrees of freedom
opt.H = T-1; % Maximum horizon over which to compute impulse responses (for historical decomp)

% Storage
rho = zeros(N,1);
HD_lb = zeros(T,N);
HD_ub = zeros(T,N);
HD_ye_lb = HD_lb;
HD_ye_ub = HD_ub;
detCont = zeros(T,N);
detCont_ye = detCont;
Et_ye = detCont;
UU = zeros(T,2,N);
nonStable = zeros(N,1);

tic

for ii = 1:N % For each category

    fprintf(strcat(catNames{ii},'\n'));

    pt = price(:,ii);
    qt = qty(:,ii);

    % LHS variables
    YY = [pt(opt.p+1:end), qt(opt.p+1:end)]; 
    % Construct matrix of regressors
    XX = lags([pt, qt],1:opt.p);
    XX = XX(opt.p+1:end,:); % Drop initial missing observations
    if opt.const == 1 % Add constant
        XX = [XX, ones(size(XX,1),1)];
    end
    if opt.trend == 1 % Add linear trend
        XX = [XX, (1:size(XX,1))'];
    end    
    
    nExog = opt.const + opt.trend; % Number of exogenous variables
    
    % Estimate coefficients
    phi.B = (XX'*XX)\XX'*YY;
    % Compute VAR innovations
    U = YY - XX*phi.B;
    UU(:,:,ii) = U;
    % Estimate innovation variance-covariance matrix
    Sigma =  (1/df)*(U'*U);
    % Compute lower-triangular Cholesky factor of Sigma
    phi.Sigmatr = chol(Sigma,'lower');
    Sigmatrinv = phi.Sigmatr\eye(2);
    % Compute reduced-form impulse responses (for computing historical decomps)
    [vma,nonStable(ii)] = genVMA(phi,opt,nExog);

    % Compute conditional identified sets for historical decompositions.
    % Definition of historical decomposition here is contribution of all past
    % shocks to realisation.
    sig11 = phi.Sigmatr(1,1);
    sig21 = phi.Sigmatr(2,1);
    sig22 = phi.Sigmatr(2,2);
    rho(ii) = sig21/sqrt(sig21^2 + sig22^2);
    
    % Compute bounds of identified set for theta
    if sig21 < 0
        theta_lb = atan(sig22/sig21);
        theta_ub = 0;
    elseif sig21 >= 0
        theta_lb = -pi/2;
        theta_ub = atan(-sig21/sig22);
    end
   
    % Strategy is to evaluate HD at endpoints of identified set plus at any
    % critical values that lie within identified set, then take min and max.
    
    % Build Omega matrix appearing in definition of historical decomposition
    % (function of reduced-form parameters and forecast errors)
    Omega = zeros(2,2,T);
    U = reshape(U',[2,1,T]);
    for tt = 1:T
        ut = U(:,:,1:tt);
        ut = flip(ut,3);
        Omega(:,:,tt) = sum(pagemtimes(vma(1,:,1:tt),'transpose',pagemtimes(Sigmatrinv,ut),'transpose'),3);
    end
    
    % Evaluate first column of Q at endpoints of identified set for theta
    q1_lb = [cos(theta_lb), sin(theta_lb)]';
    q1_ub = [cos(theta_ub), sin(theta_ub)]';
    
    for tt = 1:T
    
        % Evaluate HD at endpoints of identified set
        hd_crits = [q1_lb'*Omega(:,:,tt)*q1_lb, q1_ub'*Omega(:,:,tt)*q1_ub];
        [V,~] = eig(0.5*(Omega(:,:,tt)+Omega(:,:,tt)'));
        for vv = 1:2
            q1_star = V(:,vv).*sign(V(:,vv)'*Sigmatrinv(:,1));
            hd_star = q1_star'*Omega(:,:,tt)*q1_star;
            theta_star = acos(q1_star(1))*sign(q1_star(2));
            if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
                hd_crits = [hd_crits, hd_star];
            end
        end
        HD_ub(tt,ii) = max(hd_crits);
        HD_lb(tt,ii) = min(hd_crits);
    
    end
    
    % Compute contributions to year-ended growth rates - rolling four-quarter
    % sum of contributions to quarterly growth rates.
    Omega_ye = movsum(Omega,[freq_adj,0],3); 
    
    for tt = 1:T
    
        % Evaluate HD at endpoints of identified set
        hd_crits = [q1_lb'*Omega_ye(:,:,tt)*q1_lb, q1_ub'*Omega_ye(:,:,tt)*q1_ub];
        [V,~] = eig(0.5*(Omega_ye(:,:,tt)+Omega_ye(:,:,tt)'));
        for vv = 1:2
            q1_star = V(:,vv).*sign(V(:,vv)'*Sigmatrinv(:,1));
            hd_star = q1_star'*Omega_ye(:,:,tt)*q1_star;
            theta_star = acos(q1_star(1))*sign(q1_star(2));
            if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
                hd_crits = [hd_crits, hd_star];
            end
        end
        HD_ye_ub(tt,ii) = max(hd_crits);
        HD_ye_lb(tt,ii) = min(hd_crits);
    
    end
   
    % Compute part that is explainable by shocks occurring in sample
    % (i.e. realisation minus contribution of deterministic factors).
    C = pagemtimes(vma,Sigmatrinv);
    Et = zeros(T,2);
    
    for tt = 1:T
    
        ut = U(:,:,1:tt);
        ut = flip(ut,3);
        Et(tt,:) = sum(pagemtimes(C(:,:,1:tt),ut),3);
    
    end

    Et_ye(:,ii) = movsum(Et(:,1),[freq_adj,0],1);
    
    % Compute contribution of deterministic terms (constant, trend and initial
    % conditions) to realisations.
    detCont(:,ii) = YY(:,1) - Et(:,1);
    YY_ye = movsum(YY,[freq_adj,0],1);
    detCont_ye(:,ii) = YY_ye(:,1) - Et_ye(:,ii);

end

fprintf('\n\nCategory with smallest absolute correlation is %s (rho=%f)\n',catNames{min(abs(rho))==abs(rho)},rho(min(abs(rho))==abs(rho)));
fprintf('Category with largest absolute correlation is %s (rho=%f)\n',catNames{max(abs(rho))==abs(rho)},rho(max(abs(rho))==abs(rho)));

% Number of categories with non-stable VAR
fprintf('VAR nonstable in %d categories\n',sum(nonStable));
fprintf('Categories with non-stable VAR account for %f per cent of expenditure on average\n',100*mean(sum(wts(:,nonStable==1),2)));

% Re-normalise weights, removing categories with non-stable VAR
wts = wts(:,nonStable==0)./sum(wts(:,nonStable==0),2);

% Aggregate lower and upper bounds of historical decompositions.
HD_ye_sum_lb = sum(wts(opt.p+2:end,:).*HD_ye_lb(:,nonStable==0),2);
HD_ye_sum_ub = sum(wts(opt.p+2:end,:).*HD_ye_ub(:,nonStable==0),2);
Et_ye_sum = sum(wts(opt.p+2:end,:).*Et_ye(:,nonStable==0),2);
detCont_ye_sum = sum(wts(opt.p+2:end,:).*detCont_ye(:,nonStable==0),2);

%% Repeat for aggregate data

fprintf('Aggregate PCE \n');
% Construct quarterly growth in aggregate of expenditures and prices for
% categories with stable VAR in bottom-up exercise.
qty_agg = sum(qty(:,nonStable==0).*wts(2:end,:),2);
price_agg = sum(price(:,nonStable==0).*wts(2:end,:),2);

pt = price_agg;
qt = qty_agg;

% LHS variables
YY = [pt(opt.p+1:end), qt(opt.p+1:end)]; 
% Construct matrix of regressors
XX = lags([pt, qt],1:opt.p);
XX = XX(opt.p+1:end,:); % Drop initial missing observations
if opt.const == 1 % Add constant
    XX = [XX, ones(size(XX,1),1)];
end
if opt.trend == 1 % Add linear trend
    XX = [XX, (1:size(XX,1))'];
end    

nExog = opt.const + opt.trend; % Number of exogenous variables

% Estimate coefficients
phi.B = (XX'*XX)\XX'*YY;
% Compute VAR innovations
U = YY - XX*phi.B;
% Estimate innovation variance-covariance matrix
Sigma =  (1/df)*(U'*U);
% Compute lower-triangular Cholesky factor of Sigma
phi.Sigmatr = chol(Sigma,'lower');
Sigmatrinv = phi.Sigmatr\eye(2);
% Compute reduced-form impulse responses (for computing historical decomps)
[vma,nonStable_agg] = genVMA(phi,opt,nExog);

% Compute sets of historical decompositions.
% Definition of historical decomposition here is contribution of all past
% shocks to realisation.
sig11 = phi.Sigmatr(1,1);
sig21 = phi.Sigmatr(2,1);
sig22 = phi.Sigmatr(2,2);
rho_agg = sig21/sqrt(sig21^2 + sig22^2);

% Compute bounds of identified set for theta
if sig21 < 0
    theta_lb = atan(sig22/sig21);
    theta_ub = 0;
elseif sig21 >= 0
    theta_lb = -pi/2;
    theta_ub = atan(-sig21/sig22);
end

% Strategy is to evaluate HD at endpoints of identified set plus at any
% critical values that lie within identified set, then take min and max.
    
% Build Omega matrix appearing in definition of historical decomposition
% (function of reduced-form parameters and forecast errors)
Omega = zeros(2,2,T);
U = reshape(U',[2,1,T]);
for tt = 1:T
    ut = U(:,:,1:tt);
    ut = flip(ut,3);
    Omega(:,:,tt) = sum(pagemtimes(vma(1,:,1:tt),'transpose',pagemtimes(Sigmatrinv,ut),'transpose'),3);
end

% Evaluate first column of Q at endpoints of identified set for theta
q1_lb = [cos(theta_lb), sin(theta_lb)]';
q1_ub = [cos(theta_ub), sin(theta_ub)]';

HD_ub_agg = zeros(T,1);
HD_lb_agg = zeros(T,1);

for tt = 1:T

    % Evaluate HD at endpoints of identified set
    hd_crits = [q1_lb'*Omega(:,:,tt)*q1_lb, q1_ub'*Omega(:,:,tt)*q1_ub];
    [V,~] = eig(0.5*(Omega(:,:,tt)+Omega(:,:,tt)'));
    for vv = 1:2
        q1_star = V(:,vv).*sign(V(:,vv)'*Sigmatrinv(:,1));
        hd_star = q1_star'*Omega(:,:,tt)*q1_star;
        theta_star = acos(q1_star(1))*sign(q1_star(2));
        if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
            hd_crits = [hd_crits, hd_star];
        end
    end
    HD_ub_agg(tt) = max(hd_crits);
    HD_lb_agg(tt) = min(hd_crits);

end

% Compute contributions to year-ended growth rates - rolling four-quarter
% sum of contributions to quarterly growth rates.
Omega_ye = movsum(Omega,[freq_adj,0],3); 

HD_ye_ub_agg = zeros(T,1);
HD_ye_lb_agg = zeros(T,1);

for tt = 1:T

    % Evaluate HD at endpoints of identified set
    hd_crits = [q1_lb'*Omega_ye(:,:,tt)*q1_lb, q1_ub'*Omega_ye(:,:,tt)*q1_ub];
    [V,~] = eig(0.5*(Omega_ye(:,:,tt)+Omega_ye(:,:,tt)'));
    for vv = 1:2
        q1_star = V(:,vv).*sign(V(:,vv)'*Sigmatrinv(:,1));
        hd_star = q1_star'*Omega_ye(:,:,tt)*q1_star;
        theta_star = acos(q1_star(1))*sign(q1_star(2));
        if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
            hd_crits = [hd_crits, hd_star];
        end
    end
    HD_ye_ub_agg(tt) = max(hd_crits);
    HD_ye_lb_agg(tt) = min(hd_crits);

end

% Compute part of y_t that is explainable by shocks occurring in sample
% (i.e. y_t minus contribution of deterministic factors).
C = pagemtimes(vma,phi.Sigmatr\eye(2));

Et = zeros(T,2);

for tt = 1:T

    ut = U(:,:,1:tt);
    ut = flip(ut,3);
    Et(tt,:) = sum(pagemtimes(C(:,:,1:tt),ut),3);

end

Et_ye_agg = movsum(Et,[freq_adj,0],1);

% Compute contribution of deterministic terms (constant, trend and initial
% conditions) to realisations.
detCont_agg = YY(:,1) - Et(:,1);
YY_ye_agg = movsum(YY,[freq_adj,0],1);
detCont_ye_agg = YY_ye_agg(:,1) - Et_ye_agg(:,1);

runTime = toc;

cd Results
save('disaggregated_results.mat');
cd ..

