% Use sign-restricted SVAR to estimate forecast error variance
% decompositions and historical decompositions of US inflation (based on 
% the GDP deflator).

clear variables
% close all
% clc

addpath('auxFunctions');

%% Options for VAR
opt.p = 8; % Lag order of VAR
opt.const = 1; % =1 to include constant, = 0 to exclude
opt.trend = 0; % =1 to include linear trend, = 0 to exclude
opt.freq = "Q"; % Q = quarterly, M = monthly - used when calculating contribs to year-ended growth

%% Import data (downloaded from FRED)
cd Data
PiData = readtable('GDPDEF.xls','Range','A12:B317'); % GDP deflator
GDPData = readtable('GDPC1.xls','Range','A12:B317'); % Real GDP
cd ..

% Select dates
% Starting one year before estimation start date to account for losing 
% observations due to differencing
date_start = datetime(1988,10,01);
date_end = datetime(2023,04,01);
date_ind = (date_start <= PiData{:,1}) & (PiData{:,1} <= date_end);
dates = PiData{date_ind,1};

Pi = PiData{date_ind,2};
GDP = GDPData{date_ind,2};

% Transform into quarterly log growth
Pi = 100*(log(Pi(2:end)) - log(Pi(1:end-1)));
GDP =  100*(log(GDP(2:end)) - log(GDP(1:end-1)));
dates = dates(2:end);

%% Estimate reduced-form VAR
% LHS variables
YY = [Pi(opt.p+1:end), GDP(opt.p+1:end)]; 
% Construct matrix of regressors
XX = lags([Pi, GDP],1:opt.p);
XX = XX(opt.p+1:end,:); % Drop initial missing observations
if opt.const == 1 % Add constant
    XX = [XX, ones(size(XX,1),1)];
end
if opt.trend == 1 % Add linear trend
    XX = [XX, (1:size(XX,1))'];
end    

nExog = opt.const + opt.trend;

T = size(YY,1); % Number of observations
df = T-(2*opt.p+opt.const+opt.trend); % Degrees of freedom
opt.H = T-1; % Maximum horizon over which to compute impulse responses

% Estimate coefficients
phi.B = (XX'*XX)\XX'*YY;
% Compute VAR innovations
U = YY - XX*phi.B;
% Estimate variance-covariance matrix
Sigma =  (1/df)*(U'*U);
% Compute lower-triangular Cholesky factor of Sigma
phi.Sigmatr = chol(Sigma,'lower');
Sigmatrinv = phi.Sigmatr\eye(2);
% Compute reduced-form impulse responses (for computing historical decomps)
[vma,nonStable] = genVMA(phi,opt,nExog);

sig11 = phi.Sigmatr(1,1);
sig21 = phi.Sigmatr(2,1);
sig22 = phi.Sigmatr(2,2);
rho = sig21/sqrt(sig21^2 + sig22^2);
fprintf('Correlation in one-step-ahead forecast errors is %f \n',rho);

% Compute bounds of identified set for theta
if sig21 < 0
    theta_lb = atan(sig22/sig21);
    theta_ub = 0;
elseif sig21 >= 0
    theta_lb = -pi/2;
    theta_ub = atan(-sig21/sig22);
end

%% Compute conditional identified sets for historical decompositions
% Definition of historical decomposition here is contribution of all past
% shocks to realisation
% Strategy is to evaluate HD at endpoints of identified set plus at any
% critical values that lie within identified set, then take min and max.

% Build Omega matrix appearing in definition of historical decomposition
% (function of reduced-form parameters and forecast errors)
Omega = zeros(2,2,T);
U = reshape(U',[2,1,T]);
for tt = 1:T
    ut = U(:,:,1:tt);
    ut = flip(ut,3);
    Omega(:,:,tt) = sum(pagemtimes(vma(1,:,1:tt),'transpose',pagemtimes(Sigmatrinv,ut),'transpose'),3);
end

% Evaluate first column of Q at endpoints of identified set for theta
q1_lb = [cos(theta_lb), sin(theta_lb)]';
q1_ub = [cos(theta_ub), sin(theta_ub)]';

HD_lb = zeros(T,1);
HD_ub = zeros(T,1);

for tt = 1:T

    % Evaluate HD at endpoints of identified set
    hd_crits = [q1_lb'*Omega(:,:,tt)*q1_lb, q1_ub'*Omega(:,:,tt)*q1_ub];
    [V,~] = eig(0.5*(Omega(:,:,tt)+Omega(:,:,tt)'));
    for vv = 1:2
        q1_star = V(:,vv).*sign(V(:,vv)'*Sigmatrinv(:,1));
        hd_star = q1_star'*Omega(:,:,tt)*q1_star;
        theta_star = acos(q1_star(1))*sign(q1_star(2));
        if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
            hd_crits = [hd_crits, hd_star];
        end
    end
    HD_ub(tt) = max(hd_crits);
    HD_lb(tt) = min(hd_crits);

end

% Compute contributions to year-ended growth rates - rolling four-quarter
% sum of contributions to quarterly growth rates.
if opt.freq == 'Q'
    freq_adj = 4-1;
elseif opt.freq == 'M'
    freq_adj = 12-1;
end
Omega_ye = movsum(Omega,[freq_adj,0],3); 

HD_ye_lb = zeros(T,1);
HD_ye_ub = zeros(T,1);

for tt = 1:T

    % Evaluate HD at endpoints of identified set
    hd_crits = [q1_lb'*Omega_ye(:,:,tt)*q1_lb, q1_ub'*Omega_ye(:,:,tt)*q1_ub];
    [V,~] = eig(0.5*(Omega_ye(:,:,tt)+Omega_ye(:,:,tt)'));
    for vv = 1:2
        q1_star = V(:,vv).*sign(V(:,vv)'*Sigmatrinv(:,1));
        hd_star = q1_star'*Omega_ye(:,:,tt)*q1_star;
        theta_star = acos(q1_star(1))*sign(q1_star(2));
        if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
            hd_crits = [hd_crits, hd_star];
        end
    end
    HD_ye_ub(tt) = max(hd_crits);
    HD_ye_lb(tt) = min(hd_crits);

end

% Compute part of y_t that is explainable by shocks occurring in sample
% (i.e. y_t minus contribution of deterministic factors).
C = pagemtimes(vma,Sigmatrinv);

Et = zeros(T,2);

for tt = 1:T

    ut = U(:,:,1:tt);
    ut = flip(ut,3);
    Et(tt,:) = sum(pagemtimes(C(:,:,1:tt),ut),3);

end

Et_ye = movsum(Et,[freq_adj,0],1);

% Compute contribution of deterministic terms (constant, trend and initial
% conditions) to realisations.
detCont = YY - Et;
detCont_avg = mean(detCont);
fprintf('Average contribution of deterministic terms: %f \n',detCont_avg(1));

% Compute contribution of deterministic terms to year-ended growth
YY_ye = movsum(YY,[freq_adj,0],1);
detCont_ye = YY_ye - Et_ye;
detCont_ye_avg = mean(detCont_ye(freq_adj+1:end));
fprintf('Average contribution of deterministic terms: %f \n',detCont_ye_avg(1));

%% Plot contributions of supply shock - quarterly
cd Figures
plotDates = dates(opt.p+1:end);
figure; % Inflation
h1 = plot(plotDates,Et(:,1),'color','black','LineWidth',2);
hold on;
h2 = plot(plotDates,HD_lb(:,1),'color','blue','LineStyle','--');
plot(plotDates,HD_ub(:,1),'color','blue','LineStyle','--');
patch([plotDates; flip(plotDates)],[HD_lb(:,1); flip(HD_ub(:,1))], ...
    'blue','FaceAlpha','0.1','EdgeColor','none');
xlims = xlim;
line(xlims,[0, 0],'color','black','LineStyle',':');
title('Contributions to quarterly inflation');
ylabel('ppt');
legend([h1 h2],{'$$\pi_t$$','Set of historical decompositions'},...
    'Location','NorthWest','Interpreter','LaTeX')
legend boxoff;
ax = gca;
ax.FontSize = 14;
print('GDPDEF','-depsc');

% Year-ended
figure; % Inflation
h1 = plot(plotDates,Et_ye(:,1),'color','black','LineWidth',2);
hold on;
h2 = plot(plotDates,HD_ye_lb(:,1),'color','blue','LineStyle','--');
plot(plotDates,HD_ye_ub(:,1),'color','blue','LineStyle','--');
patch([plotDates; flip(plotDates)],[HD_ye_lb(:,1); flip(HD_ye_ub(:,1))], ...
    'blue','FaceAlpha','0.1','EdgeColor','none');
xlims = xlim;
line(xlims,[0, 0],'color','black','LineStyle',':');
title('Contributions to year-ended inflation');
ylabel('ppt');
% legend([h1 h2],{'$$\pi_t$$','Set of historical decompositions'},...
%     'Location','NorthWest','Interpreter','LaTeX')
% legend boxoff;
ax = gca;
ax.FontSize = 14;
print('GDPDEF_ye','-depsc');

cd ..

%% Compute identified sets for forecast error variance decompositions (FEVDs)
hor = 20; % Terminal horizon
Upsilon = zeros(2,2,hor+1);

for hh = 1:hor+1

    c = vma(1,:,1:hh);
    Upsilon(:,:,hh) = sum(pagemtimes(c,'transpose',c,'none'),3)./sum(pagemtimes(c,'none',c,'transpose'),3);

end

q1_lb = [cos(theta_lb), sin(theta_lb)]';
q1_ub = [cos(theta_ub), sin(theta_ub)]';
FEVD_lb = zeros(hor+1,1);
FEVD_ub = zeros(hor+1,1);

for hh = 1:hor+1

    % Evaluate FEVD at endpoints of identified set
    fevd_crits = [q1_lb'*Upsilon(:,:,hh)*q1_lb, q1_ub'*Upsilon(:,:,hh)*q1_ub];
    [V,~] = eig(Upsilon(:,:,hh));
    for vv = 1:2
        q1_star = V(:,vv).*sign(V(:,vv)'*Sigmatrinv(:,1)); % Sign normalisation
        fevd_star = q1_star'*Upsilon(:,:,hh)*q1_star;
        theta_star = acos(q1_star(1))*sign(q1_star(2)); % Corresponding theta 
        if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
            fevd_crits = [fevd_crits, fevd_star];
        end
    end
    FEVD_ub(hh) = max(fevd_crits);
    FEVD_lb(hh) = min(fevd_crits);

end

%% Plot contributions of supply shock to FEV
cd Figures
figure;
plot(0:hor,100*FEVD_lb,'color','red','LineStyle','--');
hold on;
plot(0:hor,100*FEVD_ub,'color','red','LineStyle','--');
patch([(0:hor)'; flip((0:hor)')],100*[FEVD_lb; flip(FEVD_ub)], ...
    'red','FaceAlpha','0.1','EdgeColor','none');
ylabel('%');
xlabel('Horizon (quarters)');
ax = gca;
ax.FontSize = 14;
print('GDPDEF_FEVD','-depsc');
cd ..

cd 'Results'
save('aggregate_results.mat');
cd ..

