% Conduct robust Bayesian inference about historical decompositions and
% FEVD in aggregate model.

clear variables
% close all
% clc

addpath('auxFunctions');

cd Results
load('aggregate_results.mat');
cd ..

opt.draws = 1000; % Number of posterior draws
opt.alpha = 0.68; % Credibility level for robust credible intervals

%% Posterior sampler, assuming Jeffreys prior (truncated to stable region)

% Posterior is normal-inverse-Wishart.
phiHat.B = (XX'*XX)\XX'*YY; % MLE of slope coefficients
phiHat.S = (YY - XX*phiHat.B)'*(YY-XX*phiHat.B);
phiHat.Sigma = (1/T)*phiHat.S; % MLE of VAR innovation covariance matrix
m = size(XX,2);
n = size(YY,2);
phiHat.P = (XX'*XX)\eye(m);
phiHat.cholP = chol(phiHat.P,'lower');

HD_ye_lb_draws = zeros(T,opt.draws);
HD_ye_ub_draws = HD_ye_lb_draws;
FEVD_lb_draws = zeros(hor+1,opt.draws);
FEVD_ub_draws = FEVD_lb_draws;
rho_draws = zeros(opt.draws,1);
Et_ye_draws = zeros(T,opt.draws);

nonStableCount = 0;
phiDraw = 0;
kk = 0;

tic

while kk < opt.draws
    
    % Sample from normal-inverse-Wishart posterior.
    phiDraw = phiDraw + 1;
    phi.Sigma = iwishrnd(phiHat.S,T-m);
    phi.Sigmatr = chol(phi.Sigma,'lower');
    phi.Sigmatrinv = phi.Sigmatr\eye(n);
    phi.B = phiHat.B(:) + kron(phi.Sigmatr,phiHat.cholP)*randn(m*n,1);
    phi.B = reshape(phi.B,size(phi.B,1)/n,n); % Reshape coefficients into matrix

    % Compute reduced-form impulse responses (for computing historical decomps)
    [vma,nonStable] = genVMA(phi,opt,nExog);

    if nonStable == 1
        nonStableCount = nonStableCount + 1;
        continue  % Return to beginning of while loop
    end

    kk = kk + 1

    sig11 = phi.Sigmatr(1,1);
    sig21 = phi.Sigmatr(2,1);
    sig22 = phi.Sigmatr(2,2);
    rho_draws(kk) = sig21/sqrt(sig21^2 + sig22^2);
    
    % Compute bounds of identified set for theta
    if sig21 < 0
        theta_lb = atan(sig22/sig21);
        theta_ub = 0;
    elseif sig21 >= 0
        theta_lb = -pi/2;
        theta_ub = atan(-sig21/sig22);
    end

    U = YY - XX*phi.B;
    
    % Build Omega matrix appearing in definition of historical decomposition
    % (function of reduced-form parameters and forecast errors)
    Omega = zeros(2,2,T);
    U = reshape(U',[2,1,T]);
    for tt = 1:T
        ut = U(:,:,1:tt);
        ut = flip(ut,3);
        Omega(:,:,tt) = sum(pagemtimes(vma(1,:,1:tt),'transpose',pagemtimes(Sigmatrinv,ut),'transpose'),3);
    end

    % Compute contributions to year-ended growth rates - rolling four-quarter
    % sum of contributions to quarterly growth rates.
    if opt.freq == 'Q'
        freq_adj = 4-1;
    elseif opt.freq == 'M'
        freq_adj = 12-1;
    end
    Omega_ye = movsum(Omega,[freq_adj,0],3); 
    
    % Evaluate first column of Q at endpoints of identified set for theta
    q1_lb = [cos(theta_lb), sin(theta_lb)]';
    q1_ub = [cos(theta_ub), sin(theta_ub)]';
    
    parfor tt = 1:T
    
        % Evaluate HD at endpoints of identified set
        hd_crits = [q1_lb'*Omega_ye(:,:,tt)*q1_lb, q1_ub'*Omega_ye(:,:,tt)*q1_ub];
        [V,~] = eig(0.5*(Omega_ye(:,:,tt)+Omega_ye(:,:,tt)'));
        for vv = 1:2
            q1_star = V(:,vv).*sign(V(:,vv)'*phi.Sigmatrinv(:,1));
            hd_star = q1_star'*Omega_ye(:,:,tt)*q1_star;
            theta_star = acos(q1_star(1))*sign(q1_star(2));
            if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
                hd_crits = [hd_crits, hd_star];
            end
        end
        HD_ye_ub_draws(tt,kk) = max(hd_crits);
        HD_ye_lb_draws(tt,kk) = min(hd_crits);
    
    end

    % Compute part of y_t that is explainable by shocks occurring in sample
    % (i.e. y_t minus contribution of deterministic factors).
    C = pagemtimes(vma,phi.Sigmatrinv);
    
    Et = zeros(T,2);
    
    for tt = 1:T
    
        ut = U(:,:,1:tt);
        ut = flip(ut,3);
        Et(tt,:) = sum(pagemtimes(C(:,:,1:tt),ut),3);
    
    end
    
    Et_ye_draws(:,kk) = movsum(Et(:,1),[freq_adj,0],1);

    % FEVD
    Upsilon = zeros(2,2,hor+1);
    for hh = 1:hor+1
    
        c = vma(1,:,1:hh);
        Upsilon(:,:,hh) = sum(pagemtimes(c,'transpose',c,'none'),3)./sum(pagemtimes(c,'none',c,'transpose'),3);
    
    end

    % Evaluate FEVD at endpoints of identified set

    parfor hh = 1:hor+1
        fevd_crits = [q1_lb'*Upsilon(:,:,hh)*q1_lb, q1_ub'*Upsilon(:,:,hh)*q1_ub];
        [V,~] = eig(Upsilon(:,:,hh));
        for vv = 1:2
            q1_star = V(:,vv).*sign(V(:,vv)'*phi.Sigmatrinv(:,1)); % Sign normalisation
            fevd_star = q1_star'*Upsilon(:,:,hh)*q1_star;
            theta_star = acos(q1_star(1))*sign(q1_star(2)); % Corresponding theta 
            if theta_star >= theta_lb && theta_star <= theta_ub % If inside identified set
                fevd_crits = [fevd_crits, fevd_star];
            end
        end
        FEVD_ub_draws(hh,kk) = max(fevd_crits);
        FEVD_lb_draws(hh,kk) = min(fevd_crits);
    end

end

runTime = toc;

fprintf('VAR unstable in %f per cent of draws\n',100*nonStableCount/phiDraw);

% Compute set of posterior means and robust credible intervals
HD_ye_mean = [mean(HD_ye_lb_draws,2), mean(HD_ye_ub_draws,2)];
HD_ye_cred = [quantile(HD_ye_lb_draws,opt.alpha/2,2), ...
                quantile(HD_ye_ub_draws,(1+opt.alpha)/2,2)];

FEVD_mean = [mean(FEVD_lb_draws,2), mean(FEVD_ub_draws,2)];
FEVD_cred = [quantile(FEVD_lb_draws,opt.alpha/2,2), ...
                quantile(FEVD_ub_draws,(1+opt.alpha)/2,2)];

Et_ye_mean = mean(Et_ye_draws,2);
Et_ye_cred = quantile(Et_ye_draws,[opt.alpha/2, (1+opt.alpha)/2],2);

% Compute posterior lower and upper probabilities that contribution of
% supply shocks to inflation is positive.
postProbs = [mean(HD_ye_lb_draws > 0, 2), mean(HD_ye_ub_draws > 0,2)];

cd Results
save('aggregate_inference.mat');
cd ..

%% Figures
cd Figures;
plotDates = dates(opt.p+1:end);
figure;
plot(plotDates,Et_ye_mean,'color','black','LineWidth',2);
hold on;
h2 = plot(plotDates,HD_ye_mean(:,1),'color','blue','LineStyle','--');
plot(plotDates,HD_ye_mean(:,2),'color','blue','LineStyle','--');
patch([plotDates; flip(plotDates)],[HD_ye_mean(:,1); flip(HD_ye_mean(:,2))], ...
    'blue','FaceAlpha','0.1','EdgeColor','none');
h3 = plot(plotDates,HD_ye_cred(:,1),'color','blue','LineStyle','-');
plot(plotDates,HD_ye_cred(:,2),'color','blue','LineStyle','-');
xlims = xlim;
line(xlims,[0, 0],'color','black','LineStyle',':');
title('Contributions to year-ended inflation');
ylabel('ppt');
% legend([h1 h2 h3],{'$$\pi_t$$','Set of historical decompositions','Robust credible interval'},...
%     'Location','NorthWest','Interpreter','LaTeX')
% legend boxoff;
ax = gca;
ax.FontSize = 14;
print('GDPDEF_inference','-depsc');

figure;
plot(0:hor,100*FEVD_mean(:,1),'color','red','LineStyle','--');
hold on;
plot(0:hor,100*FEVD_mean(:,2),'color','red','LineStyle','--');
patch([(0:hor)'; flip((0:hor)')],100*[FEVD_mean(:,1); flip(FEVD_mean(:,2))], ...
    'red','FaceAlpha','0.1','EdgeColor','none');
plot(0:hor,100*FEVD_cred(:,1),'color','red','LineStyle','-');
plot(0:hor,100*FEVD_cred(:,2),'color','red','LineStyle','-');
ylabel('%');
xlabel('Horizon (quarters)');
ax = gca;
ax.FontSize = 14;
% print('GDPDEF_FEVD_inference','-depsc');
cd ..
