#--------------------------------------------------------------------------------------------------
# RDP : Demand in the Repo Market: Indirect Perspectives from Open Market Operations from 2006 to 2020
#
# Chris Becker, Anny Francis, Calebe de Roure and Brendan Wilson
#
# 24 November 2020
#--------------------------------------------------------------------------------------------------
# This code creates data used for the graphs. Please refer to Stata file for the empirical estimation.
# Packages
# -------------------------------------------------------------------------------------------------
#for dates after 2011, use data. Prior to 2011 use data2




# Install required packages
list.of.packages <- c("RODBC", "arphit", "rbastattable", "stats", "tidyverse", "readrba", "imputeTS")
new.packages <- list.of.packages[!(list.of.packages %in% installed.packages()[,"Package"])]
if(length(new.packages)) install.packages(new.packages)

library(dplyr)

# Load required packages
lapply(list.of.packages, require, character.only = T)

rm(list=ls())
# Data
#-----------------------------------------------------------------

# OIS Data 

f1 <- read_rba("f1") %>% # download f1 table from RBA webpage and use OIS data
  
  filter(series %in% c("Interbank Overnight Cash Rate", "1-month OIS" , "3-month OIS", "6-month OIS"),
         date < as.Date("2020-12-01")) %>% 
  
  mutate(value = 100*value) %>% # convert from percentage points to basis points
  
  select(date,
         value,
         series) %>% 
  
  pivot_wider(names_from = series) %>%
  
  rename("cash_rate" = "Interbank Overnight Cash Rate",
         "ois_1m" = "1-month OIS",
         "ois_3m" = "3-month OIS",
         "ois_6m" = "6-month OIS")

# OMO data

omo <- odbcConnect("[SQLServer]") %>%
  sqlQuery( source("[SQLQuery]")) 
  filter(date < as.Date("2020-12-01"))%>%
  rename(chrdate=date)%>%
  mutate(date = as.Date(chrdate, "%Y-%m-%d"),
         cash_dealt = coalesce(cash_dealt,0))
  

# Data with linearly interpolated OIS
  
data <- omo %>% 
  
  left_join(f1, by = NULL) %>% 
  
  mutate(proxy_spread = case_when(term <= 30  ~ yield - ((term - 1) * ois_1m + (30 - term) * cash_rate)/(30 - 1),
                                  term <= 91  ~ yield - ((term - 30) * ois_3m + (91 - term) * ois_1m)/(91 - 30),
                                  term <= 182 ~ yield - ((term - 91) * ois_6m + (182 - term) * ois_3m)/(182 - 91),
                                  term > 182  ~ yield - ois_6m))%>% 
    
  arrange(date,
          term, 
          desc(yield)) %>%                                 
  
  group_by(date) %>% 
  
  mutate(cash_supplied = sum(cash_dealt, na.rm = T)/1000) %>% # calculate cash dealt for each day
  
  group_by(date, 
           cash_supplied, 
           yield,
           proxy_spread) %>% 
  
  summarise(approach_at_yield = sum(cash_approach, na.rm = T)/1000, 
            accepted_at_yield = sum(cash_dealt, na.rm = T)/1000) %>% # calculate approach amounts and dealt amounts for each date and yield
  
  arrange(date,
          desc(proxy_spread)) %>% 
  
  group_by(date) %>% 
  
  mutate(q_demanded     = cumsum(approach_at_yield),  
         q_RBA_demanded = cumsum(accepted_at_yield)) %>% 
  
  ungroup

# Data with Findur spread to OIS

data2 <- omo %>% 
  
  arrange(date,
          term, 
          desc(spread_to_ois)) %>%                                 
  
  group_by(date) %>% 
  
  mutate(cash_supplied = sum(cash_dealt, na.rm = T)/1000) %>% # calculate cash dealt for each day
  
  group_by(date, 
           cash_supplied, 
           spread_to_ois) %>% 
  
  summarise(approach_at_yield = sum(cash_approach, na.rm = T)/1000, 
            accepted_at_yield = sum(cash_dealt, na.rm = T)/1000) %>% # calculate approach amounts and dealt amounts for each date and yield
  
  arrange(date,
          desc(spread_to_ois)) %>% 
  
  group_by(date) %>% 
  
  mutate(q_demanded     = cumsum(approach_at_yield),  
         q_RBA_demanded = cumsum(accepted_at_yield)) %>% 
  
  ungroup %>%
  rename(proxy_spread = spread_to_ois )


# Graphs
#-----------------------------------------------------------------

# Graph 3 - 8 CBF
# G3: 2006-2007 
# G4: 2008-2009 
# G5: 2018-2019 
# G6: February 2020
# G7: 12-24 March 2020
# G8: Jul-Nov 2020

# Data for CBF Graph (adjust dates for desired time period)

graph_data <- data %>%
  
  filter(date >= as.Date("2020-03-12"),
         date <= as.Date("2020-03-24")) %>% 
  
  arrange(date, 
          desc(proxy_spread))

# Calculating confidence interval for estimated demand curve
pred <- predict(loess(proxy_spread ~ q_demanded, data = graph_data, span=0.8),  se =T)                                # fits CBF curve

pred$ci_dn <- pred$fit-1.96*pred$se.fit              # CBF 95% confidence interval lower bound
pred$ci_up <- pred$fit+1.96*pred$se.fit              # CBF 95% confidence interval upper bound




# Graph of estimated demand and supply
cbf <- arphitgg(graph_data) +
  agg_point(agg_aes(q_demanded, proxy_spread),
            colour = "lightskyblue1") +
  agg_point(agg_aes(q_RBA_demanded, proxy_spread), 
            colour ="thistle2",
            data = graph_data %>% filter(accepted_at_yield > 0)) +
  agg_line(agg_aes(q_demanded, pred$fit),
          colour = "turquoise4") +
  agg_line(agg_aes(q_demanded, pred$ci_dn),
           colour = "grey70") +
  agg_line(agg_aes(q_demanded, pred$ci_up),
           colour = "grey70") +
  agg_shading(from = pred$ci_dn,
              to   = pred$ci_up,
              panel = "1",
              colour = "grey70") +
  agg_bgshading(x1 = min(graph_data$cash_supplied), y1 = NA,
                x2 = max(graph_data$cash_supplied), y2 = NA,
                panel = "1",
                colour = "grey85") +
  agg_vline(x = mean(graph_data$cash_supplied),
            panel = "1") +
  agg_xunits("$b") +
  agg_units("bps") +
  agg_xlim(0, 25) +
  agg_ylim(-20, 100, 7) +
  agg_xaxislabel("Quantity") +
  agg_yaxislabel("Repo rate (spread to OIS)") +
  agg_label(expression(paste("Suppl",y^{"(b)"})), 
            x = 5, y = 5,
            panel = "1") +
  agg_label("Demand",
            x = 22, y = -15,
            panel = "1",
            colour = "turquoise4") +
  agg_title("RBA Open Market Operations") +
  agg_subtitle("2006 to 2007") +
  agg_footnote("Pink dots represent filled bids and blue dots represent unfilled bids") +
  agg_footnote("Mean and range") +
  agg_source("RBA")

# Save graph to EMF
agg_draw(cbf, "cbf.emf")




# Save data for Graphit graph

as_frame<-Map(as.data.frame, pred)
writexl::write_xlsx(as_frame, "Graph_.xlsx")
writexl::write_xlsx(graph_data, "Graph_.xlsx")





