# Load and clean fixed-term lending data ####

## Load raw data - using 742 ####
arf742_raw <- read_excel(paste0(input_path, "arf_742_by_institution.xlsx"))

# Add breaks

arf742_raw <- arf742_raw %>% 
  select(reporting_period, institution_code, name, reporter_type,
         BSAO28606, BSAO28627, BSAO28643, 
         BSAO29514_Large_FixedTermLoans, BSAO29514_Medium_FixedTermLoans, BSAO29514_Small_FixedTermLoans) %>% 
  pivot_longer(cols = -c("reporting_period", "institution_code", "name", "reporter_type"),
               names_to = "form_item",
               values_to = "value")


breaks <- read_excel(paste0(input_path, "Break Adjustments.xlsx"), sheet = "Breaks") %>%
  select(-institution_name, -rationale, -query_ids)

breaks <- breaks %>% 
  filter(reporting_frequency == "Monthly") %>% 
  mutate(form_item = gsub("ARF_742_0B_", "", form_item),
         form_item = gsub("ARF_742_0A_", "", form_item)) %>% 
  select(institution_code, reporting_period = period, form_item, adjustment_amount)

arf742_raw_adjusted <- arf742_raw %>% 
  left_join(breaks) %>% 
  mutate(adjustment_amount = if_else(is.na(adjustment_amount), 0, adjustment_amount)) %>% 
  mutate(value = value + (adjustment_amount / 10^9)) %>% 
  select(-adjustment_amount) %>% 
  pivot_wider(names_from = form_item, values_from = value)

## Clean 742 data ####

fixed_credit <- arf742_raw_adjusted %>% 
  rename(smallbus_credit = BSAO28606, 
         mediumbus_credit = BSAO28627, 
         largebus_credit = BSAO28643, 
         large_fixed = BSAO29514_Large_FixedTermLoans, 
         med_fixed = BSAO29514_Medium_FixedTermLoans, 
         small_fixed = BSAO29514_Small_FixedTermLoans) %>% 
  mutate(smallbus_credit = smallbus_credit * 1e9, 
         mediumbus_credit = mediumbus_credit * 1e9, 
         largebus_credit = largebus_credit * 1e9, 
         large_fixed = large_fixed * 1e9, 
         med_fixed = med_fixed * 1e9, 
         small_fixed = small_fixed * 1e9)

fixed_credit[is.na(fixed_credit)] <- 0

fixed_credit <- fixed_credit %>% 
  mutate(SME_credit = smallbus_credit + mediumbus_credit,
         SME_fixed = small_fixed + med_fixed,
         totalbus_credit = SME_credit + largebus_credit,
         total_fixed_credit = SME_fixed + large_fixed) %>% 
  arrange(reporting_period) %>% 
  arrange(institution_code) %>% 
  group_by(institution_code) %>% 
  mutate(SME_credit_3mma = rollmean(SME_credit, 3, fill = NA, align = "right"),
         largebus_credit_3mma = rollmean(largebus_credit, 3, fill = NA, align = "right"),
         SME_fixed_3mma = rollmean(SME_fixed, 3, fill = NA, align = "right"),
         large_fixed_3mma = rollmean(large_fixed, 3, fill = NA, align = "right"),
         totalbus_credit_3mma = rollmean(totalbus_credit, 3, fill = NA, align = "right"),
         fixed_total_3mma = rollmean(total_fixed_credit, 3, fill = NA, align = "right"))

# Base period calculation

fixed_base <- fixed_credit %>% 
  filter(reporting_period >= "2020-01-1" & reporting_period <= "2020-02-1" |  
           reporting_period >= "2019-12-30" & reporting_period <= "2020-01-1") %>%
  arrange(desc(reporting_period)) %>%
  distinct(institution_code, .keep_all = TRUE) %>%
  rename(SME_credit_base = SME_credit_3mma, 
         largebus_credit_base = largebus_credit_3mma,
         totalbus_credit_base = totalbus_credit_3mma,
         SME_fixed_base = SME_fixed_3mma,
         large_fixed_base = large_fixed_3mma,
         fixed_total_base = fixed_total_3mma) %>%
  select(institution_code, SME_credit_base, largebus_credit_base, 
         totalbus_credit_base, SME_fixed_base, large_fixed_base, fixed_total_base)

fixed_lending <- full_join(fixed_credit, fixed_base, by = c("institution_code"))

fixed_lending_prebind <- fixed_lending %>%
  mutate(time_dummy = ifelse(reporting_period >= "2020-03-31", 1, 0),
         SME_credit_growth_cum = (SME_credit / SME_credit_base) * 100 - 100,
         SME_credit_growth_cum_3mma = (SME_credit_3mma / SME_credit_base) * 100 - 100,
         largebus_credit_growth_cum = (largebus_credit / largebus_credit_base) * 100 - 100, 
         largebus_credit_growth_cum_3mma = (largebus_credit_3mma / largebus_credit_base) * 100 - 100,
         totalbus_credit_growth_cum = (totalbus_credit / totalbus_credit_base) * 100 - 100,
         totalbus_credit_growth_cum_3mma = (totalbus_credit / totalbus_credit_base) * 100 - 100,
         SME_fixed_growth_cum = (SME_fixed / SME_fixed_base) * 100 - 100,
         SME_fixed_growth_cum_3mma = (SME_fixed_3mma / SME_fixed_base) * 100 - 100,
         largebus_fixed_growth_cum = (large_fixed / large_fixed_base) * 100 - 100, 
         largebus_fixed_growth_cum_3mma = (large_fixed_3mma / large_fixed_base) * 100 - 100,
         totalbus_fixed_growth_cum = (total_fixed_credit / fixed_total_base) * 100 - 100,
         totalbus_fixed_growth_cum_3mma = (fixed_total_3mma / fixed_total_base) * 100 - 100)

sme_fixed_lending <- fixed_lending_prebind %>%
  mutate(business_credit = SME_credit,
         business_credit_base = SME_credit_base,
         credit_growth_cum = SME_credit_growth_cum,
         credit_growth_cum_3mma = SME_credit_growth_cum_3mma,
         business_credit_fixed = SME_fixed,
         business_credit_fixed_base = SME_fixed_base,
         credit_growth_cum_fixed = SME_fixed_growth_cum,
         credit_growth_cum_fixed_3mma = SME_fixed_growth_cum_3mma,
         SME_borrower = 1,
         borrower_type = "SME")

large_fixed_lending <- fixed_lending_prebind %>%
  mutate(business_credit = largebus_credit,
         business_credit_base = largebus_credit_base,
         credit_growth_cum = largebus_credit_growth_cum,
         credit_growth_cum_3mma = largebus_credit_growth_cum_3mma,
         business_credit_fixed = large_fixed,
         business_credit_fixed_base = large_fixed_base,
         credit_growth_cum_fixed = largebus_fixed_growth_cum,
         credit_growth_cum_fixed_3mma = largebus_fixed_growth_cum_3mma,
         SME_borrower = 0,
         borrower_type = "large business")

fixed_lending <- bind_rows(sme_fixed_lending, large_fixed_lending) %>% 
  rename(institution_name = name) %>% 
  arrange(institution_name) %>%
  mutate(time_SME_borrower = time_dummy * SME_borrower)

# Merge fixed-term lending with bank characteristics

fixed_lending <- left_join(fixed_lending, df_bank_characteristics, by = c("institution_code", "institution_name"))

# Exclude outliers for main regression model and apply date filters

fixed_lending <- fixed_lending %>%
  filter(institution_code != "Outlier 1") %>%
  filter(institution_code != "Outlier 2") %>%
  filter(reporting_period >= first_month & reporting_period <= last_month) %>% 
  rename(period = reporting_period) 

## TFF usage

fixed_lending <- fixed_lending %>% 
  left_join(tff_usage)

# Time and institution dummies

fixed_lending <- fixed_lending %>%  
  mutate(institution_type_dummy = ifelse(institution_type == "ADI", 1, 0),
         time_dummy = ifelse(period >= "2020-03-31", 1, 0))


# Fixed-term lending just banks
bank_fixed_lending <- fixed_lending %>% 
  filter(reporter_type != "Non-ADIs")

rm(arf742_raw, arf742_raw_adjusted, sme_fixed_lending, large_fixed_lending, 
   fixed_lending_prebind, fixed_credit, fixed_base, breaks)
