function unbounded = checkBoundedIS_Read(F,S)
% Function checks whether there exists a value of q satisfying the
% identifying restrictions such that the identified set for the normalising
% impulse response includes zero. 
% See Algorithm 4.1 in Read M (2022), 'Algorithms for Inference in SVARs
% Identified with Sign and Zero Restrictions', Econometrics Journal.
% Inputs:
% - F: matrix containing cofficients of zero restrictions
% - S: matrix containing coefficients of sign restrictions

% Function assumes that first sign restriction constraints impact response
% of first variable to be nonnegative (or nonpositive).

% Add sign restriction on impact response of first variable to first shock
% to set of zero restrictions.
F = [F; S(1,:)];
S = S(2:end,:);

n = size(F,2); % Number of variables in VAR
r = size(F,1); % Number of zero restrictions (plus additional binding sign restriction)
s = size(S,1); % Number of remaining sign restrictions (including normalisation)

% Options for linear programming routine.
lpoptimOptions = optimoptions('linprog');
lpoptimOptions.Display = 'off';

%% Transform restrictions so that algorithm from AD21 can be applied.
% Compute change of basis matrix.
K = [null(F), null(null(F)')];

% Apply change of basis to coefficient vectors of active restrictions.
Stilde = (K'*S')'; % Note that K^{-1} = K' since K is orthonormal

% Project sign restrictions onto hyperplane generated by zero restrictions.
b = [zeros(r,n-r), eye(r)]';
B = eye(n) - b*((b'*b)\b');
Sbar = (B*Stilde')';

Sbar = Sbar(:,1:(n-r)); % Drop last f elements (which are zero)

%% Solve for Chebyshev centre.
% Find the centre and radius of the largest ball that can be inscribed
% within the intersection of the half-spaces generated by the (projected)
% sign restrictions and the unit cube in n-r dimensions.

A = zeros(s+1,n-r+1);
A(1:end-1,2:end) = -Sbar;

for ii = 1:(s+1)

    A(ii,1) = norm(A(ii,2:end));

end

A(end,1) = -1; % Add constraint that radius is positive

% Add additional restrictions that ball lies inside unit (n-f)-cube.
A = [A; [ones(n-r,1), eye(n-r)]; [ones(n-r,1), -eye(n-r)]];

% Problem is to maximise a*x s.t. A*x <= b, with x = (R,c')'.
a = [-1, zeros(1,n-r)]; % 
b = [zeros(s+1,1); ones(2*(n-r),1)];
x = linprog(a,A,b,[],[],[],[],lpoptimOptions);

% Check if radius is positive (i.e. if set is nonempty).
if x(1) > 0
    
    unbounded = 1; % Conclude identified sets to unit shock are unbounded
    
else
    
    unbounded = 0;

end

end