% Conduct robust Bayesian inference about the impulse responses to a unit
% monetary policy shock under the identifying restrictions in Arias,
% Caldara and Rubio-Ramirez (2019).

clear variables
close all
clc

oldFolder = pwd;
addpath('auxFunctions');
cd results
resultsDir = pwd;
cd(oldFolder);

%% Import data.
% Data are obtained from the replication files to Antolin-Diaz and
% Rubio-Ramirez (2018), available on Juan Antolin-Diaz's website:
% https://sites.google.com/site/juanantolindiaz/
% The variables are:
% 1) Real GDP 2) GDP deflator 3) Commodity price index 4) Total reserves
% 5) Nonborrowed reserves 6) Federal funds rate
load('Uhlig_Data_Updated.mat');

% Re-order variables so that the federal funds rate is ordered first (the 
% monetary policy shock is the shock of interest).
data = data(:,[end 1:end-1]);
data(:,2:end) = data(:,2:end)*100;
varNames = {varNames{end}, varNames{1:end-1}};

% Declare any exogenous variables (other than the constant; leave empty 
% if none).
exog = [];

%% Options.
opt.p = 12; % No. of lags in VAR
opt.const = 1; % const = 1 if constant in VAR, = 0 otherwise
opt.ivar = [1 2 3 4 5 6];  % Indices of variables of interest
opt.cumIR = []; % Indices of variables for which cumulative impulse response should be used
opt.H = 60; % Terminal horizon for impulse responses
opt.nonEmpty = 10000; % No. of draws from posterior of phi with non-empty identified set
opt.aalpha = 0.68; % Credibility level for credible intervals
opt.Qdraws = 10000; % Number of draws of Q when approximating identified set
opt.dispIter = 100; % Print number of draws remaining every dispIter draws

rng(23032021); % Set seed for random number generator

%% Identifying restrictions from Arias, Caldara and Rubio-Ramirez (2019).

% Input identifying restrictions.
% Each row of eqRestr contains a vector (i,j,t) representing a 
% particular equality restriction, where t is the type of restriction:
% t = 1: the (ij)th element of A0 is zero
% t = 2: the (ij)th element of A0^(-1) is zero
restr.eqRestr = ...
    [1 5 1; % A0_15 = 0 - fed funds rate does not respond to total reserves
     1 6 1]; % A0_16 = 0 - fed funds rate does not respond to non-borrowed reserves

% Each row of signRestr contains a quintet (i,j,h,s,t) representing a
% particular sign restriction, where t is the type of restriction:
% t = 1: the IR of the ith variable with respect to the jth shock at the 
% hth horizon is nonnegative (s = 1) or nonpositive (s = -1).
% t = 2: the (ij)th element of A0 is nonnegative (s = 1) or nonpositive 
% (s = -1). 
% Warning: signRestr should NOT include sign normalisations on elements of 
% A0. These are imposed by default.
% Functions assume that the first sign restriction restricts the impact
% response of the first variable to the first shock to be nonnegative.
restr.signRestr = ...
   [1 1 0 1 1; % Response of fed funds rate to its own shock is nonnegative
    1 2 0 -1 2; % Response of fed funds rate to output is nonnegative
    1 3 0 -1 2]; % Response of fed funds rate to prices is nonnegative

% Conduct robust Bayesian inference.
mainfile_q1;

% Compute equi-tailed 90 per cent credible intervals under single prior.
etaUnitCredlb90 = permute(prctile(etaUnitDraws,100*(1-0.90)/2,1),[2 3 1]);
etaUnitCredub90 = permute(prctile(etaUnitDraws,100*(1+0.90)/2,1),[2 3 1]);

fprintf('\nIdentified set unbounded in %1.3g per cent of draws.\n',mean(unbounded)*100)

% Export results to Excel for graphing.
cd(resultsDir);

% Create table of results for impulse responses.
for ii = 1:length(opt.ivar)
    
    TT1 = table((0:opt.H)',etaUnitMed(:,ii),etaUnitCredlb(:,ii),...
        etaUnitCredub(:,ii),etaUnitCredlb90(:,ii),etaUnitCredub90(:,ii));
    TT1.Properties.VariableNames = ...
        {'Horizon','Median','CREDLB68','CREDUB68','CREDLB90','CREDUB90'};
    writetable(TT1,'FigureData.xlsx','Sheet',strcat(varNames{ii},'_ACR'));
    
end

save('ACR_results.mat');

cd(oldFolder);

%% Add sign restrictions on impulse responses from Uhlig (2005).
restr.signRestr = ...
    [restr.signRestr; 
       1 1 1 1 1; % Response of ffr to monetary policy shock after one month is nonnegative
       1 1 2 1 1; % As above after two months
       1 1 3 1 1; % As above after three months
       1 1 4 1 1; % As above after four months
       1 1 5 1 1; % As above after five months
       3 1 0 -1 1; % Response of gdpdef to monetary policy shock is nonpositive
       3 1 1 -1 1; % As above after one month, etc
       3 1 2 -1 1;
       3 1 3 -1 1;
       3 1 4 -1 1;
       3 1 5 -1 1;
       4 1 0 -1 1; % Response of com to monetary policy shock is nonpositive
       4 1 1 -1 1; % As above after one month, etc
       4 1 2 -1 1;
       4 1 3 -1 1;
       4 1 4 -1 1;
       4 1 5 -1 1;       
       6 1 0 -1 1; % Response of nbr to monetary policy shock is nonpositive
       6 1 1 -1 1; % As above after one month, etc
       6 1 2 -1 1;
       6 1 3 -1 1;
       6 1 4 -1 1;
       6 1 5 -1 1];   
   
% Conduct robust Bayesian inference.
mainfile_q1;

fprintf('\nIdentified set unbounded in %1.3g per cent of draws.\n',mean(unbounded)*100)

% Export results to Excel for graphing.
cd(resultsDir);

% Create table of results for impulse responses.
for ii = 1:length(opt.ivar)
    
    TT1 = table((0:opt.H)',etaUnitMed(:,ii),etaUnitCredlb(:,ii),etaUnitCredub(:,ii),...
        etaUnitMedlb(:,ii),etaUnitMedub(:,ii),etaUnitRClb(:,ii),etaUnitRCub(:,ii));
    TT1.Properties.VariableNames = {'Horizon','Median','CREDLB','CREDUB','MEDLB','MEDUB','RCREDLB','RCREDUB'};
    writetable(TT1,'FigureData.xlsx','Sheet',strcat(varNames{ii},'_ACRUhlig'));
    
end

% Compute posterior lower and upper probabilities that output declines by
% more than a given threshold.

thresh = [0 -0.25 -0.5 -1];
hors = [1 12 24 36 48]';

postLow = zeros(length(hors),length(thresh));
postUp = zeros(length(hors),length(thresh));
for hh = 1:length(hors)
postLow(hh,:) = mean(etaUnitub(:,hors(hh),2) <= thresh);
postUp(hh,:) = mean(etaUnitlb(:,hors(hh),2) <= thresh);
end

xlswrite('PosteriorProbs',[postLow, postUp],'Restriction2');

save('ACRUhlig_results.mat');

cd(oldFolder);