# Functions to Reformat Data for mops Package
# Dmitry Titkov / MPI / DM
# November 2020

#' Reformat Yieldbroker auction data to use for Findur trade entry
#'
#' @description This function produces trade details that are ready for copying and pasting into Findur for the RBA's bond purchases, based on raw bond auction output from Yieldbroker. The function uses metadata for AGS and semis from vw_USER_Issues_All, and the counterparty codes mapping in D20/330276. Source: RBA.
#'
#' @param path Set this to the path for the spreadsheet containing the raw Yieldbroker output. The Findur trade details will be saved as a CSV into the same folder as the spreadsheet.
#' @param purpose (Default = QE.) Change this to 'Bond Purchase Programme' if the bond purchases are instead in support of the RBA's 3-year AGS target or market functioning.
#'
#' @family functions that reformat data

reformat_yieldbroker_to_findur <- function (path, purpose = "QE") {

  # Load Yieldbroker auction data ####
  yieldbroker_input_raw <- readxl::read_excel(path)

  # Load metadata ####
  metadata_ags_semis <- mops::load_ags_semis_metadata()
  metadata_counterparties_url <- "https://trimweb.rba.gov.au/record/7073127"
  metadata_counterparties_file <- tempfile(fileext = ".xlsx")
  download.file(metadata_counterparties_url,
                metadata_counterparties_file,
                mode = "wb",
                quiet = TRUE)
  metadata_counterparties <- readxl::read_excel(metadata_counterparties_file)

  # Join up metadata ####
  yieldbroker_input_renamed <- dplyr::rename(yieldbroker_input_raw,
                                             cusip = ISIN,
                                             yieldbroker_counterparty = Counterparty)
  yieldbroker_input_joined <- dplyr::left_join(yieldbroker_input_renamed, metadata_ags_semis,
                                               by = "cusip")
  yieldbroker_input <- dplyr::left_join(yieldbroker_input_joined, metadata_counterparties,
                                        by = "yieldbroker_counterparty")
  yieldbroker_input_filtered <- dplyr::filter(yieldbroker_input,
                                              `Transaction Type` == "Outright")

  # Reformat data to use for Findur trade entry ####
  findur_output_arranged <- dplyr::arrange(yieldbroker_input_filtered,
                                           ticker)
  findur_output_mutated <- dplyr::mutate(findur_output_arranged,
                                         dm_purpose = purpose,
                                         findur_ticker = ticker,
                                         settle_date = lubridate::as_date(`Settlement Date`),
                                         yield = as.numeric(Yield),
                                         face_value = Volume / 1e6,
                                         yieldbroker_ticker = Security,
                                         maturity = format(maturity, "%b-%y"),
                                         proceeds = Consideration)
  findur_output_selected <- dplyr::select(findur_output_mutated,
                                          dm_purpose,
                                          findur_ticker,
                                          findur_counterparty,
                                          settle_date,
                                          yield,
                                          face_value,
                                          yieldbroker_counterparty,
                                          yieldbroker_ticker,
                                          issuer,
                                          coupon,
                                          maturity,
                                          proceeds)
  findur_output <- dplyr::rename(findur_output_selected,
                                 `DM Purpose` = dm_purpose,
                                 Ticker = findur_ticker,
                                 Counterparty = findur_counterparty,
                                 `Settle Date` = settle_date,
                                 Yield = yield,
                                 `Face Value ($m)` = face_value,
                                 `YB Counterparty` = yieldbroker_counterparty,
                                 `YB Ticker` = yieldbroker_ticker,
                                 `Check Issuer` = issuer,
                                 `Check Coupon` = coupon,
                                 `Check Maturity` = maturity,
                                 `Check Proceeds` = proceeds)

  if (sum(is.na(findur_output$Ticker)) != 0) {
    message("At least one bond ticker is missing! Please enter any trades with missing tickers manually")
  }

  if (sum(is.na(findur_output$Counterparty)) != 0) {
    message("At least one counterparty code needs to be clarified! Please use Yieldbroker to determine the correct counterparty for trade entry")
  }

  if ((nrow(yieldbroker_input) - nrow(yieldbroker_input_filtered)) != 0) {
    message("At least one trade was filtered out! Please check whether any switches were done today")
  }

  message(paste0("The output for Findur contains ",
                 nrow(findur_output),
                 " trades to purchase $",
                 sum(findur_output$`Face Value ($m)`) / 1e3,
                 "b of bonds (in face value terms) from these issuers: ",
                 paste(unique(findur_output$`Check Issuer`), collapse = ", "),
                 " - does this sound right?"))

  folder <- dirname(path)
  write.csv(findur_output, paste0(folder, "/findur_output.csv"), row.names = FALSE)
  return(findur_output)

}
