# Functions to Load Data for mops Package
# Dmitry Titkov and Alex Holmes / MPI / DM
# July 2021

# Metadata ####

#' Load metadata for AGS and semis
#'
#' @description This function loads some cleaned-up Findur metadata for instruments issued by the Commonwealth, and the state and territory central borrowing authorities. It is used in the functions that load AGS and semis yields. Source: RBA. Table: vw_USER_Issues_All (and ab_tran, ab_tran_info and header).
#'
#' @export
#'
#' @family functions that load data

load_ags_semis_metadata <- function () {

  # Load metadata from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  metadata_raw <- odbc::dbGetQuery(conn_findur,
                                   "select *
                                   from vw_USER_Issues_All
                                   where ISSUER in ('AUSC ISSUER',
                                                    'NSWT ISSUER',
                                                    'TCVM ISSUER',
                                                    'QLDT ISSUER',
                                                    'WATC ISSUER',
                                                    'SAFA ISSUER',
                                                    'TASC ISSUER',
                                                    'ACTT ISSUER',
                                                    'NTTY ISSUER')")
  metadata_gg_raw <- odbc::dbGetQuery(conn_findur,
                                      "select distinct h.ticker
                                      from ab_tran t
                                      inner join header h on h.ins_num = t.ins_num
                                      inner join ab_tran_info ti on t.tran_num = ti.tran_num
                                      where ti.type_id in (20060, 20061)
                                      and ti.value = 'Yes'")
  odbc::dbDisconnect(conn_findur)

  # Clean up metadata ####
  metadata_tibbled_gg <- tidyr::as_tibble(metadata_gg_raw)
  metadata_mutated_gg <- dplyr::mutate(metadata_tibbled_gg,
                                       gg = TRUE)
  metadata_tibbled <- tidyr::as_tibble(metadata_raw)
  metadata_mutated <- dplyr::mutate(metadata_tibbled,
                                    type = dplyr::case_when(ISSUE_TYPE %in% c("TB", "TIB", "TN") ~ "AGS",
                                                            ISSUE_TYPE %in% c("FIS", "Index FIS", "PN", "Tax Free") ~ "Semis",
                                                            TRUE ~ as.character(NA)),
                                    subtype = dplyr::case_when(ISSUE_TYPE %in% c("TB", "FIS", "Tax Free") ~ "Nominals",
                                                               ISSUE_TYPE %in% c("TIB", "Index FIS") ~ "Linkers",
                                                               ISSUE_TYPE %in% c("TN", "PN") ~ "Notes",
                                                               TRUE ~ as.character(NA)),
                                    maturity = lubridate::as_date(MATURITY_DATE),
                                    issued = lubridate::as_date(ISSUE_DATE),
                                    coupon = RATE,
                                    ticker = SERIES,
                                    issuer = dplyr::recode(ISSUER,
                                                           "AUSC ISSUER" = "Cth",
                                                           "NSWT ISSUER" = "NSW",
                                                           "TCVM ISSUER" = "Vic",
                                                           "QLDT ISSUER" = "Qld",
                                                           "WATC ISSUER" = "WA",
                                                           "SAFA ISSUER" = "SA",
                                                           "TASC ISSUER" = "Tas",
                                                           "ACTT ISSUER" = "ACT",
                                                           "NTTY ISSUER" = "NT"),
                                    cusip = CUSIP)
  metadata_selected <- dplyr::select(metadata_mutated,
                                     type,
                                     subtype,
                                     issuer,
                                     ticker,
                                     maturity,
                                     issued,
                                     coupon,
                                     cusip)
  metadata_joined <- dplyr::left_join(metadata_selected,
                                      metadata_mutated_gg,
                                      by = "ticker")
  metadata_remutated <- dplyr::mutate(metadata_joined,
                                      gg = dplyr::case_when(is.na(gg) ~ FALSE,
                                                            TRUE ~ gg))

  return(metadata_remutated)

}

# AGS ####

#' Load recent intraday yields for AGS
#'
#' @description This function loads recent yields for nominal and inflation-linked AGS on a 5-minutely intraday frequency from Findur. Source: Yieldbroker (via Refinitiv). Table: USER_DM_YLD_IntradayBonds.
#'
#' @param date_start (Default = 100 calendar days prior to today.) Change this if you need a longer time series, though this will be at the cost of some speed.
#' @param value_types (Default = mid yield.) Change this if you need to see bids and/or offers.
#'
#' @export
#'
#' @family functions that load data

load_ags_yields_intraday <- function (date_start = lubridate::today() - 100, value_types = c("Mid")) {

  # Load data from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur, paste0(
                               "select *
                               from USER_DM_YLD_IntradayBonds
                               where convert(date, Date) >= '",
                               date_start,
                               "'"))
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = lubridate::as_date(Date),
                                datetime = lubridate::as_datetime(Time),
                                ticker = IssueId)
  data_selected <- dplyr::select(data_mutated,
                                 date,
                                 datetime,
                                 ticker,
                                 Bid,
                                 Offer,
                                 Mid)
  data_gathered <- tidyr::gather(data_selected,
                                 "value_type", "value",
                                 -date, -datetime, -ticker)
  data_filtered <- dplyr::filter(data_gathered,
                                 value_type %in% value_types)

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")

  return(data_joined)

}

#' Load historical intraday yields for AGS
#'
#' @description This function loads historical yields for nominal and inflation-linked AGS on a 5-minutely intraday frequency from Findur. Source: Yieldbroker (via Refinitiv). Table: USER_DM_YLD_IntradayBonds.
#'
#' @param dates A vector of the dates for which you need the yields, e.g. dates = c("2020-03-19", "2020-11-03").
#' @param value_types (Default = mid yield.) Change this if you need to see bids and/or offers.
#'
#' @export
#'
#' @family functions that load data

load_ags_yields_intraday_hist <- function (dates, value_types = c("Mid")) {

  # Load data from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur, paste0(
                               "select *
                               from USER_DM_YLD_IntradayBonds
                               where convert(date, Date) in ('",
                               paste(dates, collapse = "', '"),
                               "')"))
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = lubridate::as_date(Date),
                                datetime = lubridate::as_datetime(Time),
                                ticker = IssueId)
  data_selected <- dplyr::select(data_mutated,
                                 date,
                                 datetime,
                                 ticker,
                                 Bid,
                                 Offer,
                                 Mid)
  data_gathered <- tidyr::gather(data_selected,
                                 "value_type", "value",
                                 -date, -datetime, -ticker)
  data_filtered <- dplyr::filter(data_gathered,
                                 value_type %in% value_types)

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")

  return(data_joined)

}

#' Load opening yields for AGS
#'
#' @description This function loads daily yields at a particular time, which defaults to 8.30 am, for nominal and inflation-linked AGS from Findur. Source: Yieldbroker (via Refinitiv). Table: USER_DM_YLD_IntradayBonds.
#'
#' @param time_set (Default = 8.30 am.) Change this if you want to use a different snapshot.
#' @param metadata_required (Default = true.) Change this if you do not want instruments lacking metadata to be filtered out. For example, metadata are not readily available for TBs prior to TB110.
#' @param value_types (Default = mid yield.) Change this if you need to see bids and/or offers.
#'
#' @export
#'
#' @family functions that load data

load_ags_yields_open <- function (time_set = "08:30", metadata_required = TRUE, value_types = c("Mid")) {

  # Load data from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur, paste0(
                               "select *
                               from USER_DM_YLD_IntradayBonds
                               where convert(time, Time) = '",
                               time_set,
                               "'"))
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = lubridate::as_date(Date),
                                datetime = lubridate::as_datetime(Time),
                                ticker = IssueId)
  data_selected <- dplyr::select(data_mutated,
                                 date,
                                 datetime,
                                 ticker,
                                 Bid,
                                 Offer,
                                 Mid)
  data_gathered <- tidyr::gather(data_selected,
                                 "value_type", "value",
                                 -date, -datetime, -ticker)
  data_filtered <- dplyr::filter(data_gathered,
                                 value_type %in% value_types)

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")
  if (metadata_required == TRUE) {
    data_joined <- dplyr::filter(data_joined,
                                 !is.na(type))
  }

  return(data_joined)

}

#' Load closing yields for AGS
#'
#' @description This function loads closing yields for nominal and inflation-linked AGS from Findur, as published in F16. Sources: from 20 May 2013, Yieldbroker (via Refinitiv); prior to then, RBA (via a survey of bond dealers). Table: USER_DM_YLD_SurveyCGS.
#'
#' @param metadata_required (Default = true.) Change this if you do not want instruments lacking metadata to be filtered out. For example, metadata are not readily available for TBs prior to TB110.
#' @param calc_types (Default = RBA.) Change this if you want to see data other than those published in F16. 'Average' is an alternative setting with a longer time series.
#' @param value_types (Default = mid yield.) Change this if you need to see bids and/or offers, though these are unavailable for the 'RBA' calc type.
#'
#' @export
#'
#' @family functions that load data

load_ags_yields_close <- function (metadata_required = TRUE, calc_types = "RBA", value_types = c("Mid")) {

  # Load data from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur,
                               "select *
                               from USER_DM_YLD_SurveyCGS")
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = lubridate::as_date(Date),
                                calc_type = Party,
                                ticker = IssueID)
  data_selected <- dplyr::select(data_mutated,
                                 date,
                                 calc_type,
                                 ticker,
                                 Bid,
                                 Offer,
                                 Mid)
  data_gathered <- tidyr::gather(data_selected,
                                 "value_type", "value",
                                 -date, -calc_type, -ticker)
  data_filtered <- dplyr::filter(data_gathered,
                                 calc_type %in% calc_types,
                                 value_type %in% value_types)

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")
  if (metadata_required == TRUE) {
    data_joined <- dplyr::filter(data_joined,
                                 !is.na(type))
  }

  return(data_joined)

}

# Semis ####

#' Load recent intraday yields for semis
#'
#' @description This function loads recent yields for nominal and inflation-linked semis on a 5-minutely intraday frequency from Findur. Source: Yieldbroker (via Refinitiv). Table: USER_DM_YLD_IntradaySemis.
#'
#' @param date_start (Default = 35 calendar days prior to today.) Change this if you need a longer time series, though this will be at the cost of some speed.
#' @param value_types (Default = mid yield.) Change this if you need to see bids and/or offers.
#'
#' @export
#'
#' @family functions that load data

load_semis_yields_intraday <- function (date_start = lubridate::today() - 35, value_types = c("Mid")) {

  # Load data from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur, paste0(
                               "select *
                               from USER_DM_YLD_IntradaySemis
                               where convert(date, Date) >= '",
                               date_start,
                               "'"))
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = lubridate::as_date(Date),
                                datetime = lubridate::as_datetime(Time),
                                ticker = IssueId)
  data_selected <- dplyr::select(data_mutated,
                                 date,
                                 datetime,
                                 ticker,
                                 Bid,
                                 Offer,
                                 Mid)
  data_gathered <- tidyr::gather(data_selected,
                                 "value_type", "value",
                                 -date, -datetime, -ticker)
  data_filtered <- dplyr::filter(data_gathered,
                                 value_type %in% value_types)

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")

  return(data_joined)

}

#' Load historical intraday yields for semis
#'
#' @description This function loads historical yields for nominal and inflation-linked semis on a 5-minutely intraday frequency from Findur. Source: Yieldbroker (via Refinitiv). Table: USER_DM_YLD_IntradaySemis.
#'
#' @param dates A vector of the dates for which you need the yields, e.g. dates = c("2021-02-02", "2021-07-06").
#' @param value_types (Default = mid yield.) Change this if you need to see bids and/or offers.
#'
#' @export
#'
#' @family functions that load data

load_semis_yields_intraday_hist <- function (dates, value_types = c("Mid")) {

  # Load data from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur, paste0(
                               "select *
                               from USER_DM_YLD_IntradaySemis
                               where convert(date, Date) in ('",
                               paste(dates, collapse = "', '"),
                               "')"))
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = lubridate::as_date(Date),
                                datetime = lubridate::as_datetime(Time),
                                ticker = IssueId)
  data_selected <- dplyr::select(data_mutated,
                                 date,
                                 datetime,
                                 ticker,
                                 Bid,
                                 Offer,
                                 Mid)
  data_gathered <- tidyr::gather(data_selected,
                                 "value_type", "value",
                                 -date, -datetime, -ticker)
  data_filtered <- dplyr::filter(data_gathered,
                                 value_type %in% value_types)

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")

  return(data_joined)

}

#' Load opening yields for semis
#'
#' @description This function loads daily yields at a particular time, which defaults to 8.30 am, for nominal and inflation-linked semis from Findur. Source: Yieldbroker (via Refinitiv). Table: USER_DM_YLD_IntradaySemis.
#'
#' @param time_set (Default = 8.30 am.) Change this if you want to use a different snapshot.
#' @param value_types (Default = mid yield.) Change this if you need to see bids and/or offers.
#'
#' @export
#'
#' @family functions that load data

load_semis_yields_open <- function (time_set = "08:30", value_types = c("Mid")) {

  # Load data from Findur ####
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur, paste0(
                               "select *
                               from USER_DM_YLD_IntradaySemis
                               where convert(time, Time) = '",
                               time_set,
                               "'"))
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = lubridate::as_date(Date),
                                datetime = lubridate::as_datetime(Time),
                                ticker = IssueId)
  data_selected <- dplyr::select(data_mutated,
                                 date,
                                 datetime,
                                 ticker,
                                 Bid,
                                 Offer,
                                 Mid)
  data_gathered <- tidyr::gather(data_selected,
                                 "value_type", "value",
                                 -date, -datetime, -ticker)
  data_filtered <- dplyr::filter(data_gathered,
                                 value_type %in% value_types)

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")

  return(data_joined)

}

#' Load closing yields for semis
#'
#' @description This function loads closing yields for semi-government bonds from Findur. Source: Yieldbroker (via Refinitiv). Tables: USER_DM_YLD_SurveySemis (which is linked to the CVR); USER_DM_YLD_IntradaySemis (for the latest day's data).
#'
#' @param metadata_required (Default = true.) Change this if you do not want instruments lacking metadata to be filtered out.
#'
#' @export
#'
#' @family functions that load data

load_semis_yields_close <- function (metadata_required = TRUE) {

  # Load data from Findur ####
  date_min <- lubridate::today() - 7
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  data_raw <- odbc::dbGetQuery(conn_findur,
                               "select *
                               from USER_DM_YLD_SurveySemis")
  data_latest_raw <- odbc::dbGetQuery(conn_findur,
                                      paste0("select *
                                             from USER_DM_YLD_IntradaySemis
                                             where convert(time, Time) = '16:40'
                                             and convert(date, Date) >= '",
                                             date_min,
                                             "'"))
  odbc::dbDisconnect(conn_findur)

  # Clean up data ####
  date_max <- max(data_raw$Date)
  data_tibbled <- tidyr::as_tibble(data_raw)
  data_latest_tibbled <- tidyr::as_tibble(data_latest_raw)
  data_latest_renamed <- dplyr::rename(data_latest_tibbled,
                                       IssueID = IssueId,
                                       Yield = Mid)
  data_latest_filtered <- dplyr::filter(data_latest_renamed,
                                        Date > date_max)
  data_latest_mutated <- dplyr::mutate(data_latest_filtered,
                                       date = lubridate::as_date(Date))
  data_mutated <- dplyr::mutate(data_tibbled,
                                date = dplyr::case_when(lubridate::as_date(Date) == lubridate::as_date("1900-01-01") ~ lubridate::as_date(Modify_DateTime),
                                                        TRUE ~ lubridate::as_date(Date)))
  data_binded <- dplyr::bind_rows(data_mutated, data_latest_mutated)
  data_renamed <- dplyr::rename(data_binded,
                                ticker = IssueID,
                                yield = Yield)
  data_selected <- dplyr::select(data_renamed,
                                 date,
                                 ticker,
                                 yield)
  data_filtered <- dplyr::filter(data_selected,
                                 ticker != "IssueID")

  # Load and join up metadata ####
  metadata <- mops::load_ags_semis_metadata()
  data_joined <- dplyr::left_join(data_filtered, metadata,
                                  by = "ticker")
  if (metadata_required == TRUE) {
    data_joined <- dplyr::filter(data_joined,
                                 !is.na(type))
  }

  return(data_joined)

}

# Cash rate ####

#' Load cash rate target changes
#'
#' @description This function loads the historical changes in the cash rate target from Findur, starting from 1990. Source: RBA. Table: USER_DM_YLD_TargetCash.
#'
#' @export
#'
#' @family functions that load data

load_cash_rate_target_changes <- function () {
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  crt_data <- odbc::dbGetQuery(conn_findur,
                               "SELECT USER_DM_YLD_TargetCash.Date, USER_DM_YLD_TargetCash.Target
                               FROM Findur_Prod.dbo.USER_DM_YLD_TargetCash USER_DM_YLD_TargetCash
                               ORDER BY USER_DM_YLD_TargetCash.Date")
  crt_data_tibbled <- dplyr::as_tibble(crt_data)
  crt_data_mutated <- dplyr::mutate(crt_data_tibbled,
                                    Date = lubridate::as_date(Date))
  crt_data_renamed <- dplyr::rename(crt_data_mutated,
                                    date = Date,
                                    crt = Target)
  return(crt_data_renamed)
}

# OIS rates ####

#' Load intraday OIS rates
#'
#' @description This function loads intraday OIS rates, starting from 2001 and for tenors out to 5 years. Source: Tullett Prebon previously; and Fenics currently. Table: USER_DM_YLD_IntradayOIS.
#'
#' @export
#'
#' @family functions that load data

load_ois_rates_intraday <- function () {
  conn_findur <- odbc::dbConnect(odbc::odbc(), "Findur_R")
  ois_data <- odbc::dbGetQuery(conn_findur,
                               "SELECT  Date,
                                        Time,
                                        Maturity,
                                        Mid
                               FROM USER_DM_YLD_IntradayOIS")
  ois_data_tibbled <- dplyr::as_tibble(ois_data)
  ois_data_filtered <- dplyr::filter(ois_data_tibbled,
                                     Maturity != "Matur")
  ois_data_mutated <- dplyr::mutate(ois_data_filtered,
                                    date = lubridate::as_date(Date),
                                    datetime = lubridate::as_datetime(Time),
                                    maturity_months = suppressWarnings(dplyr::case_when(stringr::str_detect(Maturity, "w") ~ as.numeric(stringr::str_remove(stringr::str_remove(Maturity, "w"), "k")) / 4,
                                                                                        stringr::str_detect(Maturity, "m") ~ as.numeric(stringr::str_remove(Maturity, "m")),
                                                                                        stringr::str_detect(Maturity, "y") ~ as.numeric(stringr::str_remove(Maturity, "y")) * 12,
                                                                                        TRUE ~ as.numeric(NA))),
                                    ois = Mid)
  ois_data_selected <- dplyr::select(ois_data_mutated,
                                     date, datetime, maturity_months, ois)
  return(ois_data_selected)
}
