# readaofm: Read Positions, Issuance and Buybacks Data from the AOFM's Data Hub
# Dmitry Titkov and Kimia Ghomashchi / MO / DM
# October 2020

#' Clean up AOFM positions data
#'
#' @description This function is used within aofm_data() to clean up positions data. There's no need to use it on its own.
#'
#' @param positions_data The output of read_excel() for a particular tab of the AOFM's spreadsheet.
#' @param set_sheet What that tab contains (e.g. 'fv').
#'
#' @export
#'
#' @family cleaning functions

clean_positions_data <- function (positions_data, set_sheet) {
  tibbled <- dplyr::as_tibble(positions_data,
                              .name_repair = "unique")
  if (tibbled[2, 1] != "Coupon (%)") { # to address the fact that T-notes don't pay coupons
    coupon <- as.list(rep(NA, ncol(tibbled)))
    tibbled <- DataCombine::InsertRow(tibbled, NewRow = coupon, RowNum = 2)
    tibbled[2, 1] <- "Coupon (%)"
  }
  colnames(tibbled) <- make.unique(paste(tibbled[1, ], tibbled[2, ]), sep = "_")
  dropped <- tibbled[-(1:2), -ncol(tibbled)]
  set_origin <- dplyr::if_else(lubridate::is.POSIXt(dropped$`Maturity Coupon (%)`[1]), as.character(NA), "1899-12-30") # checks date formatting and adjusts it if Excel has botched something
  gathered <- tidyr::gather(dropped,
                            "instrument", "value",
                            -`Maturity Coupon (%)`)
  mutated <- dplyr::mutate(gathered,
                           date = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(`Maturity Coupon (%)`),
                                                   !is.na(set_origin) ~ lubridate::as_date(as.numeric(`Maturity Coupon (%)`), origin = set_origin),
                                                   TRUE ~ lubridate::as_date(NA)),
                           instrument = stringr::str_remove(instrument, "_1"),
                           value = as.numeric(value))
  separated <- tidyr::separate(mutated,
                               instrument, c("maturity", "coupon"),
                               sep = " ")
  remutated <- dplyr::mutate(separated,
                             maturity = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(maturity),
                                                         !is.na(set_origin) ~ lubridate::as_date(as.numeric(maturity), origin = set_origin),
                                                         TRUE ~ lubridate::as_date(NA)),
                             coupon = as.numeric(coupon),
                             value_type = set_sheet)
  selected <- dplyr::select(remutated,
                            date, maturity, coupon, value, value_type)
}

#' Clean up AOFM issuance data
#'
#' @description This function is used within aofm_data() to clean up issuance data. There's no need to use it on its own.
#'
#' @param issuance_data The output of read_excel() for the AOFM's spreadsheet.
#'
#' @export
#'
#' @family cleaning functions

clean_issuance_data <- function (issuance_data) {
  tibbled <- dplyr::as_tibble(issuance_data)
  dropped <- tibbled[-1, ]
  set_origin <- dplyr::if_else(lubridate::is.POSIXt(dropped$`Date Held`[1]), as.character(NA), "1899-12-30") # checks date formatting and adjusts it if Excel has botched something
  if (!("Coupon" %in% colnames(dropped))) { # again, no coupons for T-notes
    dropped <- dplyr::mutate(dropped,
                             Coupon = as.numeric(NA))
  }
  gathered <- tidyr::gather(dropped,
                            "value_type", "value",
                            -`Date Held`, -`Date Settled`, -`Tender Number`, -Maturity, -Coupon, -ISIN)
  mutated <- dplyr::mutate(gathered,
                           date_issue = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(`Date Held`),
                                                         !is.na(set_origin) ~ lubridate::as_date(as.numeric(`Date Held`), origin = set_origin),
                                                         TRUE ~ lubridate::as_date(NA)),
                           date_settle = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(`Date Settled`),
                                                          !is.na(set_origin) ~ lubridate::as_date(as.numeric(`Date Settled`), origin = set_origin),
                                                          TRUE ~ lubridate::as_date(NA)),
                           tender = `Tender Number`,
                           maturity = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(Maturity),
                                                       !is.na(set_origin) ~ lubridate::as_date(as.numeric(Maturity), origin = set_origin),
                                                       TRUE ~ lubridate::as_date(NA)),
                           coupon = as.numeric(Coupon),
                           isin = ISIN,
                           value = as.numeric(value),
                           value_type = dplyr::recode(value_type,
                                                      "Amount Offered" = "amount_offer",
                                                      "Amount Allotted" = "amount_issue",
                                                      "Amount of Bids" = "amount_bid",
                                                      "Coverage Ratio" = "coverage_ratio",
                                                      "Weighted Average Issue Yield" = "yield_wm_issue",
                                                      "Lowest Accepted Yield" = "yield_min_issue",
                                                      "Highest Accepted Yield" = "yield_max_issue",
                                                      "Highest Bid Yield" = "yield_max_bid",
                                                      "Weighted Average Bid" = "yield_wm_bid",
                                                      "Secondary Market Mid Rate" = "yield_mid_market",
                                                      "Number of Bids" = "bids_total",
                                                      "Number of Successful Bids" = "bids_successful",
                                                      "Number of Bids Accepted in Full" = "bids_full",
                                                      "Settlement Proceeds" = "amount_proceeds"))
  selected <- dplyr::select(mutated,
                            date_issue, date_settle, tender, maturity, coupon, isin, value, value_type)
}

#' Clean up AOFM buybacks data
#'
#' @description This function is used within aofm_data() to clean up buybacks data. There's no need to use it on its own.
#'
#' @param buybacks_data The output of read_excel() for the AOFM's spreadsheet.
#'
#' @export
#'
#' @family cleaning functions

clean_buybacks_data <- function (buybacks_data) {
  tibbled <- dplyr::as_tibble(buybacks_data)
  dropped <- tibbled[-1, ]
  set_origin <- dplyr::if_else(lubridate::is.POSIXt(dropped$`Date Held`[1]), as.character(NA), "1899-12-30") # checks date formatting and adjusts it if Excel has botched something
  gathered <- tidyr::gather(dropped,
                            "value_type", "value",
                            -`Date Held`, -`Date Settled`, -`Tender Number / Buyback Method`, -Maturity, -Coupon, -ISIN)
  mutated <- dplyr::mutate(gathered,
                           date_repurchase = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(`Date Held`),
                                                              !is.na(set_origin) ~ lubridate::as_date(as.numeric(`Date Held`), origin = set_origin),
                                                              TRUE ~ lubridate::as_date(NA)),
                           date_settle = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(`Date Settled`),
                                                          !is.na(set_origin) ~ lubridate::as_date(as.numeric(`Date Settled`), origin = set_origin),
                                                          TRUE ~ lubridate::as_date(NA)),
                           buyback = `Tender Number / Buyback Method`,
                           maturity = dplyr::case_when(is.na(set_origin) ~ lubridate::as_date(Maturity),
                                                       !is.na(set_origin) ~ lubridate::as_date(as.numeric(Maturity), origin = set_origin),
                                                       TRUE ~ lubridate::as_date(NA)),
                           coupon = as.numeric(Coupon),
                           isin = ISIN,
                           value = as.numeric(value),
                           value_type = dplyr::recode(value_type,
                                                      "Amount Repurchased" = "amount_repurchase",
                                                      "Amount of Offers" = "amount_offer",
                                                      "Weighted Average Repurchase Yield" = "yield_wm_repurchase",
                                                      "Lowest Accepted Yield" = "yield_min_repurchase",
                                                      "Highest Accepted Yield" = "yield_max_repurchase",
                                                      "Lowest Offer" = "yield_min_offer",
                                                      "Weighted Average Offer" = "yield_wm_offer",
                                                      "Secondary Market Mid Rate" = "yield_mid_market",
                                                      "Number of Offer" = "offers_total",
                                                      "Number of Successful Offers" = "offers_successful",
                                                      "Number of Offers Accepted in Full" = "offers_full",
                                                      "Settlement Proceeds" = "amount_proceeds"))
  selected <- dplyr::select(mutated,
                            date_repurchase, date_settle, buyback, maturity, coupon, isin, value, value_type)
}

#' Load data from the AOFM's Data Hub
#'
#' @description This is the workhorse function of the package. Use it to load and process data from the AOFM's website.
#'
#' @param type Either 'positions', 'issuance' or 'buybacks'. For buybacks data, retail buybacks are not included and no buybacks data are available for T-notes.
#' @param instrument Either 'tb', 'ti' or 'tn'.
#' @param positions_basis (Default = NA.) Set this to 'dealt' or 'settlement' if you are loading positions data.
#'
#' @export
#'
#' @family functions that load data

aofm_data <- function (
  type, # positions / issuance / buybacks (only wholesale; excludes retail buybacks)
  instrument, # tb / ti / tn (no buybacks for T-notes)
  positions_basis = as.character(NA) # dealt / settlement
) {
  # Give errors for bad settings ####
  if (type == "positions" & is.na(positions_basis)) {
    stop("For positions data, please specify your desired reporting basis. For example, positions_basis = 'dealt' or positions_basis = 'settlement'.")
  }
  if (type == "buybacks" & instrument == "tn") {
    stop("Oops. There are no buybacks data for T-notes. Sorry.")
  }

  # Pick the correct file for processing ####
  aofm_urls <- tibble::tribble(~types, ~instruments, ~positions_bases, ~urls,
                               "positions", "tb", "dealt", "https://www.aofm.gov.au/media/578",
                               "positions", "tb", "settlement", "https://www.aofm.gov.au/media/580",
                               "positions", "ti", "dealt", "https://www.aofm.gov.au/media/581",
                               "positions", "ti", "settlement", "https://www.aofm.gov.au/media/582",
                               "positions", "tn", "dealt", "https://www.aofm.gov.au/media/563",
                               "positions", "tn", "settlement", "https://www.aofm.gov.au/media/564",
                               "issuance", "tb", as.character(NA), "https://www.aofm.gov.au/media/591", # excludes issuance via conversions or switches
                               "issuance", "ti", as.character(NA), "https://www.aofm.gov.au/media/429",
                               "issuance", "tn", as.character(NA), "https://www.aofm.gov.au/media/430",
                               "buybacks", "tb", as.character(NA), "https://www.aofm.gov.au/media/424",
                               "buybacks", "ti", as.character(NA), "https://www.aofm.gov.au/media/433")
  aofm_url <- dplyr::filter(aofm_urls,
                            types == type &
                              instruments == instrument &
                              (positions_bases == positions_basis | is.na(positions_basis)))
  if (nrow(aofm_url) == 0) {
    stop("No dice. The only types of data available are 'positions', 'issuance' or 'buybacks'. For instrument, choose either 'tb', 'ti' or 'tn'.")
  }
  aofm_url <- unique(aofm_url$urls)
  aofm_file <- tempfile(fileext = ".xlsx")
  download.file(aofm_url,
                aofm_file,
                quiet = TRUE,
                mode = "wb")

  # Apply some special sauce ####
  skip_fv <- 3
  skip_mv <- dplyr::if_else(instrument == "ti", 4, 3)
  skip_delta <- 3
  skip_duration <- dplyr::if_else(instrument == "tn", 4, 3)
  skip_tenor <- 3
  skip_transactions <- 1
  guess_max_transactions <- 1250

  # Make the data tidy ####
  if (type == "positions") {
    positions_fv <- suppressMessages(suppressWarnings(clean_positions_data(readxl::read_excel(aofm_file, sheet = "FaceValue", skip = skip_fv), "fv")))
    positions_mv <- suppressMessages(suppressWarnings(clean_positions_data(readxl::read_excel(aofm_file, sheet = "MarketValue", skip = skip_mv), "mv")))
    positions_delta <- suppressMessages(suppressWarnings(clean_positions_data(readxl::read_excel(aofm_file, sheet = "Delta", skip = skip_delta), "delta")))
    positions_duration <- suppressMessages(suppressWarnings(clean_positions_data(readxl::read_excel(aofm_file, sheet = "Duration", skip = skip_duration), "duration")))
    positions_tenor <- suppressMessages(suppressWarnings(clean_positions_data(readxl::read_excel(aofm_file, sheet = "Tenor", skip = skip_tenor), "tenor")))
    positions <- dplyr::bind_rows(positions_fv,
                                  positions_mv,
                                  positions_delta,
                                  positions_duration,
                                  positions_tenor)
    return(positions)
  }
  if (type == "issuance") {
    issuance <- clean_issuance_data(readxl::read_excel(aofm_file, sheet = "Transactions", skip = skip_transactions, guess_max = guess_max_transactions))
    return(issuance)
  }
  if (type == "buybacks") {
    buybacks <- clean_buybacks_data(readxl::read_excel(aofm_file, sheet = "Transactions", skip = skip_transactions, guess_max = guess_max_transactions))
    return(buybacks)
  }
}
