#' Rolling Join
#'
#' @param df_master master dataframe.
#' @param df_join dataframe to join onto the `df_master`.
#' @param roll_type (default = TRUE) type of rolling join. For backward rolling
#'   set `roll_type = -Inf` (Values taken from nearest future date). For forward rolling join (values taken from nearest past date), set `roll_type = T` (default)
#'
#' @return joined dataframe
#' @import data.table
#' @export
#' @details https://www.r-bloggers.com/understanding-data-table-rolling-joins/
#'
#' @examples
#' df_master <- data.frame(date = seq.Date(from = as.Date("2020-01-01"), to = as.Date("2020-01-31"), by = "day" ), value = runif(31))
#' df_join <- data.frame(date = seq.Date(from = as.Date("2020-01-01"), to = as.Date("2020-01-31"), by = "3 days" ), output = 10*runif(11))
#' rolling_join(df_master, df_join) # roll values forward
#' rolling_join(df_master, df_join, roll_type = -Inf) #roll values backwards
rolling_join <- function(df_master, df_join, roll_type = T ) {
  # Check column 'date' exists.
  if (! "date" %in% colnames(df_master)) stop(glue::glue("Can't find `date` column in master dataframe."))
  if (! "date" %in% colnames(df_join)) stop(glue::glue("Can't find `date` column in joining dataframe."))
  
  # Convert to data table
  df_master <- data.table::data.table(df_master)
  df_join <- data.table::data.table(df_join)
  
  # Set keys
  data.table::setkey(df_master, date)
  data.table::setkey(df_join, date)
  
  # rolling join
  output <- df_join[df_master, roll = roll_type]
  
  output <- tibble::tibble(output) %>% dplyr::relocate(colnames(df_master))
  
  return(output)
  
}
