#' Wrapper function to perform information effect regressions
#'
#' @param reg_data dataframe containing regression data
#' @param y_var dependent variable in regression
#' @param asymmetry (default = F) asymmetric specification i.e. have one
#'   coefficient if independent variable is positive and another for if the
#'   independent variable is negative
#'
#' @return list containing various regression models 
#' @export

info_regressions <- function(reg_data, y_var = "revision", asymmetry = F){
  
  if (asymmetry) {
    info_regressions_asymmetry(reg_data, y_var = y_var)
  } else{
    info_regressions_linear(reg_data, y_var = y_var)
  }
  
}



#' Standard information effect regressions
#'
#' @param reg_data dataframe containing regression data
#' @param y_var dependent variable in regression
#'
#' @return list containing 3 regression outputs:
#'        * `pc1_scaled`: `y_var ~ pc1_scaled`
#'        * `cashrate_change`: `y_var ~ cashrate_change`
#'        * `pc1_scaled_move`: `y_var ~ pc1_scaled*I[cashrate_change != 0] + pc1_scaled*I[cashrate_change == 0]`
#' @export

info_regressions_linear <- function(reg_data, y_var = "revision"){
  
  # define RHS of lm equation
  x_vars <- list(pc1_scaled = "pc1_scaled",
                 cashrate_change = "cashrate_change",
                 pc1_scaled_move = "pc1_scaled:as.logical(cashrate_change)")
  
  # define entire equation
  eqns <- map2( x_vars, y_var, ~ as.formula(glue::glue("{.y} ~ {.x}"))) %>% set_names(names(x_vars))
  
  # perform regressions and set names
  reg_output <- map(eqns,  ~ lm(., data = reg_data )) %>% set_names(names(x_vars))
  return(reg_output)
  
}


#' Asymmetric information effect regressions
#'
#' @description This function will perform the information effect regresisons
#'   but interact the independent variable based on if the independent variable
#'   is positive or negative
#'
#' @param reg_data dataframe containing regression data
#' @param y_var dependent variable in regression
#'
#' @return list containing 2 regression outputs:
#'        * `pc1_scaled`: `y_var ~ pc1_scaled*I[pc1_scaled >= 0] +pc1_scaled*I[pc1_scaled < 0]`
#'        * `cashrate_change`: `y_var ~ pc1_scaled*I[cashrate_change >= 0] + pc1_scaled*I[cashrate_change < 0]`
#' @export
#'
info_regressions_asymmetry <- function(reg_data, y_var = "revision"){
  
  # define RHS of lm equation
  x_vars <- list(pc1_scaled = "pc1_scaled",
                 cashrate_change_asymmetry = "cashrate_change")
  
  # define entire equation
  eqns <- map2( x_vars, y_var, ~ as.formula(glue::glue("{.y} ~ {.x}:positive + {.x}:negative"))) %>%
    set_names(names(x_vars))
  
  # perform regressions and set names
  reg_output <- map2(x_vars, eqns,  ~ lm(.y, data = reg_data %>%
                                           mutate(negative = as.numeric(!! sym(.x) < 0) , positive = as.numeric(!! sym(.x) >= 0) ) )) %>%
    set_names(names(x_vars))
  
  return(reg_output)
  
}


