#' RBA colours
#'
#' This colour palette was made by Angus Moore for the [`arphit`](http://economic/econometrics/software/R/arphit/index.html) package
#'
#' To see a list of all the available colours, type `vignette("rbacolours", package = "arphit")`
#'
#' @export
rba <- c() #nolint

rba["red1"] <- rgb(255, 159, 174, maxColorValue = 255)
rba["red2"] <- rgb(255, 131, 139, maxColorValue = 255)
rba["red3"] <- rgb(251, 64, 90, maxColorValue = 255)
rba["red4"] <- rgb(249, 90, 121, maxColorValue = 255)
rba["red5"] <- rgb(243, 41, 56, maxColorValue = 255)
rba["red6"] <- rgb(220, 40, 30, maxColorValue = 255)
rba["red7"] <- rgb(193, 56, 49, maxColorValue = 255)
rba["red8"] <- rgb(162, 49, 56, maxColorValue = 255)
rba["red9"] <- rgb(122, 36, 39, maxColorValue = 255)

rba["orange1"] <- rgb(255, 159, 113, maxColorValue = 255)
rba["orange2"] <- rgb(255, 111, 32, maxColorValue = 255)
rba["orange3"] <- rgb(255, 90, 0, maxColorValue = 255)
rba["orange4"] <- rgb(253, 176, 114, maxColorValue = 255)
rba["orange5"] <- rgb(249, 148, 63, maxColorValue = 255)
rba["orange6"] <- rgb(215, 119, 0, maxColorValue = 255)
rba["orange7"] <- rgb(212, 170, 133, maxColorValue = 255)
rba["orange8"] <- rgb(200, 78, 0, maxColorValue = 255)
rba["orange9"] <- rgb(191, 139, 94, maxColorValue = 255)

rba["brown1"] <- rgb(165, 81, 33, maxColorValue = 255)
rba["brown2"] <- rgb(152, 98, 19, maxColorValue = 255)
rba["brown3"] <- rgb(137, 68, 30, maxColorValue = 255)
rba["brown4"] <- rgb(112, 51, 24, maxColorValue = 255)

rba["olive1"] <- rgb(255, 182, 17, maxColorValue = 255)
rba["olive2"] <- rgb(218, 170, 0, maxColorValue = 255)
rba["olive3"] <- rgb(214, 200, 52, maxColorValue = 255)
rba["olive4"] <- rgb(209, 142, 0, maxColorValue = 255)
rba["olive5"] <- rgb(199, 172, 0, maxColorValue = 255)
rba["olive6"] <- rgb(163, 119, 0, maxColorValue = 255)
rba["olive7"] <- rgb(142, 147, 0, maxColorValue = 255)
rba["olive8"] <- rgb(134, 105, 34, maxColorValue = 255)
rba["olive9"] <- rgb(115, 100, 33, maxColorValue = 255)
rba["olive10"] <- rgb(100, 88, 24, maxColorValue = 255)

rba["green1"] <- rgb(118, 209, 183, maxColorValue = 255)
rba["green2"] <- rgb(114, 174, 86, maxColorValue = 255)
rba["green3"] <- rgb(94, 193, 82, maxColorValue = 255)
rba["green4"] <- rgb(0, 195, 118, maxColorValue = 255)
rba["green5"] <- rgb(0, 182, 144, maxColorValue = 255)
rba["green6"] <- rgb(0, 174, 66, maxColorValue = 255)
rba["green7"] <- rgb(0, 153, 121, maxColorValue = 255)
rba["green8"] <- rgb(0, 153, 74, maxColorValue = 255)

rba["darkgreen1"] <- rgb(90, 142, 35, maxColorValue = 255)
rba["darkgreen2"] <- rgb(64, 118, 49, maxColorValue = 255)
rba["darkgreen3"] <- rgb(38, 78, 55, maxColorValue = 255)
rba["darkgreen4"] <- rgb(0, 121, 79, maxColorValue = 255)
rba["darkgreen5"] <- rgb(0, 120, 54, maxColorValue = 255)
rba["darkgreen6"] <- rgb(0, 104, 61, maxColorValue = 255)
rba["darkgreen7"] <- rgb(0, 102, 83, maxColorValue = 255)
rba["darkgreen8"] <- rgb(0, 87, 61, maxColorValue = 255)

rba["aqua1"] <- rgb(84, 132, 118, maxColorValue = 255)
rba["aqua2"] <- rgb(65, 218, 197, maxColorValue = 255)
rba["aqua3"] <- rgb(41, 149, 181, maxColorValue = 255)
rba["aqua4"] <- rgb(25, 205, 215, maxColorValue = 255)
rba["aqua5"] <- rgb(0, 177, 170, maxColorValue = 255)
rba["aqua6"] <- rgb(0, 137, 151, maxColorValue = 255)
rba["aqua7"] <- rgb(0, 101, 122, maxColorValue = 255)
rba["aqua8"] <- rgb(0, 65, 81, maxColorValue = 255)

rba["blue1"] <- rgb(100, 158, 201, maxColorValue = 255)
rba["blue2"] <- rgb(91, 181, 228, maxColorValue = 255)
rba["blue3"] <- rgb(83, 130, 172, maxColorValue = 255)
rba["blue4"] <- rgb(72, 145, 220, maxColorValue = 255)
rba["blue5"] <- rgb(18, 86, 136, maxColorValue = 255)
rba["blue6"] <- rgb(0, 159, 223, maxColorValue = 255)
rba["blue7"] <- rgb(0, 115, 207, maxColorValue = 255)
rba["blue8"] <- rgb(0, 62, 115, maxColorValue = 255)
rba["blue9"] <- rgb(0, 40, 88, maxColorValue = 255)
rba["blue10"] <- rgb(0, 28, 167, maxColorValue = 255)

rba["violet1"] <- rgb(181, 143, 187, maxColorValue = 255)
rba["violet2"] <- rgb(166, 125, 201, maxColorValue = 255)
rba["violet3"] <- rgb(154, 96, 194, maxColorValue = 255)
rba["violet4"] <- rgb(147, 120, 208, maxColorValue = 255)
rba["violet5"] <- rgb(132, 76, 176, maxColorValue = 255)
rba["violet6"] <- rgb(128, 119, 192, maxColorValue = 255)
rba["violet7"] <- rgb(125, 93, 198, maxColorValue = 255)
rba["violet8"] <- rgb(98, 91, 196, maxColorValue = 255)

rba["purple1"] <- rgb(137, 56, 185, maxColorValue = 255)
rba["purple2"] <- rgb(117, 47, 165, maxColorValue = 255)
rba["purple3"] <- rgb(107, 58, 120, maxColorValue = 255)
rba["purple4"] <- rgb(98, 77, 124, maxColorValue = 255)
rba["purple5"] <- rgb(80, 45, 126, maxColorValue = 255)
rba["purple6"] <- rgb(73, 20, 159, maxColorValue = 255)
rba["purple7"] <- rgb(51, 29, 84, maxColorValue = 255)
rba["purple8"] <- rgb(26, 16, 81, maxColorValue = 255)

rba["maroon1"] <- rgb(188, 52, 186, maxColorValue = 255)
rba["maroon2"] <- rgb(173, 63, 105, maxColorValue = 255)
rba["maroon3"] <- rgb(172, 0, 64, maxColorValue = 255)
rba["maroon4"] <- rgb(145, 64, 109, maxColorValue = 255)
rba["maroon5"] <- rgb(140, 34, 69, maxColorValue = 255)
rba["maroon6"] <- rgb(135, 1, 80, maxColorValue = 255)
rba["maroon7"] <- rgb(109, 26, 65, maxColorValue = 255)
rba["maroon8"] <- rgb(93, 36, 90, maxColorValue = 255)

rba["pink1"] <- rgb(242, 140, 210, maxColorValue = 255)
rba["pink2"] <- rgb(251, 124, 179, maxColorValue = 255)
rba["pink3"] <- rgb(183, 116, 154, maxColorValue = 255)
rba["pink4"] <- rgb(206, 102, 205, maxColorValue = 255)
rba["pink5"] <- rgb(202, 94, 155, maxColorValue = 255)
rba["pink6"] <- rgb(223, 62, 125, maxColorValue = 255)
rba["pink7"] <- rgb(222, 29, 132, maxColorValue = 255)
rba["pink8"] <- rgb(209, 0, 93, maxColorValue = 255)

rba["grey1"] <- rgb(255, 255, 255, maxColorValue = 255)
rba["grey2"] <- rgb(204, 204, 204, maxColorValue = 255)
rba["grey3"] <- rgb(179, 179, 179, maxColorValue = 255)
rba["grey4"] <- rgb(153, 153, 153, maxColorValue = 255)
rba["grey5"] <- rgb(128, 128, 128, maxColorValue = 255)
rba["grey6"] <- rgb(102, 102, 102, maxColorValue = 255)
rba["grey7"] <- rgb(77, 77, 77, maxColorValue = 255)
rba["grey8"] <- rgb(51, 51, 51, maxColorValue = 255)
rba["grey9"] <- rgb(26, 26, 26, maxColorValue = 255)
rba["grey10"] <- rgb(0, 0, 0, maxColorValue = 255)

# Add aliases for default colours
rba["default1"] <- rba["aqua8"]
rba["default2"] <- rba["orange2"]
rba["default3"] <- rba["darkgreen7"]
rba["default4"] <- rba["violet1"]
rba["default5"] <- rba["blue7"]
rba["default6"] <- rba["red5"]
rba["default7"] <- rba["brown4"]
rba["default8"] <- rba["pink2"]
rba["default9"] <- rba["grey7"]

# Alias' for country/state default colours

rba["australia"] <- rba["blue10"]
rba["us"] <- rba["blue6"]
rba["china"] <- rba["red5"]
rba["euro"] <- rba["olive1"]
rba["japan"] <- rba["pink8"]
rba["uk"] <- rba["darkgreen5"]
rba["southkorea"] <- rba["aqua8"]
rba["restofworld"] <- rba["grey7"]

rba["nsw"] <- rba["green6"]
rba["vic"] <- rba["blue10"]
rba["qld"] <- rba["olive7"]
rba["wa"] <- rba["orange3"]
rba["sa"] <- rba["purple1"]
rba["act"] <- rba["pink7"]
rba["nt"] <- rba["blue6"]
rba["tas"] <- rba["brown4"]
rba["austotal"] <- rba["grey10"]
rba["regional"] <- rba["red5"]
rba["otherregion"] <- rba["olive1"]

rba["sydney"] <- rba["nsw"]
rba["melbourne"] <- rba["vic"]
rba["brisbane"] <- rba["qld"]
rba["perth"] <- rba["wa"]
rba["adelaide"] <- rba["sa"]
rba["canberra"] <- rba["act"]
rba["darwin"] <- rba["nt"]
rba["hobart"] <- rba["tas"]

# A lot of the code below is taken from this post: https://drsimonj.svbtle.com/creating-corporate-colour-palettes-for-ggplot2


#' Function to extract rba colors as hex codes
#'
#' @param ... Character names of `rba` colours
#'
#' @examples
#' rba_colours()
#' rba_colours("red5")
#' rba_colours("orange1", "aqua8")

rba_colours <- function(...) {
  cols <- c(...)

  if (is.null(cols))
    return (rba)

  rba[cols]
}



#' RBA colour palettes
#'
#'
#' @export
rba_palettes <- list(
  `default`  = rba_colours(paste0("default", 1:9)),

  `classic` = rba_colours("orange2", "blue7",  "pink8", "grey8",  "violet2", "olive4",  "darkgreen5", "brown4", "grey5"   ),

  `classic2` = rba_colours("red5", "blue10","grey5", "purple6", "olive5",   "grey8",     "brown4",  "darkgreen5"),

  `classic3` = rba_colours("green4", "blue10", "olive1", "violet1", "grey8",  "aqua5", "brown4", "grey5"  ),

  `classic4` = c("#009473","purple4","dodgerblue3","darkgoldenrod1","#E2583E","#8ECA86","#A67DC9","darkgrey","deeppink3","darkorchid1","#45B5AA"),   # random palette suggested by Michelle Bergmann

  `cooper` =  rba_colours("blue4", "olive1", "green2", "violet3", "orange2", "aqua5"),

  `cooper2` = rba_colours("blue2", "olive3", "green3", "violet1", "orange1", "aqua2"),

  `aus_city` = rba_colours("sydney", "melbourne", "brisbane", "perth", "adelaide", "canberra", "darwin", "hobart", "regional", "otherregion", "austotal"),

  `aus_state` = rba_colours("nsw", "vic", "qld", "wa", "sa", "act", "nt", "tas", "regional", "otherregion", "austotal"),

  `country` = rba_colours("australia", "us", "china", "euro", "japan", "uk", "southkorea", "restofworld"),

  `oranges` = rba_colours(paste0("orange", 1:9)),

  `blues`  = rba_colours(paste0("blue", 1:10))
)



#' RBA palette to be used as a function of n
#' @description output parsed through `discrete_scale` to for sca
#'
#' @param palette Character name of palette in `rba_palettes`, default = "default".
#' @param reverse Logical indicating whether the palette should be reversed; default = FALSE
#'
#' @return rba colour palette as a function of n

rba_pal <- function( palette = "default", reverse = FALSE){

  # Check palette name is valid
  if (! palette %in% names(rba_palettes)) {
    stop(glue::glue("'{palette}' is not a valid rba palette. See `names(rba_palettes)` for valid inputs."))
  }


  pal <- unname(rba_palettes[[palette]]) # extract entire palette

  function(n){

    if (n > length(pal)){ # check palette length
      warning("Number of colours exceeds number of colours in palette. Returning the entire palette.")
      return(pal)
    }

    pal <-  unname(rba_palettes[[palette]])[seq_len(n)] # extract truncated palette

    if (reverse) rev(pal) # reverse

    return(pal) # return palette
  }

}

#' Continuous rba palettes using colorRampPalette
#'
#' @param palette Character name of palette in `rba_palettes``
#' @param reverse Logical indicating whether the palette should be reversed; default = FALSE
#' @param ... Additional arguments passed to colorRampPalette()

rba_pal_continuous <- function(palette = "default", reverse = F, ...){
  pal <- rba_palettes[[palette]]

  if (reverse) pal <- rev(pal)

  colorRampPalette(pal, ...)

}

#' Colour scale constructor for rba colours
#'
#' @param palette Character name of palette in `rba_palettes``
#' @param discrete Logical indicating whether colour aesthetic is discrete or not; default = TRUE
#' @param reverse Logical indicating whether the palette should be reversed; default = FALSE
#' @param ... Additional arguments passed to discrete_scale() or
#'            scale_fill_gradientn(), used respectively when discrete is TRUE or FALSE
#'
#' @details A lot of this code is taken from this [post](https://drsimonj.svbtle.com/creating-corporate-colour-palettes-for-ggplot2)
#' @export
#'
#' @examples
#' ggplot(mtcars, aes(x = cyl, y = mpg, colour = factor(am))) + geom_point() +  scale_colour_rba()
#' ggplot(mtcars, aes(x = cyl, y = mpg, colour = wt )) + geom_point() +  scale_colour_rba(palette = "classic", discrete = F)
scale_colour_rba <- function(palette = "default",  reverse = FALSE, discrete = TRUE, ...) {


  rlang::arg_match(palette, names(rba_palettes))

  if (discrete) {
    pal <- rba_pal(palette = palette, reverse = reverse)
    discrete_scale("colour", paste0("rba_", palette), palette = pal, ...)

  } else {
    pal <- rba_pal_continuous(palette = palette, reverse = reverse)
    scale_color_gradientn(colours = pal(256), ...)
  }
}



#' Fill scale constructor for rba colours
#'
#' @param palette Character name of palette in `rba_palettes``
#' @param discrete Logical indicating whether colour aesthetic is discrete or not; default = TRUE
#' @param reverse Logical indicating whether the palette should be reversed; default = FALSE
#' @param ... Additional arguments passed to discrete_scale() or
#'            scale_fill_gradientn(), used respectively when discrete is TRUE or FALSE
#'
#' @details A lot of this code is taken from this [post](https://drsimonj.svbtle.com/creating-corporate-colour-palettes-for-ggplot2)
#' @export
#' @examples
#' ggplot(mtcars, aes(x = gear, y = mpg, fill = factor(cyl) )) + geom_col() +  scale_fill_rba()
#' ggplot(mtcars, aes(x = gear, y = mpg, fill = factor(cyl) )) + geom_col() +  scale_fill_rba(reverse = T, palette = "classic2")
scale_fill_rba <- function(palette = "default", reverse = FALSE, discrete = TRUE,  ...) {

  rlang::arg_match(palette, names(rba_palettes))

  if (discrete) {
    pal <- rba_pal(palette = palette, reverse = reverse) # obtain palette (as function of n)
    discrete_scale("fill", paste0("rba_", palette), palette = pal , ...)
  } else {
    pal <- rba_pal_continuous(palette = palette, reverse = reverse)
    scale_fill_gradientn(colours = pal(256), ...)
  }
}


#' View RBA palettes
#'
#' @param palette rba_palette name to view as a string
#'
#' @return ggplot of colour palette
#' @import ggplot2
#' @export
#'
#' @examples
#' rba_palettes_view("default")
#' rba_palettes_view("classic")
#' rba_palettes_view("aus_city")
#' rba_palettes_view("aus_state")
#' rba_palettes_view("country")

rba_palettes_view <- function(palette){


    graph_data <- tibble::tibble(x = factor(names(rba_palettes[[palette]]), levels = names(rba_palettes[[palette]])),
                                 y = 1)

    ggplot(graph_data) +
      geom_col(aes(x = x, y = y, fill = x)) +
      scale_fill_manual(values = rba_palettes[[palette]]) +
      rba_theme()+
      labs(title = glue::glue("Palette: {palette}")) +
      rba_rotate_x_text() +
      rba_ylim(0, 1)+
      rba_syc() +
      ylab("")

}
