####################################################################################################
# Misc R functions used by other programs
# Luke Hartigan
####################################################################################################
# Scale a series relative to a backward rolling window 'roll_len'
####################################################################################################

rolling_scale <- function(x, roll_len = 40L, center = TRUE, scale = TRUE)
{
    # Preliminary stuff
    if (!is.matrix(x)) { x <- as.matrix(x) }
    nobs <- dim(x)[1L]
    nvar <- dim(x)[2L]
    roll_len_p1 <- roll_len + 1L
    roll_len_m1 <- roll_len - 1L

    # Check if colnames is NULL and if so set to "vi" for i = 1,...,nvar
    if (is.null(colnames(x))) {
        var_names <- paste0('v', seq_len(nvar))
    } else {
        var_names <- colnames(x)
    }

    if (roll_len > nobs) {
        stop(sprintf("Rolling window length \'%d\' greater than time series length \'%d\'\n",
            roll_len, nobs))
    }

    if (nvar > 1L) {

        # Allocate storage for the scaled series
        sx <- matrix(data = NA_real_, nrow = nobs, ncol = nvar)

        # Scale the initial part of the sample
        sx[seq_len(roll_len), ] <- scale(x = x[seq_len(roll_len), ], center = center, scale = scale)

        for (i in roll_len_p1:nobs) {
            sx[(i - roll_len_m1):i, ] <- scale(x = x[(i - roll_len_m1):i, ], center = center, scale = scale)
        }

    } else {

        # Allocate storage for the scaled series
        sx <- matrix(data = NA_real_, nrow = nobs, ncol = 1L)

        # Scale the initial part of the sample
        sx[seq_len(roll_len)] <- scale(x = x[seq_len(roll_len)], center = center, scale = scale)

        for (i in roll_len_p1:nobs) {
            sx[(i - roll_len_m1):i] <- scale(x = x[(i - roll_len_m1):i], center = center, scale = scale)
        }

    }

    # Assign colnames before returning sx
    colnames(sx) <- var_names

    return (sx)
}

####################################################################################################
# foo <- skewness(x, bias_fix)
#
# PURPOSE:
#   Function to calculate the skewness (3rd moment) of a variable.
#
# INPUTS:
#   x = data series to use
#   bias_fix options:
#       bias_fix == TRUE to correct for bias. (Default)
#       bias_fix == FALSE to not correct for bias.
#
# OUTPUTS:
#   skewness = estimate of the third moment of the data series x.
#
# DEPENDENCIES:
#   None.
#
# REFERENCES:
#   None.
#
# AUTHOR:
#   Luke Hartigan (2014)
####################################################################################################

skewness <- function(x, bias_fix = TRUE)
{
    # Preliminary stuff
    x <- as.matrix(x)
    n <- length(x)
    x0 <- x - mean(x, na.rm = TRUE)
    s2 <- mean(x0^2, na.rm = TRUE) # NB biased variance estimator

    # Bias corrected skewness
    if (bias_fix) {
        m3 <- mean(x0^3, na.rm = TRUE)
        s1 <- m3 / s2^1.5
        skewness <- s1 * sqrt((n - 1) / n) * n / (n - 2)
    } else {
        m3 <- mean(x0^3, na.rm = TRUE)
        skewness <- m3 / s2^1.5
    }

    # Return skewness
    return (skewness)

}

####################################################################################################
# foo <- kurtosis(x, bias_fx)
#
# PURPOSE:
#   Function to calculate the kurtosis (4th moment) of a variable.
#
# INPUTS:
#   x = data series to use
#   bias_fix options:
#       bias_fix == TRUE to correct for bias. (Default)
#       bias_fix == FALSE to not correct for bias.
#
# OUTPUTS:
#   kurtosis = estimate of the fourth moment of the data series x.
#
# DEPENDENCIES:
#   None.
#
# REFERENCES:
#   None.
#
# AUTHOR:
#   Luke Hartigan (2014)
####################################################################################################

kurtosis <- function(x, bias_fix = TRUE)
{
    # Preliminary stuff
    x <- as.matrix(x)
    n <- length(x)
    x0 <- x - mean(x, na.rm = TRUE)
    s2 <- mean(x0^2, na.rm = TRUE) # NB biased variance estimator

    # Bias corrected Kurtosis
    if (bias_fix) {
        m4 <- mean(x0^4, na.rm = TRUE)
        k1 <- m4 / s2^2
        kurtosis <- ((n + 1) * k1 - 3 * (n - 1)) * (n - 1) / ((n - 2) * (n - 3)) + 3
    } else {
        m4 <- mean(x0^4, na.rm = TRUE)
        kurtosis <- m4 / s2^2
    }

    # Return kurtosis
    return (kurtosis)

}

####################################################################################################
# foo <- jb_test(x)
#
# PURPOSE:
#   Function to calculate the Jarque-Bera test of normality. Under the null the data should have
#   skewness = 0 (symmetric) and have kurtosis = 3. The Jarque-Bera statistic is chi-square
#   distributed with 2 degrees of freedom.
#
# INPUTS:
#   x = data series to use
#
# OUTPUTS (List object with class "jb_test")
#   statistic = critical value of the Jarque-Bera test ~ chi-square(2)
#   pvalue = p value for the JB critical value.
#
# DEPENDENCIES:
#   None.
#
# REFERENCES:
#   Jarque, C. M., Bera, A. K. (1980) Efficient test for normality, homoscedasticity and
#       serial independence of residuals, Economic Letters, Vol. 6 Issue 3, 255-259.
#
# AUTHOR:
#   Luke Hartigan (2014)
####################################################################################################

jb_test <- function(x)
{
    # Preliminary settings
    x <- as.matrix(x)
    n <- length(x)
    x0 <- x - mean(x, na.rm = TRUE)
    s2 <- mean(x0^2, na.rm = TRUE) # NB biased variance estimator

    # Calculate sample skewness and excess kurtosis
    skew <- mean(x0^3, na.rm = TRUE) / s2^1.5
    kurt <- mean(x0^4, na.rm = TRUE) / s2^2 - 3

    # Calculate the test statistic
    statistic <- n * (skew * (skew / 6) + kurt * (kurt / 24))

    # Test results
    pvalue <- pchisq(q = statistic, df = 2, lower.tail = FALSE)

    # Collect results as a list object
    results <- list()
    results$statistic <- statistic
    results$pvalue <- pvalue

    # Return results with class "jb"
    results$call = match.call()
    class(results) <- "jb_test"
    return (results)

}

# Print method for "jb_test"
print.jb_test <- function(x, digits = max(3L, getOption("digits") - 3L), ...)
{
    cat("\nCall: ", paste(deparse(x$call), sep = '\n', collapse = '\n'), '\n', sep = "")
    cat("\nJarque & Bera (1980) test for normality:\n\n")
    cat("H0 : The process is \'normal\'\n\n")
    cat("Statistic  P-value\n")
    cat(sprintf("%9.4g  %7.4g\n", x$statistic, x$pvalue))
    cat('\n')
}

####################################################################################################
# foo <- transform_series(x, take_log, tcode, pcode)
#
# PURPOSE:
#   Function returns suitably transformed, lagged and/or differenced series.
#
# INPUTS:
# take_log = take the natural logarithm? Default is no (FALSE)
#
# tcode = option to specify how y is transformed:
#   tcode == "t1" [No difference, i.e., Level] -- Default --
#   tcode == "t2" [1st Difference, i.e., (1 - B)y]
#   tcode == "t3" [4th Difference, i.e., (1 - B^4)y, use with quarterly data]
#   tcode == "t4" [12th Difference, i.e., (1 - B^12)y, use with monthly data]
#   tcode == "t5" [2nd Difference, i.e., (1 - B)^2y, double difference]
#   tcode == "t6" [1st diff & seasonal diff for quaterly data, i.e., ((1 - B)(1 - B^4)y]
#   tcode == "t7" [1st diff & seasonal diff for monthly data, i.e., ((1 - B)(1 - B^12)y]
#
# pcode = option to specify if percentages are computed:
#   pcode == "p1" [no change] -- Default --
#   pcode == "p2" [multiply by 100]
#   pcode == "p3" [multiply by 400, annualised quarterly rate]
#   pcode == "p4" [multiply by 1200, annualised monthly rate]
#
# OUTPUTS:
#   y = transformed numeric vector x which is short than x if differencing is undertaken.
#
# DEPENDENCIES:
#	None.
#
# REFERENCES:
#   Adapted from the GAUSS procs of Stock and Watson(2005), 'Implications of Dynamic Factor
#     Models for VAR analysis', manuscript. link: http://www.princeton.edu/~mwatson/wp.html
#
# AUTHOR:
#   Luke Hartigan (2017)
####################################################################################################

transform_series <- function(x, take_log = FALSE,
    tcode = c("t1", "t2", "t3", "t4", "t5", "t6", "t7"), pcode = c("p1", "p2", "p3", "p4"))
{
    # Save 'ts' attributes if 'y' is a 'ts' object
    if (is.ts(x)) {
        x_ts <- TRUE
        endx <- end(x)
        freqx <- frequency(x)
    } else {
        x_ts <- FALSE
    }

    # Preliminary stuff
    tx <- as.matrix(x)

    # Log transformation
    if (take_log) {
        if (any(tx < 0, na.rm = TRUE)) {
            tx[which(tx < 0)] <- NaN
        }
        tx <- log(as.matrix(tx))
    }

    # Difference transform if requested
    tcode <- match.arg(tcode)

    switch (EXPR = tcode,
        "t1" = { tx }, # do nothing
        "t2" = { tx <- diff(x = tx, lag = 1L) },
        "t3" = { tx <- diff(x = tx, lag = 4L) },
        "t4" = { tx <- diff(x = tx, lag = 12L) },
        "t5" = { tx <- diff(diff(x = tx, lag = 1L), lag = 1L) },
        "t6" = { tx <- diff(diff(x = tx, lag = 1L), lag = 4L) },
        "t7" = { tx <- diff(diff(x = tx, lag = 1L), lag = 12L) }
    )

    # Convert to percentage if requested
    pcode <- match.arg(pcode)

    switch (EXPR = pcode,
        "p1" = { tx }, # do nothing
        "p2" = { tx <- tx * 100.0 },
        "p3" = { tx <- tx * 400.0 },
        "p4" = { tx <- tx * 1200.0 }
    )

    # Add 'ts' attribute to 'tx' if is a 'ts' object
    if (x_ts) {
        tx <- ts(tx, end = endx, frequency = freqx)
    }

    return (tx)
}

####################################################################################################
# foo <- trim_row(x, a, b)
#
# PURPOSE:
#   Function to remove specified rows from a matrix (or vector).
#
# INPUTS:
#   x = matrix or (vector) on dimension N x T
#   a = scalar value to trim nrow(x) from top of the matrix (or vector)
#   b = scale value to trim nrow(x) from bottom of the matrix (or vector)
#
# OUTPUTS:
#   x_adj = x[(a+1):(T-b), ]
#
# DEPENDENCIES:
#   None.
#
# REFERENCES:
#   Translation of GAUSS 'trimr' function.
#
# AUTHOR:
#   Luke Hartigan (2014)
####################################################################################################

trim_row <- function(x, a, b)
{
    # Preliminary stuff
    x <- as.matrix(x)
    mdim <- dim(x)[1L]
    ndim <- dim(x)[2L]

    # Do some checks on the dimensions of 'x' matrix
    if ((a < 0) || (b < 0)) {
        stop("trim_row(): Neither 'a' nor 'b' can be negative!!\n", call. = FALSE)
    }

    if ((a > mdim ) || (b > mdim)) {
        stop(sprintf("trim_row(): Length of 'a' or 'b' cannot exceed %d.\n", nrow(x)), call. = FALSE)
    }

    if ((a + b >= mdim)) {
        stop(sprintf("trim_row(): Length of 'a + b' cannot exceed %d.\n", nrow(x)), call. = FALSE)
    }

    x_adj <- x[(a + 1L):(mdim - b), , drop = FALSE]

    return (x_adj)
}

####################################################################################################
# R emulation of MATLAB zeros()
####################################################################################################

zeros <- function(...)
{
    return (array(data = 0, dim = c(...)))
}

# EOF
