####################################################################################################
# 'Growth-at-Risk' Helper Functions
# Luke Hartigan, 2020
####################################################################################################

# Add a legend at the bottom of the plot window
add_legend <- function(...)
{
    opar <- graphics::par(fig = c(0, 1, 0, 1), oma = c(0, 0, 0, 0), mar = c(0, 0, 0, 0), new = TRUE)
    on.exit(graphics::par(opar))
    graphics::plot(x = 0, y = 0, type = 'n', bty = 'n', xaxt = 'n', yaxt = 'n')
    graphics::legend(...)
}

# Non-parametric estimate of the variance of the CCF when
# not assuming iid observations -- see Eqn 11.4.1 in Cryer and Chan (2008)
ccf_var <- function(x, y, n)
{
    # Check x and y have the same number of elements
    if (length(x) != length(y)) {
        stop("np_ccf_var(): x and y have different lengths\n", call. = FALSE)
    }

    # Sum acfs from lag k = 1,...,K
    vn <- (1.0 + 2.0 * sum(x[-1L] * y[-1L])) / n

    return (vn)

}

# Get the year and quarter as numerical values
get_year_quarter <- function(x)
{
    date_str <- as.Date(x, format = "%Y-%m-%d")
    year <- as.numeric(substr(x = date_str, start = 1L, stop = 4L))
    quarter <- as.numeric(substr(x = date_str, start = 6L, stop = 7L)) / 3L
    return (c(year, quarter))
}


# Compute lag matrix with dimensions (T - p, N x p)
lag_mat <- function(x, nlags)
{
    # Preliminary stuff
    x <- as.matrix(x)
    nobs <- dim(x)[1L]
    nvar <- dim(x)[2L]

    # Storage for the matrix of lagged variables
    xlags <- matrix(data = NA_real_, nrow = nobs - nlags, ncol = nvar * nlags)

    # Construct the matrix of lagged variables
    for (i in seq_len(nlags)) {
        xlags[, (nvar*i - nvar + 1L):(nvar*i)] <- x[((nlags + 1L) - i):(nobs - i), ]
    }

    # Return the matrix of lagged variables
    return (xlags)
}

# Compute the square of variable/elements of 'x'
square <- function(x)
{
    return (x * x)
}

# Granger Causality Test (based on the Chi-sq test in Hamilton (1994), eqn. 11.2.10, p. 305)
# Tests whether series 'y/x' Granger-causes series 'x/y'
# Note: Series 'a' is said to be Granger-caused by 'b' if 'b' helps in the prediction of 'a'.
granger_cause <- function(x, y, plag, intercept = TRUE)
{
    # Preliminary stuff
    if(!is.matrix(x)) { x <- as.matrix(x) }
    if(!is.matrix(y)) { y <- as.matrix(y) }

    # Check x and y have same dimension
    if (dim(y)[1L] != dim(y)[1L]) {
        stop("granger_cause(): Series \'x\' and \'y\' have different length.\n", call. = FALSE)
    }

    # Adjust x and y for plags
    xp <- x[-seq_len(plag), , drop = FALSE]
    yp <- y[-seq_len(plag), , drop = FALSE]
    nobs <- dim(xp)[1L]

    # Compute matrices of lagged values
    xlags <- lag_mat(x = x, nlag = plag)
    ylags <- lag_mat(x = y, nlag = plag)
    xylags <- cbind(xlags, ylags)

    # Include an intercept if requested
    if (intercept) {
        xlags <- cbind(1.0, xlags)
        ylags <- cbind(1.0, ylags)
        xylags <- cbind(1.0, xylags)
    }

    # Estimate the unrestricted sum of squared residuals for each series
    rss1x <- sum(square(x = qr.resid(qr = qr(x = xylags), y = xp)))
    rss1y <- sum(square(x = qr.resid(qr = qr(x = xylags), y = yp)))

    # Estimate the restricted sum of squared residuals for each series
    rss0x <- sum(square(x = qr.resid(qr = qr(x = xlags), y = xp)))
    rss0y <- sum(square(x = qr.resid(qr = qr(x = ylags), y = yp)))

    # Compute the Chi-squared[plag] test statistic and p-value for each series
    Sx <- (nobs * (rss0x - rss1x)) / rss1x
    Sy <- (nobs * (rss0y - rss1y)) / rss1y
    pvx <- pchisq(q = Sx, df = plag, lower.tail = FALSE)
    pvy <- pchisq(q =  Sy, df = plag, lower.tail = FALSE)
    statistic <- c(Sx, Sy)
    pvalue <- c(pvx, pvy)

    # Collect results as a list object
    results <- list()
    results$statistic <- statistic
    results$pvalue <- pvalue
    results$lags <- plag

    # Return object with class "granger_cause"
    results$call <- match.call()
    class(results) <- "granger_cause"
    return (results)

}

# Print method for "granger_cause"
print.granger_cause <- function(x, digits = 4L, print.gap = 3L, right = TRUE, ...)
{
    cat("\nCall: ", paste(deparse(x$call), sep = '\n', collapse = '\n'), '\n', sep = "")
    cat(sprintf("\nPairwise Granger Causality Tests (%d lags):\n", x$lags))
    results <- round(cbind(x$statistic, x$pvalue), 4L)
    colnames(results) <- c("Statistic", "P-value")
    rownames(results) <- c("Series \'y\' does not Granger cause series \'x\'",
                           "Series \'x\' does not Granger cause series \'y\'")
    print.default(results, digits = digits, print.gap = print.gap, right = right, ...)
    cat('\n')
}

# Compute the percentage point growth rate
ppt_growth <- function(x, nlag)
{
    if(!is.matrix(x)) { x <- as.matrix(x) }
    nobs <- dim(x)[1L]
    ppt <- (x[(1L + nlag):nobs, ] / x[1L:(nobs - nlag), ]) * 100.0 - 100.0
    return (ppt)
}

# Make a growth rate series
make_growth_rate <- function(x, nlag)
{
    xn <- na.omit(x)
    dxn <- ppt_growth(x = xn, nlag = nlag)
    return (dxn)
}

# Prewhiten function taken (and formatted) from the 'TSA' package:
# https://cran.r-project.org/package=TSA
prewhiten <- function(x, y, x.model = ar.res, ylab = "CCF", ...)
{
    filter.mod <- function(x, model) {
        if (length(model$Delta) >= 1L) {
            x <- stats::filter(x = x, filter = c(1.0, -model$Delta),
                               method = "convolution", sides = 1L)
        }
        if (length(model$theta) >= 1L && any(model$theta != 0.0)) {
            x <- stats::filter(x = x, filter = -model$theta,
                               method = "recursive", sides = 1L)
        }
        if (length(model$phi) >= 1L && any(model$phi != 0.0)) {
            x <- stats::filter(x = x, filter = c(1.0, -model$phi),
                               method = "convolution", sides = 1L)
        }
        return (x)
    }

    if(!missing(x.model)) {
        x <- filter.mod(x = x, model = x.model$model)
        y <- filter.mod(x = y, model = x.model$model)
    } else {
        ar.res <- ar.ols(x, ...)
        x <- stats::filter(x = x, filter = c(1.0, -ar.res$ar),
                           method = "convolution", sides = 1L)
        y <- stats::filter(x = y, filter = c(1.0, -ar.res$ar),
                           method = "convolution", sides = 1L)
    }

    ccf.xy <- ccf(x = x, y = y, na.action = na.omit, ylab = ylab, ...)
    invisible(list(ccf = ccf.xy, model = x.model))
}

# Matrix of colours to pass to persp()
# Source: https://stat.ethz.ch/pipermail/r-help/2003-September/039104.html
surf_colours <- function(x, col = topo.colors(20)) {

    # First we drop the 'borders' and average the facet corners
    # we need (nx - 1) x (ny - 1) facet colours
    nr <- dim(x)[1L]
    nc <- dim(x)[2L]
    xavg <- (x[-1L, -1L] + x[-1L, -(nc - 1L)] +
        x[-(nr - 1L), -1L] + x[-(nr - 1L), -(nc - 1)]) / 4.0

    # Now we construct the actual colours matrix
    mcolors <- col[cut(xavg, breaks = length(col), include.lowest = TRUE)]
    return (mcolors)
}

####################################################################################################
# Skewed T Distribution Functions
####################################################################################################

# Clamp a value to be between min_x and max_x
clamp <- function(x, min_x, max_x)
{
    return (pmax(pmin(max_x, x), min_x))
}

# Truncated Gaussian Density function
trunc_dnorm <- function(x, mu, sig, a, b)
{
    # Do some checks
    if (a > b) {
        tmp <- a
        a <- b
        b <- tmp
    }
    # Define some useful values
    Xi <- (x - mu) / sig
    Alpha <- (a - mu) / sig
    Beta <- (b - mu) / sig
    Z <- stats::pnorm(q = Beta) - stats::pnorm(q = Alpha)
    f <- stats::dnorm(x = Xi) / (sig * Z)
    return (f)
}

# Truncated Gaussian Distribution function
trunc_pnorm <- function(q, mu, sig, a, b)
{
    # Do some checks
    if (a > b) {
        tmp <- a
        a <- b
        b <- tmp
    }
    # Define some useful values
    Xi <- (q - mu) / sig
    Alpha <- (a - mu) / sig
    Beta <- (b - mu) / sig
    num <- stats::pnorm(q = Xi) - stats::pnorm(q = Alpha)
    den <- stats::pnorm(q = Beta) - stats::pnorm(q = Alpha)
    return (num / den)
}

# Truncated Gaussian Inverse Distribution function
trunc_qnorm <- function(p, mu, sig, a, b)
{
    # Do some checks
    if (a > b) {
        tmp <- a
        a <- b
        b <- tmp
    }
    # Define some useful values
    Alpha <- (a - mu) / sig
    Beta <- (b - mu) / sig
    Z <- stats::pnorm(q = Beta) - stats::pnorm(q = Alpha)
    f <- stats::qnorm(p = (stats::pnorm(q = Alpha) + (p * Z)))
    return (f)
}

# Skewed T parameter restrictions
# sigma > 0; -1 << lambda << 1; qn >= 1
st_param_trans <- function(x)
{
    mu <- x[1L]
    sig <- square(x[2L])
    lam <- trunc_qnorm(p = stats::pnorm(q = x[3L]), mu = 0.0, sig = 1.0,
                                        a = -0.5, b = 0.5)
    qn <- ifelse(test = (x[4L] < 1.0),
                 yes = 1.0,                     # df restricted so variance will exist...
                 no = 1.0 + log(abs(x[4L])))    # ...and is positive
    return (c(mu, sig, lam, qn))
}

# Fit Skewed T inverse CDF function given a set of probabilities
fit_st_inv_cdf <- function(params, x, probs, mean_opt, var_opt)
{
    # Transform the parameters to unrestricted parameter space
    params <- st_param_trans(params)

    # Compute the actual quantile values
    actual <- sgt::qsgt(prob = probs, mu = params[1L], sigma = params[2L],
                        lambda = params[3L], p = 2.0, q = params[4L],
                        mean.cent = mean_opt, var.adj = var_opt,
                        lower.tail = TRUE, log.p = FALSE)

    # Compute the sum of squared errors
    sse <- sum(square(x - actual))
    return (sse)
}

# Estimate a sequence of Skewed T Distributions densities based on fitted quantiles
seq_st_density <- function(rqfit, tau, init_params, min_x, max_x,
                           length.out = 500L, mean_opt = FALSE, var_opt = FALSE)
{
    # Do some checks
    if (dim(rqfit)[2L] != length(tau)) {
        stop("seq_st_pdf(): dimension of fitted rq different to length of tau.\n", call. = FALSE)
    }

    # Compute a few useful items
    nobs <- dim(rqfit)[1L]
    xseries <- seq(from = min_x, to = max_x, length.out = length.out)

    # Initial parameter values: mu, sigma, lambda, qn
    param_label <- c("mu", "sigma", "lambda", "qn")

    # Results storage
    st_params <- zeros(nobs, length(init_params))
    colnames(st_params) <- param_label

    # Do nonlinear curve fitting by minimising the sum of squared errors
    # NB: The function fn can return NaN if the function cannot be evaluated at
    # the supplied value, but the function must be computable at the initial value.
    for (i in seq_len(nobs)) {

        # Using ucminf to get the Skewed T Distribution parameters
        st_params[i, ] <- st_param_trans(ucminf(par = init_params, fn = fit_st_inv_cdf, gr = NULL,
                                         x = rqfit[i, ], probs = tau, mean_opt = mean_opt, var_opt = var_opt,
                                         control = list(trace = -1), hessian = 0)$par)

    }

    # Compute a sequence of Skewed T densities
    st_density <- zeros(nobs, length(xseries))
    colnames(st_density) <- xseries

    for (i in seq_len(nobs)) {
        param_t <- st_params[i, ]
        st_density[i, ] <- sgt::dsgt(x = xseries, mu = param_t[1L], sigma = param_t[2L],
                                     lambda = param_t[3L], p = 2.0, q = param_t[4L],
                                     mean.cent = mean_opt, var.adj = var_opt, log = FALSE)
    }

    # Collect results as a list object
    results <- list()
    results$st_params <- st_params
    results$st_density <- st_density

    # Return results
    return (results)

}

# Skewed T expected short fall / long rise
st_expected_shortfall <- function(mu, sigma, lambda, qn, prob = 0.05,
                                  mean_opt = FALSE, var_opt = FALSE, lower_tail = TRUE)
{
    if (prob > 0.5) { prob <- 1.0 - prob }
    val <- sgt::qsgt(prob = prob, mu = mu, sigma = sigma,
                     lambda = lambda, p = 2.0, q = qn,
                     mean.cent = mean_opt, var.adj = var_opt,
                     lower.tail = lower_tail, log.p = FALSE)
    fn <- function(x) { x * sgt::dsgt(x = x, mu = mu, sigma = sigma,
                                      lambda = lambda, p = 2.0, q = qn,
                                      mean.cent = mean_opt, var.adj = var_opt, log = FALSE) }
    if (lower_tail) {
        tail_area <- stats::integrate(f = fn, lower = -Inf, upper = val)$value
    } else {
        tail_area <- stats::integrate(f = fn, lower = val, upper = Inf)$value
    }
    result <- tail_area / prob
    return (result)
}

# Estimate a sequence of expected Short fall and Long rise
# at alpha per cent based on the Skewed T distribution
seq_ex_loss <- function(st_params, alpha = 0.05, mean_opt = FALSE, var_opt = FALSE)
{
    # Compute a few useful things
    nobs <- dim(st_params)[1L]

    # Results storage
    ex_loss <- zeros(nobs, 2L)

    # Compute the the sequence of short fall / long rise
    for (i in seq_len(nobs)) {

        param_t <- st_params[i, ]

        ex_loss[i, 1L] <- st_expected_shortfall(mu = param_t[1L], sigma = param_t[2L],
                                                lambda = param_t[3L], qn = param_t[4L],
                                                prob = alpha, mean_opt = mean_opt,
                                                var_opt = var_opt, lower_tail = TRUE)

        ex_loss[i, 2L] <- st_expected_shortfall(mu = param_t[1L], sigma = param_t[2L],
                                                lambda = param_t[3L], qn = param_t[4L],
                                                prob = (1.0 - alpha), mean_opt = mean_opt,
                                                var_opt = var_opt, lower_tail = FALSE)
    }

    # Return the sequence of short fall / long rise
    return (ex_loss)

}

# EOF
