####################################################################################################
# RDP 2021-03: Financial Conditions and Downside Risk to Economic Activity in Australia
####################################################################################################
# Construct a Financial Conditions Index (FCI) for Australia to be used in 'Growth-at-Risk' analysis
#
# The FCI is based on a dynamic factor model with one dynamic factor estimated by QMLE for the
# full panel of financial data.
#
# Luke Hartigan, 10-03-2021
####################################################################################################

# Clear the workspace
rm(list = ls(all = TRUE))

# Set directories
d_location <- "Data/"
c_location <- "Code/methods/"
r_location <- "Results/csv/"

# Source required functions
source(paste0(c_location, "qmle_dfm_methods.R"))   # DFM with dynamic loadings
source(paste0(c_location, "gar_methods.R"))        # project-specific functions

# Set up a few options
options(digits = 4)

# What are we doing?
cat("Estimating a Financial Conditions Index for Australia...\n")

####################################################################################################
# Read in the data -- already transformed and conditionally standardised (tfs)
####################################################################################################

# String time series dates
ts_begin_str <- "1976-12-01"    # first difference
ts_end_str <- "2020-09-01"      # *** Manually edit this line ***

# Transformed (tf) and conditionally standardised (s) financial dataset
fci_data_file <- "fci_data_tfs.RData"
load(paste0(d_location, fci_data_file))
y <- paneltfs

# Category codes
info <- read.csv(paste0(d_location, "fci_info.csv"), header = TRUE, sep = ',', row.names = 1L)
tgroup <- info["tgroup", , drop = FALSE]

# Time series dates
ts_begin <- get_year_quarter(ts_begin_str)
ts_freq <- 4
ts_freq_str <- "quarter"
ts_qtr_seq <- seq(from = as.Date(ts_begin_str), to = as.Date(ts_end_str), by = ts_freq_str)

# Convert to a 'ts' object
y <- ts(data = y, start = ts_begin, frequency = ts_freq)

####################################################################################################
# Financial Conditions Index (FCI) Estimation
####################################################################################################

# FCI -- Cross-sectional mean (robustness check)
fac_mu <- rowMeans(x = y, na.rm = TRUE)

# FCI -- Static factor model estimator (robustness check)
# How to handle NAs in the dataset?
na_opt <- "exclude"     # Remove NAs

# Dataset treated for NAs
yna <- remove_na_values(x = y, na_opt = na_opt)

# PC Factor estimation options
r <- 1L                 # Number of static factors (based on estimate of q)
norm_opt <- "LN"
scale_opt <- FALSE      # Already standardised
sign_opt <- TRUE
vardec_opt <- FALSE      # We don't want the explained variation of each factor

# Estimate PCA for balanced panel
sfm <- pc_factor(x = yna, r = r, norm_opt = norm_opt, scale_opt = scale_opt,
                 sign_opt = sign_opt, vardec_opt = vardec_opt)

fac_pc <- rbind(matrix(data = NA_real_, nrow = (dim(y)[1L] - dim(yna)[1L]), ncol = 1L),
                sfm$factors) # account for missing obs
lam_pc <- sfm$loadings

# FCI -- QMLE-DFM estimator (main estimator)
# QMLE-DFM estimation options
q <- 1L                 # Number of dynamic factors
s <- 1L                 # Number of lags in the loadings filter
p <- 1L                 # Number of lags in the factor VAR
id_opt <- "DFM2"        # 'Named-factor' normalisation method
scale_opt <- FALSE      # Already standardised
sign_opt <- FALSE       # Not necessary as using named factor normalisation
max_iter <- 500L
threshold <- 1E-4       # Not too strict!
check_increased <- TRUE
verbose <- TRUE

# Estimate DFM for unbalanced panel
dfm <- qmle_dfm(x = y, q = q, s = s, p = p,
                id_opt = id_opt, na_opt = na_opt, scale_opt = scale_opt,
                sign_opt = sign_opt, max_iter = max_iter, threshold = threshold,
                check_increased = check_increased, verbose = verbose)

fac_em <- dfm$factors
fac_se <- dfm$factor_se
lam_em <- dfm$loadings

# Real-time DFM
rt_dfm <- real_time_factor(x = y, q = q, s = s, p = p,
                           params = dfm$parameters, scale_opt = scale_opt)
rt_fac <- rt_dfm$factors
rt_fac_se <- rt_dfm$factor_se

# Compute the decomposition of the FCI by data category
decomp <- y * repmat(mat = t(as.matrix(rowSums(lam_em))), m = dim(y)[1L], n = 1L)
group_names <- c('A', 'B', 'C', 'D', 'E', 'F', 'G', 'H', 'I')
ngroups <- length(group_names)

# Decomposition results storage
decomp_by_cat <- zeros(dim(y)[1L], ngroups)
for (i in seq_len(ngroups)) {
    decomp_by_cat [, i] <- rowSums(x = decomp[, which(tgroup == group_names[i]), drop = FALSE], na.rm = TRUE)
}

# Get the aggregate for all categories
decomp_total <- rowSums(x = decomp_by_cat, na.rm = TRUE)
sd_decomp_total <- sd(x = decomp_total)

# Convert to 'ts' objects
fac_mu <- ts(data = fac_mu, start = ts_begin, frequency = ts_freq)
fac_pc <- ts(data = fac_pc, start = ts_begin, frequency = ts_freq)
fac_em <- ts(data = fac_em, start = ts_begin, frequency = ts_freq)
rt_fac <- ts(data = rt_fac, start = ts_begin, frequency = ts_freq)

# Get FCI -- Cross-sectional mean
fci_mu <- as.matrix(fac_mu)
rownames(fci_mu) <- as.character(ts_qtr_seq)
colnames(fci_mu) <- "FCI-MU"

# Get FCI -- PCA (including missing observations coded as NA)
fci_pc <- as.matrix(fac_pc)
rownames(fci_pc) <- as.character(ts_qtr_seq)
colnames(fci_pc) <- "FCI-PC"

# Get the FCI -- QMLE DFM (dynamic loadings)
fci <- fac_em[, 1L, drop = FALSE]
rownames(fci) <- as.character(ts_qtr_seq)
colnames(fci) <- "FCI"

# Get the FCI standard error
fci_se <- fac_se[, 1L, drop = FALSE]
rownames(fci_se) <- as.character(ts_qtr_seq)
colnames(fci_se) <- "FCI_SE"

# Get real-time FCI
rt_fci <- rt_fac[, 1L, drop = FALSE]
rownames(rt_fci) <- as.character(ts_qtr_seq)
colnames(rt_fci) <- "RT-FCI"

# Get real-time FCI standard error
rt_fci_se <- rt_fac_se[, 1L, drop = FALSE]
rownames(rt_fci_se) <- as.character(ts_qtr_seq)
colnames(rt_fci_se) <- "RT-FCI_SE"

# Get the FCI weights -- dynamic loadings
fci_wts <- cbind(lam_em, rowSums(lam_em))
colnames(fci_wts) <- c(colnames(lam_em), "Sum")

# Get the FCI decomposition by data category -- rescaled to have unit variance
fci_decomp <- cbind(decomp_total, decomp_by_cat)
fci_decomp <- sweep(x = fci_decomp, MARGIN = 2L, STATS = sd_decomp_total, FUN = '/', check.margin = TRUE)
rownames(fci_decomp) <- as.character(ts_qtr_seq)
colnames(fci_decomp) <- c("Total", group_names)

####################################################################################################
# Write results to disk
####################################################################################################

# DFM-FCI estimate -- '.RData'
save(fci, file = paste0(r_location, sprintf("fci_q_%d_s_%d_p_%d", q, s, p), ".RData"))

# DFM-FCI estimate -- '.csv.
write.table(x = fci,
            file = paste0(r_location, sprintf("fci_q_%d_s_%d_p_%d", q, s, p), ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

# FCI standard errors estimate
write.table(x = fci_se,
            file = paste0(r_location, sprintf("fci_se_q_%d_s_%d_p_%d", q, s, p), ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

# FCI weights
write.table(x = fci_wts,
            file = paste0(r_location, sprintf("fci_wts_q_%d_s_%d_p_%d", q, s, p), ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

# FCI decomposition by data category
write.table(x = fci_decomp,
            file = paste0(r_location, sprintf("fci_decomp_q_%d_s_%d_p_%d", q, s, p), ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

# Cross-sectional mean FCI estimate
write.table(x = fci_mu,
            file = paste0(r_location, "mu_fci", ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

# PCA-FCI estimate
write.table(x = fci_pc,
            file = paste0(r_location, sprintf("pca_fci_r_%d", q), ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

# Real-time FCI estimate
write.table(x = rt_fci,
            file = paste0(r_location, sprintf("rt_fci_q_%d_s_%d_p_%d", q, s, p), ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

# Real-time FCI standard errors estimate
write.table(x = rt_fci_se,
            file = paste0(r_location, sprintf("rt_fci_se_q_%d_s_%d_p_%d", q, s, p), ".csv"),
            append = FALSE, quote = FALSE, sep = ',', row.names = TRUE, col.names = NA)

cat(sprintf("All files written to: %s\n", r_location))

# EOF
