clear all
*************************** 4. MAIN STRESS MODEL ******************************
**************************** DATE: AUGUST 2020 ********************************
**************************** AUTHOR: MIKE MAJOR *******************************
*************** ADAPTED FROM BILSTON, JOHNSTON AND READ (2015) ****************
set matsize 1600

*****************************************************************************
*TABLE OF CONTENTS
*****************************************************************************
*0. IMPORT DATA AND DEFINE KEY VARIABLES
*1. GENERATE BASIC CONSUMPTION AND UNEMPLOYMENT BENEFITS
*2. CREATING MINIMUM MORTGAGE REPAYMENT VARIABLE, PRE STRESS
*3. GENERATE PDs, PRE-STRESS
*4. GENERATE EADs, PRE-STRESS
*5. GENERATE NET DEBT VARIABLES, PRE-STRESS
*6. GENERATE LMI PAYOUT RATIOS
*7. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL PRE-STRESS, USING TOTAL ASSETS
*8. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL PRE-STRESS, USING ONLY PROPERTY ASSETS
*9. CALCULATE LGDs AT POPULATION LEVEL
*10. CALCULATE PRE-STRESS SYSTEM LOSSES = EAD*LGD
*11. GENERATE PRE-STRESS SUMMARY STATS
*12. GENERATE STRESSED VARIABLES
*13. GENERATE MINIMUM MORTGAGE REPAYMENT VARIABLE, POST-STRESS
*14. GENERATE MPCs of WEALTH AND INCOME
*15. CARRY OUT MONTE CARLO SIMULATION FOR UNEMPLOYMENT
*16. GENERATE PDs, POST-STRESS
*17. GENERATE EADs, POST-STRESS
*18. GENERATE NET DEBT VARIABLES POST-STRESS
*19. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL POST-STRESS, USING TOTAL ASSETS
*20. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL POST-STRESS, USING ONLY PROPERTY ASSETS
*21. CALCULATE LGDs AT POPULATION LEVEL
*22. CALCULATE SYSTEM LOSSES, POST-STRESS
*23. EFFECT ON CONSUMPTION FROM WEALTH AND INCOME SHOCKS
*24. GENERATE VARIABLES TO SUMMARIZE RESULTS
*25. RUN SIMULATION AND EXPORT RESULTS

*****************************************************************************
*0. IMPORT DATA AND DEFINE KEY VARIABLES
*****************************************************************************

*****************************************************************************
*Import data                          			  				
*****************************************************************************
*Combine household, loan and person-level data from SIH/HES data and unemployment model
use "Data\hh_${year}_${hes}.dta", clear 
drop _merge
if ${hes} == 1 {
	merge 1:1 new_abshid using "Data\unempl_stress_${year}_hes_${batchUR}.dta" 
}
if ${hes} == 0 {
	merge 1:1 new_abshid using "Data\unempl_stress_${year}_sih_${batchUR}.dta" 
}

*For missing persons, set weights to zero
foreach x of varlist pwt* {
	replace `x'=0 if `x'==. 
}

*Annualise income variables 
foreach x of varlist disp_inc* emp_inc* emp_extras* inv_inc* oth_inc* tot_inc* {
	replace `x' = `x' * 52 
	replace `x' = 0 if `x' == . // Replace missing individual estimates with zero 
} 

gen year = ${year}	
*Assume interest rates for credit card and other personal loans are based on June of survey year
	*Source: https://www.rba.gov.au/statistics/tables/xls/f05hist.xls
	//Credit card 		// Lending rates; Personal loans; Revolving credit; Credit cards; Standard
	//Other personal 	// Lending rates; Personal loans; Term loans (unsecured);
if ${year}==2018 {
	global ccrate = 0.199
	global oprate = 0.031 
}
if ${year}==2016 {
	global ccrate = 0.1975
	global oprate = 0.1435
}
if ${year}==2010 {	
	global ccrate = 0.195		
	global oprate = 0.147	
}	
if ${year}==2004 {	
	global ccrate = 0.165		
	global oprate = 0.118
}		

*****************************************************************************
*Define key variables                           			  				
*****************************************************************************

*INCOME	
//disposable income including imputed rent
gen hhtotaldis = dincann + netir*52
xtile hhtotaldis_x = hhtotaldis[aw=hwt], nquantiles(5) 

//disposable income excluding imputed rent (identical to dispscp8)
gen hhtotaldis_noIR = dincann	 

//disposable income excluding imputed rent (from person-level data)
gen disp_inc = disp_inc1 + disp_inc2 + disp_inc3 + disp_inc4 + disp_inc5 + disp_inc6 

//gross household income excluding imputed rent
gen tot_inc = tot_inc1 + tot_inc2 + tot_inc3 + tot_inc4 + tot_inc5 + tot_inc6 

//goods and services expenditure including imputed rent	
gen gs_exp = extlcser + netir 
gen gs_exp_ann = gs_exp * 52
gen gs_exp_ann_nonIR = gs_exp_ann - (netir + wkrentch)*52 // excludes net imputed rent and rent in order to compare to bascon (which excludes rent and borrower housing costs)
	
*ASSETS 
//Assume consumption of young households are not sensitive to superannuation falls
gen share_super_asset = .
replace share_super_asset = (age - ${age_super_low} )/(${age_super_high} - ${age_super_low})
replace share_super_asset = 1 if age>${age_super_high}
replace share_super_asset = 0 if age<${age_super_low}				
	
g asset_sp = vsuptot //super
g asset_sp_lin = asset_sp*share_super_asset // super after sensitivity
g asset_hs = hvaluech+vrprch // housing
g asset_eq = vsharch //equity
g asset_nres = vnrprch // non-res
g asset_tr = vtrtch // trusts
g asset_oth_fin = vfinch+voftch+vdebch+vplnch+vinvotch // other financial assets	
g asset_oth = vsipch+vibusch+vubusch+vcontch+vvehich+votassch // other assets
g asset_to = asset_sp+asset_hs+asset_eq+asset_nres+asset_tr+asset_oth_fin+asset_oth //total assets
g asset_to_linsup = asset_sp_lin+asset_hs+asset_eq+asset_nres+asset_tr+asset_oth_fin+asset_oth // total assets after super adjustment	

gen propasset = asset_hs + asset_nres // Property assets, pre-stress						
gen liquid_financial_assets = vfinch + voftch + vsharch + vdebch // liquid assets, pre-stress	
	
*LIABILITIES (defined in merge files because they remain unchanged in stress)		
	
*WEALTH		
gen wealth = asset_to_linsup - hhdebt 
label variable wealth "Household wealth (with super adjustment and excluding student debt)"
				
*****************************************************************************
*1. GENERATE BASIC CONSUMPTION AND UNEMPLOYMENT BENEFITS
*****************************************************************************
	
* Create household composition variable to determine unemployment benefits	
gen hhtype =.
replace hhtype=1 if famcomp==1 /* one couple */ & dpkidhbc == 0 /* no. deps under 25 */ 
replace hhtype=2 if famcomp==1 /* one couple */ & dpkidhbc == 1 /* no. deps under 25 */ 
replace hhtype=3 if famcomp==1 /* one couple */ & dpkidhbc == 2 /* no. deps under 25 */ 
replace hhtype=4 if famcomp==1 /* one couple */ & dpkidhbc == 3 /* no. deps under 25 */ 
replace hhtype=5 if famcomp==1 /* one couple */ & dpkidhbc > 3 /* no. deps under 25 */ 
replace hhtype=6 if famcomp >=2 & famcomp <=4 /* one parent */ & dpkidhbc == 0 /* no. deps under 25 */
replace hhtype=6 if famcomp ==8 /* lone person */ & dpkidhbc == 0 /* no. deps under 25 */ 
replace hhtype=7 if famcomp >=2 & famcomp <=4 /* one parent */ & dpkidhbc == 1 /* no. deps under 25 */ 
replace hhtype=8 if famcomp >=2 & famcomp <=4 /* one parent */ & dpkidhbc == 2 /* no. deps under 25 */ 
replace hhtype=9 if famcomp >=2 & famcomp <=4 /* one parent */ & dpkidhbc == 3 /* no. deps under 25 */ 
replace hhtype=10 if famcomp >=2 & famcomp <=4 /* one parent */ & dpkidhbc > 3 /* no. deps under 25 */

*Assume other households get couple rates based on dependent children
replace hhtype=1 if famcomp>=5 & famcomp<=7 /* other/multi-couple fam */ & dpkidhbc == 0 /* no. deps under 25 */ 
replace hhtype=1 if famcomp>=9 & famcomp<=10 /* other/multi-couple fam */ & dpkidhbc == 0 /* no. deps under 25 */ 
replace hhtype=2 if famcomp>=5 & famcomp<=7 /* other/multi-couple fam */ & dpkidhbc == 1 /* no. deps under 25 */ 
replace hhtype=2 if famcomp>=9 & famcomp<=10 /* other/multi-couple fam */ & dpkidhbc == 1 /* no. deps under 25 */ 
replace hhtype=3 if famcomp>=5 & famcomp<=7 /* other/multi-couple fam */ & dpkidhbc == 2 /* no. deps under 25 */ 
replace hhtype=3 if famcomp>=9 & famcomp<=10 /* other/multi-couple fam */ & dpkidhbc == 2 /* no. deps under 25 */ 
replace hhtype=4 if famcomp>=5 & famcomp<=7 /* other/multi-couple fam */ & dpkidhbc == 3 /* no. deps under 25 */ 
replace hhtype=4 if famcomp>=9 & famcomp<=10 /* other/multi-couple fam */ & dpkidhbc == 3 /* no. deps under 25 */ 
replace hhtype=5 if famcomp>=5 & famcomp<=7 /* other/multi-couple fam */ & dpkidhbc > 3 /* no. deps under 25 */ 
replace hhtype=5 if famcomp>=9 & famcomp<=10 /* other/multi-couple fam */ & dpkidhbc > 3 /* no. deps under 25 */ 

label variable hhtype "household type"
label define hhtype_1 1 "couple" 2 "couple plus 1" 3 "couple plus 2" 4 "couple plus 3" 5 "couple plus 4 or more" 6 "single person" 7 "single parent plus 1" 8 "single parent plus 2" 9 "single parent plus 3" 10 "single parent plus 4 or more"
label values hhtype hhtype_1

*Head of household working (for Henderson Poverty Lines calibration)
gen headworking = 0 
replace headworking = 1 if lfsrh == 1 	

* Generate a variable for basic consumption 
gen bascon = .
//HPL:using the June quarter Henderson Poverty Lines (excluding housing). 
	*For households with more than four children, we assume that they receive the same marginal amount of money for an additional child as households moving from three to four children
	*Source: https://melbourneinstitute.unimelb.edu.au/assets/documents/poverty-lines/2016/Poverty-Lines-Australia-June-2016.pdf
//HEM:using the June quarter Household Expenditure Measure
if ${year} == 2018 {
	egen y_bucket = cut(tot_inc), at(-1000000,21000,31000,42000,52000,63000,84000,105000,126000,146000,167000,209000,262000,314000,20000000) icodes
	tabulate y_bucket
	merge m:1 hhtype y_bucket headworking using "Data\bascon.dta", keep(match master) nogenerate
	replace bascon = hpl18*52 if ${bascontype} == 0 // use HPL 
	replace bascon = hem18*52 if ${bascontype} == 1 // use HEM	
	}
	
if ${year} == 2016 {
	egen y_bucket = cut(tot_inc), at(-1000000,23156,28945,34733,40522,46311,52100,57889,69467,81045,92623,104200,115778,138934,162090,185245,231557,20000000) icodes
	tabulate y_bucket
	merge m:1 hhtype y_bucket headworking using "Data\bascon.dta", keep(match) nogenerate
	replace bascon = hpl16*52 if ${bascontype} == 0 // use HPL 
	replace bascon = hem16*52 if ${bascontype} == 1 // use HEM
	}

if ${year} == 2010 {
	egen y_bucket = cut(tot_inc), at(-1000000,20000,25000,30000,35000,40000,45000,50000,60000,70000,80000,90000,100000,120000,140000,160000,200000,20000000) icodes
	merge m:1 hhtype y_bucket headworking using "Data\bascon.dta", keep(match) nogenerate
	replace bascon = hpl10*52 if ${bascontype} == 0 // use HPL
	replace bascon = hem10*52 if ${bascontype} == 1 // use HEM
	}
	
if ${year} == 2004 {
	egen y_bucket = cut(tot_inc), at(-1000000,20000,25000,30000,35000,40000,45000,50000,60000,70000,80000,90000,100000,120000,140000,160000,200000,20000000) icodes
	merge m:1 hhtype y_bucket headworking using "Data\bascon.dta", keep(match) nogenerate
	replace bascon = hpl04*52 if ${bascontype} == 0 // use HPL
		if ${bascontype} == 1 {
		di "cannot use HEM for 2004 data"
		exit
		}
	}	
	
gen subsist = 0
replace subsist = 1 if gs_exp_ann_nonIR<bascon & bascon!=.
label variable subsist "G&S expenditure (excluding IR and rent) is less than bascon, pre-stress"

* Generate a variable for estimated maximum benefits that unemployed individuals in a household would be eligible to receive
gen unben = .
label variable unben "maximum unmployment benefits"
*For simplicity, assume Family Tax Benefit Part A and B stay the same after the unemployment shock
*Source: http://guides.dss.gov.au/guide-social-security-law/5/2/1/20 
*Source: https://guides.dss.gov.au/guide-social-security-law/5/2/1/30
*For households without children, assume Newstart allowance after unemployment shock
*Couples without children
replace unben = 246.40*52 if hhtype == 1 | hhtype == 2 | hhtype == 3 | hhtype == 4 | hhtype == 5 & ${year} == 2018
replace unben = 238.20*52 if hhtype == 1 | hhtype == 2 | hhtype == 3 | hhtype == 4 | hhtype == 5 & ${year} == 2016
replace unben = 208.85*52 if hhtype == 1 | hhtype == 2 | hhtype == 3 | hhtype == 4 | hhtype == 5 & ${year} == 2010
replace unben = 175.55*52 if hhtype == 1 | hhtype == 2 | hhtype == 3 | hhtype == 4 | hhtype == 5 & ${year} == 2004

*Lone person without children
replace unben = 273.90*52 if hhtype == 6 & ${year} == 2018
replace unben = 263.80*52 if hhtype == 6 & ${year} == 2016
replace unben = 231.40*52 if hhtype == 6 & ${year} == 2010
replace unben = 194.60*52 if hhtype == 6 & ${year} == 2004

*For parents with children, assume Parenting Payment after unemployment shock
*http://guides.dss.gov.au/guide-social-security-law/5/2/4/50
replace unben = 762.40*26 if hhtype == 7 | hhtype == 8 | hhtype == 9 | hhtype == 10 & ${year} == 2018
replace unben = 737.10*26 if hhtype == 7 | hhtype == 8 | hhtype == 9 | hhtype == 10 & ${year} == 2016
replace unben = 601.30*26 if hhtype == 7 | hhtype == 8 | hhtype == 9 | hhtype == 10 & ${year} == 2010
replace unben = 464.20*26 if hhtype == 7 | hhtype == 8 | hhtype == 9 | hhtype == 10 & ${year} == 2004

*****************************************************************************
*2. CREATING MINIMUM MORTGAGE REPAYMENT VARIABLE, PRE STRESS
*****************************************************************************	
*Generate flag for households with no repayments in household level data but repayments in loan level data
gen flag1 = 0
replace flag1 = 1 if trpaych==0 & totwkrep_home>0 & propertydebt > 0 & totwkrep_home!=. 		

* Years since inception of property debt
gen homeloanage = .
replace homeloanage = ${year} - yr_rp_start_home 
replace homeloanage = ${loanterm} - 1 if homeloanage >= ${loanterm} & homeloanage!=. // Assume loans older than loan term are equal to loan term for credit frontier model

*Years until debt matures, assuming a typical loan term mortgage
gen homeloan_yr_to_maturity = ${loanterm} - homeloanage

*Interest rate
gen mortir = .
replace mortir = aveintr_home/100

*Estimate scheduled balance to calculate redraw
*Adjust mortgage rate for interest rate changes since origination 
gen homeloanage_round=round(homeloanage, .5) 
merge m:1 homeloanage_round using "Data\rate_changes.dta", keep(master match) nogenerate
gen mortir_av = mortir + av_to_${year}/100	
*Historical required payment using original balance
gen historical_required_br = (totbr_home * mortir_av/12)/(1 - (1 + mortir_av/12)^-(${loanterm}*12))*12	
*Scheduled balance
gen scheduled_balance = (totbr_home * (1+mortir_av/12)^(homeloanage*12)) - (historical_required_br/12 * (((1 + mortir_av/12)^(homeloanage*12)) - 1) / (mortir_av/12))
replace scheduled_balance = 0 if scheduled_balance<0	

*Generate buffer from redraw and offset accounts for hand-to-mouth calculation	
gen redraw = 0 
replace redraw = scheduled_balance - toto_home 
replace redraw = 0 if redraw < 0 | redraw == . 

gen bal_after_redraw_and_offset = toto_home
replace bal_after_redraw_and_offset = toto_home - voftch if voftch!=.
replace bal_after_redraw_and_offset = 0 if bal_after_redraw_and_offset<0
	
*Calculate minimum required mortgage repayments for all property loans
*Make conservative assumption that households required minimum is the scheduled minimum, but then allow household to use redraw and offset to meet repayments
gen required =  (scheduled_balance * mortir/12)/(1 - (1 + mortir/12)^-(homeloan_yr_to_maturity*12))*12 

*Define interest-only loans as those with a current balance close to orig balance & more than one year since origination
gen interest_only = 0 if totbr_home!=. & totbr_home>0
replace interest_only = 1 if totbr_home!=. & totbr_home>0 & toto_home>0 & toto_home!=. & toto_home/totbr_home*100>=${min_share_for_io} & toto_home/totbr_home*100<=${max_share_for_io} & homeloanage_round>=1

* Adjust required interest for interest-only borrowers
replace required = toto_home * mortir if interest_only==1 
*Allow offset balances to reduce minimum required repayment
if ${offsets_reduce_interest} == 1 {		
	replace required = (toto_home-voftch) * mortir if interest_only==1 		
}		
* Ensure required is greater than zero
replace required = 0 if required<0

*Weekly minimum required repayment
gen requiredwkhome = required / 52

*Replace minimum repayment with actual repayment for some households:
*For households without repayments in household data but repayments in loan-level data:
*...and required repayment is more than actual
gen flag5=0			
replace flag5=1 if totwkrep_home < requiredwkhome & requiredwkhome!=. & flag1==1 
*...and required repayment is undefined
gen flag6=0
replace flag6=1 if requiredwkhome==. & flag1==1

*For households with repayments in household data:
*...where actual repayment < minimum repayment
gen flag7=0
replace flag7=1 if trpaych < requiredwkhome & requiredwkhom!=. & flag1==0 & trpaych>0

*...(that are positive) where minimum repayment is undefined
gen flag8=0
replace flag8=1 if requiredwkhome==. & trpaych>0 & flag1==0
*...(that are 0) where minimum repayment is undefined
gen flag9=0
replace flag9=1 if requiredwkhome==. & trpaych==0 & flag1==0

replace requiredwkhome = totwkrep_home if flag6==1 | flag5==1 		
replace requiredwkhome = trpaych if flag7==1 | flag8 ==1 | flag9 == 1		
	
*Define hand-to-mouth individuals
*Define liquid and illiquid wealth
gen liqnworth = vfinch + voftch + vsharch + vdebch - ccdebt	
gen liqnw_adj = liqnworth + redraw 
gen credit_limit = dincann/12 //credit limit set to monthly pay (see KWV 2014)
gen pay = dincann/26 //fornightly pay 

//HTM according to KVW (2014)
gen handtomouth = 0 if pay/2<liqnworth & liqnworth>=0
replace handtomouth=1 if ((pay/2-credit_limit)>=liqnworth & liqnworth<=0)
replace handtomouth=0 if ((pay/2-credit_limit)<liqnworth & liqnworth<=0)
replace handtomouth=1 if (pay/2>=liqnworth & liqnworth>=0)			

//HTM according to KVW (2014) adjusted for prepayment buffers 
gen htm_adj = 0 if pay/2<liqnw_adj & liqnw_adj>=0
replace htm_adj=1 if ((pay/2-credit_limit)>=liqnw_adj & liqnw_adj<=0)
replace htm_adj=0 if ((pay/2-credit_limit)<liqnw_adj & liqnw_adj<=0)
replace htm_adj=1 if (pay/2>=liqnw_adj & liqnw_adj>=0)			

if ${htm_with_buffer} == 1 /* include buffers */ {    		
replace handtomouth = htm_adj
}	
	
*****************************************************************************
*3. GENERATE PDs, PRE-STRESS
*****************************************************************************	
	
*Generate an estimate of annual credit card interest repayments 
gen ccdebtrepay = ccdebt*(${ccrate}+${ccprincipalrepay})

*Generate estimate of repayment on non-property, non-business, non-cc debt 
gen opdebtrepay = nonpropbusccdebt*(${oprate}+${opdebtprincipalrepay})
	
*Generate annual debt servicing 
gen debtserv = requiredwkhome*52 + ccdebtrepay + opdebtrepay 
label variable debtserv "debt servicing and rent"
	
*Generate financial margin of the household (inlcuding actual and imputed rent)
gen finmarg = hhtotaldis - bascon - debtserv - (wkrentch + netir)*52
label variable finmarg "financial margin"
	
*include asset measures into financial margins
if ${fm_switch} == 1 {    // liquid assets
	replace finmarg = finmarg + (liquid_financial_assets+redraw)/${stress_length}  
	}	
		
*Calculate PD from financial margin
gen pd = 1 if finmarg < 0
replace pd = 0 if finmarg >= 0
replace pd =. if finmarg==.

*Calculate population PD
egen pdp_numerator = mean(pd*100*hwt) if pd!=.
egen pdp_denominator = mean(hwt) if pd!=.
gen pdp = pdp_numerator / pdp_denominator
drop pdp_numerator pdp_denominator			
		
*****************************************************************************
*4. GENERATE EADs, PRE-STRESS
*****************************************************************************	
		
*****************************
* INDIVIDUAL HOUSEHOLD LEVEL 
*****************************

*Generate exposure at default (EAD) for types of debt for each household
*Total secured EAD
gen secead = propertydebt*pd
label variable secead "individual EAD total secured"	
*First mortgage EAD
gen mortead = hdebt*pd
label variable mortead "individual EAD first mortgage"
*Other secured EAD
gen othsecead = nonhomepropertydebt*pd
label variable othsecead "individual EAD other secured"

*Credit card EAD
gen ccead = ccdebt*pd
label variable ccead "individual EAD credit card"
*Other unsecured EAD
gen othead = nonpropbusccdebt*pd
label variable othead "individual EAD other unsecured"
			
*****************************
* POPULATION LEVEL
*****************************

*sum together household EAD's to give system EAD (need to population weight the data) i.e. each household will have this population value for this variable. 
*Total secured
egen popsecead = total(secead*hwt), missing
label variable popsecead "population EAD total secured"
*First mortgage
egen popmortead = total(mortead*hwt), missing
label variable popmortead "population EAD first mortgage"
*Other secured
egen popothsecead = total(othsecead*hwt), missing
label variable popsecead "population EAD other secured"
*Credit card
egen popccead = total(ccead*hwt), missing
label variable popccead "population EAD credit card"
*Other unsecured
egen popothead = total(othead*hwt), missing
label variable popothead "population EAD other unsecured"

*In percentage terms
*first calculate population weighted debt
egen popsecdebt = total(propertydebt*hwt)
label variable popsecdebt "population total secured debt"
egen popmortdebt = total(hdebt*hwt)
label variable popmortdebt "population first mortgage debt"
egen popothsecdebt = total(nonhomepropertydebt*hwt)
label variable popothsecdebt "population other secured debt"
egen popccdebt = total(ccdebt*hwt)
label variable popccdebt "population total credit card debt"
egen popothdebt = total(nonpropbusccdebt*hwt)
label variable popothdebt "population total other unsecured debt"

*Calculation EADs as a percentage of debt, i.e. the percentage of debt that belongs to vunerable households
gen poptoteadp = (popothead + popsecead + popccead)/(popothdebt + popsecdebt + popccdebt)*100   
label variable poptoteadp "population EAD - % of total debt"
gen popseceadp = popsecead/popsecdebt*100
label variable popseceadp "population total secured EAD, % of total secured debt"
gen popmorteadp = popmortead/popmortdebt*100
label variable popmorteadp "population mortgage EAD, % of total mortgage debt"
gen popothseceadp = popothsecead/popothsecdebt*100
label variable popothseceadp "population other unsecured EAD, % of total secured debt"
gen popcceadp = popccead/popccdebt*100
label variable popcceadp "population credit card EAD, % of total credit card debt"
gen popotheadp = popothead/popothdebt*100
label variable popotheadp "population other unsecured EAD, % of total other unsecured debt"

*****************************************************************************
*5. GENERATE NET DEBT VARIABLES, PRE-STRESS
*****************************************************************************	

*Net debt taking into account all assets
gen netdebt = hhdebt - asset_to_linsup
replace netdebt = 0 if netdebt<0
label variable netdebt "net debt, using total assets, pre-stress"

*Net debt taking into account only property assets
gen netdebth = hhdebt - propasset
replace netdebth = 0 if netdebth<0
label variable netdebth "net debt, using property assets, pre-stress"					
	
*****************************************************************************
*6. GENERATE LMI PAYOUT RATIOS
*****************************************************************************	

*Current LVR estimate of first mortgage
gen LVR=(hdebt)/hvaluech*100
label variable LVR "Current LVR, first mortgage"
replace LVR = 0  if hvaluech == 0 | LVR==.

*Current LVR estimate for all property
gen LVRproperty = propertydebt/propasset*100
label variable LVRproperty "Current LVR, all property"
replace LVRproperty = 0 if propasset == 0 | LVRproperty==.

*Current LVR estimate for all non-home property
gen nonhomeLVR = nonhomepropertydebt/(vrprch+vnrprch)*100
label variable nonhomeLVR "Current LVR, non-home property"
replace nonhomeLVR = 0  if vrprch+vnrprch == 0 | nonhomeLVR==.

*Generate pre-stress payout ratios
gen poratio = 0
if ${LMI_flag} == 0 { // original method
	*Generate payout ratios based, following APRA
	replace poratio = 0.98
	label variable poratio "pre-stress LMI payout ratio"
	}

*Generate post-stress payout ratios
gen poratio_ = 0
if ${LMI_flag} == 0 { 
	replace poratio_ = 0.7
	label variable poratio_ "post-stress LMI payout ratio"
	}

*****************************************************************************
*7. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL PRE-STRESS, USING TOTAL ASSETS
*****************************************************************************

*****************************
* EXCLUDING LMI
*****************************
gen lgd = pd*netdebt
replace lgd = pd*netdebt + (hdebt+nonhomepropertydebt)*${foreclosure}*pd if LVR>${foreclosure_lvr}
label variable lgd "household total LGD - using total assets, excl LMI"

*****************************
* INCLUDING LMI
*****************************

*To capture LMI, need to assume that debt is reclaimed from a defaulted household's assets in a particular order
*Assume the following order for assets being used to repay a defaulted household's obligations
	*1 First mortgage  
	*2 Other secured debts
	*3 Personal loans
	*4 Credit card

*Generate first mortgage LGD
gen mortlgd = 0
replace mortlgd = (hdebt - asset_to_linsup)*pd
replace mortlgd = 0 if mortlgd < 0 // if assets>debt, LGD=0
label variable mortlgd "mortgage LGD - using total assets, excl LMI"

*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio
gen mortlgdlmi = mortlgd
replace mortlgdlmi = mortlgd*(1-poratio) if LVR>80 
label variable mortlgdlmi "mortgage LGD - using total assets, incl LMI"

*Generate other secured LGD
gen othseclgd = 0
replace othseclgd = (nonhomepropertydebt - (asset_to_linsup - hdebt))*pd   
replace othseclgd = 0 if othseclgd < 0 // if assets>debt, LGD=0
*If already exhausted all assets paying back first mortgage, entire other unsecured debt is written off
replace othseclgd = nonhomepropertydebt*pd if mortlgd>0
label variable othseclgd "Other secured LGD - using total assets"

*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio
gen othseclgdlmi = 0
replace othseclgdlmi = nonhomepropertydebt*pd if mortlgdlmi>0
label variable othseclgdlmi "Other secured LGD - using total assets, incl LMI"
replace othseclgdlmi = othseclgdlmi*(1-poratio) if nonhomeLVR>80 

*LGDs for remaining assets will be the same regardless of whether LMI included, as we assume that the household would exhaust total assets before LMI is drawn down

*Personal loan LGD
gen othunseclgd = 0
replace othunseclgd = (nonpropbusccdebt - (asset_to_linsup - hdebt - nonhomepropertydebt))*pd   
replace othunseclgd = 0 if othunseclgd < 0 // if assets>debt, LGD=0
*If already exhausted all assets paying back secured debts then entire personal loan amount is written off
replace othunseclgd = nonpropbusccdebt*pd if othseclgd>0 | mortlgd>0
label variable othunseclgd "Other unsecured LGD - using total assets"

*Credit card LGD
gen cclgd = 0
replace cclgd = (ccdebt- (asset_to_linsup - hdebt - nonhomepropertydebt - nonpropbusccdebt))*pd  
replace cclgd = 0 if cclgd < 0 // if assets>debt, LGD=0
*If already exhausted all assets paying back other debt types, entire outstanding credit card debt is written off
replace cclgd = ccdebt*pd if othunseclgd>0 | othseclgd>0 | mortlgd>0
label variable cclgd "Credit card loss given default - using total assets"

*Foreclosure costs
*Add foreclosure costs to estimates of LGD
*Add these after calculating other LGDs because these are paid by the bank so shouldn't affect other loans
replace mortlgd = mortlgd + hdebt*${foreclosure}*pd if LVR>${foreclosure_lvr}
replace mortlgdlmi = mortlgdlmi + hdebt*${foreclosure}*pd if LVR>${foreclosure_lvr}
replace othseclgd = othseclgd + nonhomepropertydebt*${foreclosure}*pd if nonhomeLVR>${foreclosure_lvr}
replace othseclgdlmi = othseclgdlmi + nonhomepropertydebt*${foreclosure}*pd if nonhomeLVR>${foreclosure_lvr}

*All property LGD, excluding and including LMI
gen seclgd = othseclgd + mortlgd
gen seclgdlmi = othseclgdlmi + mortlgdlmi

*Total lgd
gen totallgd = mortlgd + othseclgd + othunseclgd + cclgd
label variable totallgd "household total LGD - using total assets, excl LMI"

*Total lgd, taking into account LMI
gen totallgdlmi = mortlgdlmi + othseclgdlmi + othunseclgd + cclgd
label variable totallgdlmi "household total LGD - using total assets, incl LMI"

*****************************************************************************
*8. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL PRE-STRESS, USING ONLY PROPERTY ASSETS
*****************************************************************************

*****************************
* EXCLUDING LMI
*****************************
gen lgdh = pd*netdebth 
replace lgdh = pd*netdebth + (hdebt+nonhomepropertydebt)*${foreclosure}*pd if LVR>${foreclosure_lvr}
label variable lgdh "household total LGD - using property assets, excl LMI"

*****************************
* INCLUDING LMI
*****************************

*To capture LMI, need to assume that debt is reclaimed from a defaulted household's assets in a particular order
*Assume the following order for assets being used to repay a defaulted household's obligations
	*1 First mortgage  
	*2 Other secured debts
	*3 Personal loans
	*4 Credit card

*Generate first mortgage LGD
gen mortlgdh = 0
replace mortlgdh = (hdebt - propasset)*pd   
replace mortlgdh = 0 if mortlgdh < 0 // if assets>debt, LGD=0
label variable mortlgdh "mortgage LGD - using property assets, excl LMI"

*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio
gen mortlgdlmih = mortlgdh
replace mortlgdlmih = mortlgdh*(1-poratio) if LVR>80 
label variable mortlgdlmih "mortgage LGD - using property assets, incl LMI"

*Generate other secured LGD
gen othseclgdh = 0
replace othseclgdh = (nonhomepropertydebt - (propasset - hdebt))*pd   
replace othseclgdh = 0 if othseclgdh < 0 // if assets>debt, LGD=0
*If already exhausted assets paying back first mortgage, entire other unsecured debt is written off
replace othseclgdh = nonhomepropertydebt*pd if mortlgdh>0
label variable othseclgdh "Other secured LGD - using property assets"

*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio
gen othseclgdlmih = 0
replace othseclgdlmih = nonhomepropertydebt*pd if mortlgdlmih>0
label variable othseclgdlmih "Other secured LGD - using property assets, incl LMI"
replace othseclgdlmih = othseclgdlmih*(1-poratio) if nonhomeLVR>80 

*LGDs for remaining assets will be the same regardless of whether LMI included, as we assume that the household would exhaust total assets before LMI is drawn down

*Personal loan LGD
gen othunseclgdh = 0
replace othunseclgdh = (nonpropbusccdebt - (propasset - hdebt - nonhomepropertydebt))*pd   
replace othunseclgdh = 0 if othunseclgdh < 0 // if assets>debt, LGD=0
*If already exhausted property assets paying back secured debts then entire personal loan amount is written off
replace othunseclgdh = nonpropbusccdebt*pd if othseclgdh>0 | mortlgdh>0
label variable othunseclgdh "Other unsecured LGD - using property assets"

*Credit card LGD
gen cclgdh = 0
replace cclgdh = (ccdebt- (propasset - hdebt - nonhomepropertydebt - nonpropbusccdebt))*pd  
replace cclgdh = 0 if cclgdh < 0 // if assets>debt, LGD=0
*If already exhausted property assets paying back other debt types, entire outstanding credit card debt is written off
replace cclgdh = ccdebt*pd if othunseclgdh>0 | othseclgdh>0 | mortlgdh>0
label variable cclgdh "Credit card loss given default - using property assets"

*Foreclosure costs
*Add foreclosure costs to estimates of LGD
*Add these after calculating other LGDs because these are paid by the bank so shouldn't affect other loans
replace mortlgdh = mortlgdh + hdebt*${foreclosure}*pd if LVR>${foreclosure_lvr}
replace mortlgdlmih = mortlgdlmih + hdebt*${foreclosure}*pd if LVR>${foreclosure_lvr}
replace othseclgdh = othseclgdh + nonhomepropertydebt*${foreclosure}*pd if nonhomeLVR>${foreclosure_lvr}
replace othseclgdlmih = othseclgdlmih + nonhomepropertydebt*${foreclosure}*pd if nonhomeLVR>${foreclosure_lvr}		
*All property LGD, excluding and including LMI
gen seclgdh = othseclgdh + mortlgdh
gen seclgdlmih = othseclgdlmih + mortlgdlmih
		
*Total lgd
gen totallgdh = mortlgdh + othseclgdh + othunseclgdh + cclgdh
label variable totallgdh "household total LGD - using property assets, excl LMI"

*Total lgd, taking into account LMI
gen totallgdlmih = mortlgdlmih + othseclgdlmih + othunseclgdh + cclgdh
label variable totallgdlmih "household total LGD - using property assets, incl LMI"

*****************************************************************************
*9. CALCULATE LGDs AT POPULATION LEVEL
*****************************************************************************	

*Calculate population LGDs (excluding LMI), total assets
*First mortgage
egen popmortlgd = total(mortlgd*hwt)
label variable popmortlgd "population mortgage LGD, using total assets"
*in percentage terms
gen popmortlgdp = popmortlgd/(popmortead)*100
label variable popmortlgdp "population mortgage LGD, % of vulnerable debt, using total assets"
*Other secured
egen popothseclgd = total(othseclgd*hwt)
label variable popothseclgd "population other secured LGD, using total assets"
*in percentage terms
gen popothseclgdp = popothseclgd/(popothsecead)*100
label variable popothseclgdp "population other secured LGD, % of vulnerable debt, using total assets"
*All secured
egen popseclgd = total(seclgd*hwt)
label variable popseclgd "population mortgage LGD, using total assets"
*in percentage terms
gen popseclgdp = popseclgd/(popsecead)*100
label variable popmortlgdp "population mortgage LGD, % of vulnerable debt, using total assets"
*Oth unsecured
egen popothunseclgd = total(othunseclgd*hwt)
label variable popothunseclgd "population other unsecured LGD, using total assets"
*in percentage terms
gen popothunseclgdp = popothunseclgd/(popothead)*100
label variable popothunseclgdp "population other unsecured LGD, % of vulnerable debt, using total assets"
*Credit card
egen popcclgd = total(cclgd*hwt)
label variable popcclgd "population credit card LGD, using total assets"
*in percentage terms
gen popcclgdp = popcclgd/(popccead)*100
label variable popcclgdp "population credit card LGD, % of vulnerable debt, using total assets"
*Total LGD
egen poptotallgd = total(totallgd*hwt)
label variable poptotallgd "population total LGD, using total assets"
*in percentage terms
gen poptotallgdp = poptotallgd/(popmortead + popothsecead + popothead + popccead)*100
label variable poptotallgdp "population total LGD, % of vulnerable debt, using total assets"

*Calculate population LGDs (excluding LMI), using only property assets
*First ortgage
egen popmortlgdh = total(mortlgdh*hwt)
label variable popmortlgdh "population mortgage LGD, using property assets"
*in percentage terms
gen popmortlgdhp = popmortlgdh/(popmortead)*100
label variable popmortlgdhp "population mortgage LGD, % of vulnerable debt, using property assets"
*Other secured
egen popothseclgdh = total(othseclgdh*hwt)
label variable popothseclgdh "population other secured LGD, using total assets"
*in percentage terms
gen popothseclgdhp = popothseclgdh/(popothsecead)*100
label variable popothseclgdhp "population other secured LGD, % of vulnerable debt, using property assets"
*All secured
egen popseclgdh = total(seclgdh*hwt)
label variable popseclgd "population mortgage LGD, using total assets"
*in percentage terms
gen popseclgdhp = popseclgdh/(popsecead)*100
label variable popmortlgdp "population mortgage LGD, % of vulnerable debt, using total assets"
*Other unsecured
egen popothunseclgdh = total(othunseclgdh*hwt)
label variable popothunseclgdh "population other unsecured LGD, using total assets"
*in percentage terms
gen popothunseclgdhp = popothunseclgdh/(popothead)*100
label variable popothunseclgdhp "population other unsecured LGD, % of vulnerable debt, using property assets"
*Credit card
egen popcclgdh = total(cclgdh*hwt)
label variable popcclgdh "population credit card LGD, using total assets"
*in percentage terms
gen popcclgdhp = popcclgdh/(popccead)*100
label variable popcclgdhp "population credit card LGD, % of vulnerable debt, using property assets"
*Total LGD
egen poptotallgdh = total(totallgdh*hwt)
label variable poptotallgdh "population total LGD, using property assets"
*in percentage terms
gen poptotallgdhp = poptotallgdh/(popmortead+ popothsecead + popothead + popccead)*100
label variable poptotallgdhp "population total LGD, % of vulnerable debt, using property assets"

*Calculate population LGDs (including LMI), total assets 
*LGDs are only different for property loans
*First mortgage
*Population first mortgage LGD taking into account all assets 
egen popmortlgdlmi = total(mortlgdlmi*hwt)
label variable popmortlgdlmi "population mortgage LGD, using total assets, including LMI"
*in percentage terms
gen popmortlgdlmip = popmortlgdlmi/(popmortead)*100
label variable popmortlgdlmip "population mortgage LGD, % of vulnerable debt, using total assets, including LMI"
*other secured
egen popothseclgdlmi = total(othseclgdlmi*hwt)
label variable popothseclgdlmi "population other secured LGD, using total assets, including LMI"
*in percentage terms
gen popothseclgdlmip = popothseclgdlmi/(popothsecead)*100
label variable popothseclgdlmip "population other secured LGD, % of vulnerable debt, using total assets, including LMI"
*secured 
egen popseclgdlmi = total(seclgdlmi*hwt)
label variable popseclgdlmi "population mortgage LGD, using total assets"
*in percentage terms
gen popseclgdlmip = popseclgdlmi/(popsecead)*100
label variable popseclgdlmip "population mortgage LGD, % of vulnerable debt, using total assets"
*Total LGD
egen poplgdlmi = total(totallgdlmi*hwt)
label variable poplgdlmi "population LGD, using total assets, including LMI"
*in percentage terms
gen poplgdlmip = poplgdlmi/(popmortead+ popothsecead + popothead + popccead)*100
label variable poplgdlmip "population LGD, % of vulnerable debt, using total assets, including LMI"

*Calculate population LGDs (including LMI), property assets
*LGDs are only different for property loans
*First mortgage
egen popmortlgdlmih = total(mortlgdlmih*hwt)
label variable popmortlgdlmih "population mortgage LGD, using property assets, including LMI"
*in percentage terms
gen popmortlgdlmihp = popmortlgdlmih/(popmortead)*100
label variable popmortlgdlmihp "population mortgage LGD, % of vulnerable debt, using property assets, including LMI"
*other secured
egen popothseclgdlmih = total(othseclgdlmih*hwt)
label variable popothseclgdlmih "population other secured LGD, using property assets, including LMI"
*in percentage terms
gen popothseclgdlmihp = popothseclgdlmih/(popothsecead)*100
label variable popothseclgdhp "population other secured LGD, % of vulnerable debt, using property assets, including LMI"
*secured 
egen popseclgdlmih = total(seclgdlmih*hwt)
label variable popseclgdlmih "population mortgage LGD, using property assets"
*in percentage terms
gen popseclgdlmihp = popseclgdlmih/(popsecead)*100
label variable popmortlgdhp "population mortgage LGD, % of vulnerable debt, using property assets"
*Total LGD
egen poplgdlmih = total(totallgdlmih*hwt)
label variable poplgdlmih "population LGD, using property assets, including LMI"
*in percentage terms
gen poplgdlmihp = poplgdlmih/(popmortead+ popothsecead + popothead + popccead)*100
label variable poplgdlmihp "population LGD, % of vulnerable debt, using property assets, including LMI"

*****************************************************************************
*10. CALCULATE PRE-STRESS SYSTEM LOSSES = EAD*LGD
*****************************************************************************			

*************
*TOTAL LOSSES 
*************

*EXCLUDING LMI

*Calculate total pre-stress system losses, in percentage terms, total assets, excluding LMI
gen systemloss = (poptoteadp/100)*(poptotallgdp/100)*100
label variable systemloss "pre-stress losses, % total debt, using total assets, excl LMI"

*Calculate total pre-stress system losses, in percentage terms, property assets only, excluding LMI
gen systemlossh = (poptoteadp/100)*(poptotallgdhp/100)*100
label variable systemlossh "pre-stress losses, % total debt, using property assets, excl LMI"

*INCLUDING LMI

*Calculate total pre-stress system losses, in percentage terms, total assets, including LMI
gen systemlosslmi = (poptoteadp/100)*(poplgdlmip/100)*100
label variable systemlosslmi "pre-stress losses, % total debt, using total assets, incl LMI"

*Calculate total pre-stress system losses, in percentage terms, property assets only, including LMI
gen systemlosslmih = (poptoteadp/100)*(poplgdlmihp/100)*100
label variable systemlosslmih "pre-stress losses, % total debt, using property assets, incl LMI"

****************
*LOSS COMPONENTS
****************

*EXCLUDING LMI, USING TOTAL ASSETS

*first mortgage losses taking into account all assets 
gen mortloss = (popmorteadp/100)*(popmortlgdp/100)*100
label variable mortloss "pre-stress mortgage losses, % total debt, using total assets, excl LMI"
*other secured losses taking into account all assets
gen othsecloss = (popothseceadp/100)*(popothseclgdp/100)*100
label variable othsecloss "pre-stress other secured losses, % total debt, using total assets, excl LMI"
*other unsecured losses taking into account all assets
gen othunsecloss = (popotheadp/100)*(popothunseclgdp/100)*100
label variable othunsecloss "pre-stress other unsecured losses, % total debt, using total assets, excl LMI"
*credit card losses taking into account all assets
gen ccloss = (popcceadp/100)*(popcclgdp/100)*100
label variable ccloss "pre-stress credit card losses, % total debt, using total assets, excl LMI"
* all property
gen secloss = (popseceadp/100)*(popseclgdp/100)*100
label variable secloss "pre-stress all property losses, % total debt, using total assets, excl LMI"
 
*EXCLUDING LMI, USING ONLY PROPERTY ASSETS

*first mortgage losses taking into account all assets 
gen mortlossh = (popmorteadp/100)*(popmortlgdhp/100)*100
label variable mortlossh "pre-stress mortgage losses, % total debt, using property assets, excl LMI"
*other secured losses taking into account all assets
gen othseclossh = (popothseceadp/100)*(popothseclgdhp/100)*100
label variable othseclossh "pre-stress other secured losses, % total debt, using property assets, excl LMI"
*other unsecured losses taking into account all assets
gen othunseclossh = (popotheadp/100)*(popothunseclgdhp/100)*100
label variable othunseclossh "pre-stress other unsecured losses, % total debt, using property assets, excl LMI"
*credit card losses taking into account all assets
gen cclossh = (popcceadp/100)*(popcclgdhp/100)*100
label variable cclossh "pre-stress credit card losses, % total debt, using property assets, excl LMI"
* all property
gen seclossh = (popseceadp/100)*(popseclgdhp/100)*100
label variable seclossh "pre-stress all property losses, % total debt, using total assets, excl LMI"

*INCLUDING LMI

*first mortgage losses taking into account all assets, including LMI 
gen mortlosslmi = (popmorteadp/100)*(popmortlgdlmip/100)*100
label variable mortlosslmi "pre-stress mortgage losses, % total debt, using total assets, incl LMI"

*first mortgage losses taking into account only property assets, including LMI 
gen mortlosslmih = (popmorteadp/100)*(popmortlgdlmihp/100)*100
label variable mortlosslmih "pre-stress mortgage losses, % total debt, using property assets, incl LMI"

* total property variables
gen seclosslmi = (popseceadp/100)*(popseclgdlmip/100)*100
label variable seclosslmi "pre-stress total prop , % total debt, using total assets, incl LMI"

gen seclosslmih = (popseceadp/100)*(popseclgdlmihp/100)*100 
label variable seclosslmih "pre-stress total prop , % total debt, using property assets, incl LMI"

*****************************************************************************
*11. GENERATE PRE-STRESS SUMMARY STATS
*****************************************************************************

* EAD household levels
n tabstat hdebt hinvdebt othpropdebt ccdebt nonpropbusccdebt [aw = hwt], stats (sum n) save
	matrix results_hlEAD = r(StatTotal)
	
* EAD summary stats (population levels)
n tabstat poptoteadp popmorteadp popothseceadp popotheadp popcceadp, stats (mean) save
	matrix results_EAD = r(StatTotal)

* LGD summary stats, using total assets
n tabstat poptotallgdp poplgdlmip popmortlgdp popmortlgdlmip  popothseclgdp  popothunseclgdp  popcclgdp, stats (mean) save
	matrix results_LGD_totAssets = r(StatTotal)

* LGD summary stats, using property assets
n tabstat poptotallgdhp poplgdlmihp popmortlgdhp popmortlgdlmihp  popothseclgdhp  popothunseclgdhp  popcclgdhp, stats (mean) save
	matrix results_LGD_houseAssets = r(StatTotal)

* Income and debt distributions
n tabstat hhtotaldis hdebt hinvdebt othpropdebt propertydebt, stats(p5 p10 p25 p50 p75 p90 p95 n) save
	matrix results_IncomeDebtDistributions = r(StatTotal)

*Losses on individual asset classes (total assets)
n tabstat mortloss mortlosslmi othsecloss othunsecloss ccloss, stats(mean) save
	matrix results_DAR_totAssets = r(StatTotal)
	
*Losses on individual asset classes (property assets only)
n tabstat mortlossh mortlosslmih othseclossh othunseclossh cclossh, stats(mean) save
	matrix results_DAR_DAR_HouseAssets = r(StatTotal)
		
* Financial margin
	gen fm1 = 0 if finmarg < .
	replace fm1 = 1 if finmarg < 0
	n 	tabstat fm1 [aw = hwt], save 

* Unemployment rate
forvalues i=1/6 {
	gen unempl`i' = 0
	replace unempl`i' = 1 if lfscp`i' == 2
	gen attlf`i' = 0
	replace attlf`i' = 1 if lfscp1 < 3
	}
gen unem = (unempl1*pwt1 + unempl2*pwt2 + unempl3*pwt3 + unempl4*pwt4 + unempl5*pwt5 + unempl6*pwt6) / hwt //unweighted 
egen unempl = total(unem*hwt)
gen hh_attlf = (attlf1*pwt1 + attlf2*pwt2 + attlf3*pwt3 + attlf4*pwt4 + attlf5*pwt5 + attlf6*pwt6) / hwt //unweighted	
egen attlf = total(hh_attlf*hwt)
gen unrate = unempl/attlf
label variable unrate "unemployment rate, weighted"	

*****************************************************************************
*12. GENERATE STRESSED VARIABLES
*****************************************************************************				
*Generate asset price stress
*Equities
gen vsharch_ = vsharch*(1 - ${stress_equity})
*Trusts stress	
gen vtrtch_ = vtrtch*(1 - ${stress_trust})
*Superannuation (decreasing in age)
gen vsuptot_ = vsuptot*(1 - ${stress_super}*(1-age/100)) 
*First property
gen hvaluech_ = hvaluech*(1 - ${stress_housing}) 
*Other property 
gen vrprch_ = vrprch*(1 - ${stress_housing})
gen vnrprch_ = vnrprch*(1 - ${stress_nonres})
*Household contents and other assets
gen vcontch_ = vcontch*(1 - ${stress_other})
gen votassch_ = votassch*(1 - ${stress_other})
*Businesses  
gen vibusch_ = vibusch*(1 - ${stress_other})
gen vubusch_ = vubusch*(1 - ${stress_other})
gen vsipch_ = vsipch*(1 - ${stress_other})
*Vehicles
gen vvehich_ = vvehich*(1 - ${stress_other})	
*Debentures and bonds 
gen vdebch_ = vdebch*1 
*Other financial investments 
gen vinvotch_ = vinvotch*1 
*Other loans to non-household persons
gen vplnch_ = vplnch*1 
*Bank account 
gen vfinch_ = vfinch*1
*Offset account
gen voftch_ = voftch*1

*ASSETS 
//Assets
g asset_sp_ = vsuptot_ //super
g asset_sp_lin_ = asset_sp_*share_super_asset // linear super	
g asset_hs_ = hvaluech_+vrprch_ // housing
g asset_eq_ = vsharch_ //equity
g asset_nres_ = vnrprch_ // non-res	
g asset_tr_ = vtrtch_ // trusts
g asset_oth_fin_ = vfinch_+voftch_+vdebch_+vplnch_+vinvotch_ // other financial assets
g asset_oth_ = vsipch_+vibusch_+vubusch_+vcontch_+vvehich_+votassch_ // other assets

g asset_to_ = asset_sp_+asset_hs_+asset_eq_+asset_nres_+asset_tr_+asset_oth_fin_+asset_oth_
label variable asset_to_ "total assets, all super, post-stress"	
g asset_to_linsup_ = asset_sp_lin_+asset_hs_+asset_eq_+asset_nres_+asset_tr_+asset_oth_fin_+asset_oth_
label variable asset_to_linsup "total assets, linear super treatment, post-stress"		

gen propasset_ = asset_hs_ + asset_nres_ // Property assets, post-stress	
gen liquid_financial_assets_ = vfinch_ + voftch_ + vsharch_ + vdebch_ // liquid assets, post-stress

*WEALTH		
gen wealth_ = asset_to_linsup_ - hhdebt 
label variable wealth_ "Household wealth (with super adjustment and excluding student debt)"				

*Define hand-to-mouth individuals, post-shock
*Define liquid and illiquid wealth
gen liqnworth_ = vfinch_ + voftch_ + vsharch_ + vdebch_ - ccdebt	
gen liqnw_adj_ = liqnworth_ + redraw 

//HTM according to KVW (2014)
gen handtomouth_ = 0 if pay/2<liqnworth_ & liqnworth_>=0
replace handtomouth_=1 if ((pay/2-credit_limit)>=liqnworth_ & liqnworth_<=0)
replace handtomouth_=0 if ((pay/2-credit_limit)<liqnworth_ & liqnworth_<=0)
replace handtomouth_=1 if (pay/2>=liqnworth_ & liqnworth_>=0)			

//HTM according to KVW (2014) adjusted for prepayment buffers
gen htm_adj_ = 0 if pay/2<liqnw_adj_ & liqnw_adj_>=0
replace htm_adj_=1 if ((pay/2-credit_limit)>=liqnw_adj_ & liqnw_adj_<=0)
replace htm_adj_=0 if ((pay/2-credit_limit)<liqnw_adj_ & liqnw_adj_<=0)
replace htm_adj_=1 if (pay/2>=liqnw_adj_ & liqnw_adj_>=0)			

if ${htm_with_buffer} == 1 /* include buffers */ {    		
	replace handtomouth_ = htm_adj_
	}		

*****************************************************************************
*13. GENERATE MINIMUM MORTGAGE REPAYMENT VARIABLE, POST-STRESS
*****************************************************************************	

*New mortgage rate
gen mortir_ = mortir + ${stress_cashrate_change}
label variable mortir_ "mortgage interest rate"
	
*Required mortgage repayments for all property loans after cash rate change
gen required_ = (scheduled_balance * mortir_/12)/(1 - (1 + (mortir_/12))^-(homeloan_yr_to_maturity*12))*12

* Adjust required for interest-only borrowers		
replace required_ = toto_home * mortir_ if interest_only==1 
	*Allow offset balances to reduce minimum required repayment	
if ${offsets_reduce_interest} == 1 {		
	replace required_ = (toto_home-voftch) * mortir_ if interest_only==1 		
	}			
replace required_ = 0 if required_<0 // ensure required is above 0
	
*Weekly minimum required repayment	
gen requiredwkhome_ = required_ / 52	

* Change other repayment variables by the change from credit foncier
gen repay_change = required_ / required
tabstat repay_change [aw = hwt], s(p50) save 
mat temp = r(StatTotal)
gen med_repay_change = temp[1,1]
mat drop temp	

gen totwkrep_home_ = totwkrep_home * med_repay_change
gen trpaych_ = trpaych * med_repay_change 
*Change individual repay change if available 
replace totwkrep_home_ = totwkrep_home * repay_change if repay_change!=.
replace trpaych_ = trpaych * repay_change if repay_change!=.	

*Replace minimum repayment with actual repayment for some households:	
*For households without repayments in household data but repayments in loan-level data:
*...and required repayment is more than actual
gen flag5_=0			
replace flag5_=1 if totwkrep_home_ < requiredwkhome_ & requiredwkhome_!=. & flag1==1 
*...and required repayment is undefined
gen flag6_=0
replace flag6_=1 if requiredwkhome_==. & flag1==1

*For households with repayments in household data:
*...where actual repayment < minimum repayment
gen flag7_=0
replace flag7_=1 if trpaych_ < requiredwkhome_ & requiredwkhome_!=. & flag1==0 & trpaych_ > 0
*...(that are positive) where minimum repayment is undefined
gen flag8_=0
replace flag8_=1 if requiredwkhome_==. & trpaych_>0 & flag1==0
*...(that are 0) where minimum repayment is undefined
gen flag9_=0
replace flag9_=1 if requiredwkhome_==. & trpaych_==0 & flag1==0

replace requiredwkhome_ = totwkrep_home_ if flag6_==1 | flag5_==1 		
replace requiredwkhome_ = trpaych_ if flag7_==1 | flag8_ ==1 | flag9_ == 1

*Generate an estimate of annual credit card interest repayments 
gen ccdebtrepay_ = ccdebt*(${ccrate}+${ccprincipalrepay}) 	

*Generate estimate of repayment on non-property, non-business, non-cc debt 
gen opdebtrepay_ = nonpropbusccdebt*(${oprate}+${stress_cashrate_change}+${opdebtprincipalrepay})

*Generate annual debt servicing 
gen debtserv_ = requiredwkhome_*52 + ccdebtrepay_ + opdebtrepay_
label variable debtserv_ "debt servicing post-stress"
		
*****************************************************************************
*14. GENERATE MPCs of WEALTH AND INCOME
*****************************************************************************			

*Calculate DTI
g hDTI = (hhdebt/(hhtotaldis)) 
replace hDTI = 0 if hhtotaldis== 0 & hhdebt== 0      
replace hDTI = 0 if hhdebt < 0 
summarize hDTI, d
replace hDTI = r(p95) if hDTI < 0 | hDTI==.	 // replace with 95th percentile if hDTI<0
*hDTI quintiles
xtile DTI_x = hDTI[aw=hwt] if hhdebt>0, nquantiles(4) 
replace DTI_x = DTI_x +1	
replace DTI_x = 1 if hhdebt<=0 // to ensure all households with no debt are in the same quintile	
replace DTI_x = 3 if hhtotaldis == 0 & hhdebt == 0	| DTI_x==.

*Calculate DTA 
gen hDTA = hhdebt/asset_to_linsup
replace hDTA = 0 if asset_to_linsup== 0 & hhdebt== 0
replace hDTA = 0 if hhdebt<0
summarize hDTA, d
replace hDTA = r(p95) if hDTA < 0 | hDTA==.	 // replace with 95th percentile if hDTA<0		
*DTA quintiles			
xtile DTA_x = hDTA[aw=hwt] if hhdebt>0, nquantiles(4) 
replace DTA_x = DTA_x +1	
replace DTA_x = 1 if hhdebt<=0  // to ensure all households with no debt are in the same quintile
replace DTA_x = 3 if hhtotaldis == 0 & hhdebt == 0	| DTA_x==.	

*************
*	MPCs of wealth
*************	
*Set initial values
gen mpcw=.
gen mpcw_a_target=.
gen mpcw_uni=0

*Set up temporary variables for loop
gen mpcw_uni_old=.
gen diff_old = .
gen diff =.

*Set MPC for wealth based on method
if ${mpcw_type} == 1 /* use average */ {    			
	replace mpcw = ${mpcw_av}
	}

if ${mpcw_type} == 2 /* use DTA formula */ {    			
	*Choose constant that minimises difference between calculated and target average MPC
	forvalues y = $mpcw_a_low($sensitivity_w)$mpcw_a_high {
		replace mpcw_uni_old = mpcw_uni // keep last period constant
		replace diff_old = mpcw_uni_old - ${mpcw_av} // determine difference between average MPC from LAST period and the target average MPC
		replace diff_old = abs(diff_old) // take absolute value for comparison
		global mpcw_a = `y'
		replace mpcw = ${mpcw_a} + ${mpcw_b}*hDTA 
		replace mpcw = ${min_mpcw} if mpcw<${min_mpcw} 
		replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.	
		
		summarize mpcw [aw=hwt]
		replace mpcw_uni = r(mean) // calculate new average MPC

		replace diff = mpcw_uni - ${mpcw_av} // determine difference between calculated average MPC from THIS period and the target average MPC
		replace diff = abs(diff) // take absolute value for comparison
		replace mpcw_a_target = ${mpcw_a} if diff < diff_old // compare with last periods, and replace with THIS periods if it is closer to target average MPC 	
		}
	*Run desired model 
	replace mpcw = mpcw_a_target + ${mpcw_b}*hDTA 
	replace mpcw = ${min_mpcw} if mpcw<${min_mpcw}
	replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.		
	}
if ${mpcw_type} == 3 /* use age formula */ {    			
	*Choose constant that minimises difference between calculated and target average MPC
	forvalues y = $mpcw_age_a_low($sensitivity_w_age)$mpcw_age_a_high {
		replace mpcw_uni_old = mpcw_uni // keep last period constant
		replace diff_old = mpcw_uni_old - ${mpcw_av} // determine difference between calculated average MPC from LAST period and the target average MPC
		replace diff_old = abs(diff_old) // takes absolute value for comparison
		global mpcw_age_a = `y'
		replace mpcw = ${mpcw_age_a} + ${mpcw_age_b}*age // set new MPC with new constant
		replace mpcw = ${min_mpcw} if mpcw<${min_mpcw} 
		replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.	
		
		summarize mpcw [aw=hwt]
		replace mpcw_uni = r(mean) // calculate new average MPC

		replace diff = mpcw_uni - ${mpcw_av} // determine difference between calculated average MPC from THIS period and the target average MPC
		replace diff = abs(diff) // takes absolute value for comparison
		replace mpcw_a_target = ${mpcw_age_a} if diff < diff_old // compare with last periods, and replace with THIS periods if it is closer to target average MPC 	
		}
	
	*Run desired model 
	replace mpcw = mpcw_a_target + ${mpcw_age_b}*age 
	replace mpcw = ${min_mpcw} if mpcw<${min_mpcw}
	replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.		
	}		

if ${mpcw_type} == 4 /* use HtM */ {    					
replace mpcw_a_target=0
	*Choose constant that minimises difference between calculated and target average MPC
	forvalues y = $mpcw_inc_a_low($sensitivity_w_age)$mpcw_inc_a_high {
		replace mpcw_uni_old = mpcw_uni // keep last period constant
		replace diff_old = mpcw_uni_old - ${mpcw_av} // determine difference between calculated average MPC from LAST period and the target average MPC
		replace diff_old = abs(diff_old) // takes absolute value for comparison
		global mpcw_a = `y'
		replace mpcw = ${mpcw_a}*${mpcw_htm} if handtomouth_ ==1
		replace mpcw = ${mpcw_a} if handtomouth_==0 			
		replace mpcw = ${min_mpcw} if mpcw<${min_mpcw} 
		replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.	

		summarize mpcw [aw=hwt]
		replace mpcw_uni = r(mean) // calculate new average MPC

		replace diff = mpcw_uni - ${mpcw_av} // determine difference between calculated average MPC from THIS period and the target average MPC
		replace diff = abs(diff) // takes absolute value for comparison
		replace mpcw_a_target = ${mpcw_a} if diff < diff_old // compare with last periods, and replace with THIS periods if it is closer to target average MPC 	
		}
		
	*Run desired model 
	replace mpcw = mpcw_a_target*${mpcw_htm} if handtomouth_ ==1
	replace mpcw = mpcw_a_target if handtomouth_ ==0					
	replace mpcw = ${min_mpcw} if mpcw<${min_mpcw}
	replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.		
	}		

if ${mpcw_type} == 5 /* use DTI formula */ {    					
	*Choose constant that minimises difference between calculated and target average MPC
	forvalues y = $mpcw_a_low($sensitivity_w)$mpcw_a_high {
		replace mpcw_uni_old = mpcw_uni // keep last period constant
		replace diff_old = mpcw_uni_old - ${mpcw_av} // determine difference between calculated average MPC from LAST period and the target average MPC
		replace diff_old = abs(diff_old) // takes absolute value for comparison
		global mpcw_a = `y'
		replace mpcw = ${mpcw_a} + ${mpcw_dti_b}*hDTI 
		replace mpcw = ${min_mpcw} if mpcw<${min_mpcw} 
		replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.	

		summarize mpcw [aw=hwt]
		replace mpcw_uni = r(mean) // calculate new average MPC

		replace diff = mpcw_uni - ${mpcw_av} // determine difference between calculated average MPC from THIS period and the target average MPC
		replace diff = abs(diff) // takes absolute value for comparison
		replace mpcw_a_target = ${mpcw_a} if diff < diff_old // compare with last periods, and replace with THIS periods if it is closer to target average MPC 	
		}
	*Run desired model 
	replace mpcw = mpcw_a_target + ${mpcw_dti_b}*hDTI 
	replace mpcw = ${min_mpcw} if mpcw<${min_mpcw}
	replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.		
	}

if ${mpcw_type} == 6 /* use income */ {    					
replace mpcw_a_target=0
	*Choose constant that minimises difference between calculated and target average MPC
	forvalues y = $mpcw_inc_a_low($sensitivity_w_age)$mpcw_inc_a_high {
		replace mpcw_uni_old = mpcw_uni // keep last period constant
		replace diff_old = mpcw_uni_old - ${mpcw_av} // determine difference between calculated average MPC from LAST period and the target average MPC
		replace diff_old = abs(diff_old) // takes absolute value for comparison
		global mpcw_a = `y'
		replace mpcw = ${mpcw_a} + ${mpcw_inc_b}*hhtotaldis 
		replace mpcw = ${min_mpcw} if mpcw<${min_mpcw} 
		replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.	

		summarize mpcw [aw=hwt]
		replace mpcw_uni = r(mean) // calculate new average MPC

		replace diff = mpcw_uni - ${mpcw_av} // determine difference between calculated average MPC from THIS period and the target average MPC
		replace diff = abs(diff) // takes absolute value for comparison
		replace mpcw_a_target = ${mpcw_a} if diff < diff_old // compare with last periods, and replace with THIS periods if it is closer to target average MPC 	
		}
	
	*Run desired model 
	replace mpcw = mpcw_a_target + ${mpcw_inc_b}*hhtotaldis 
	replace mpcw = ${min_mpcw} if mpcw<${min_mpcw}
	replace mpcw = ${max_mpcw} if mpcw>${max_mpcw} & mpcw!=.		
}		
		
drop diff diff_old mpcw_uni_old
*************
* MPCs of income
*************	
*Set initial values
gen mpcy=.
gen mpcy_a_target=.	
gen mpcy_uni=0

*Set up temporary variables for loop			
gen mpcy_uni_old=.	
gen diff_old = .
gen diff =.	

*Set MPC for income based on method	
if ${mpcy_type} == 1 /* use average */ {    			
	replace mpcy = ${mpcy_av}			
	}
		
if ${mpcy_type} == 2 /* use DTI formula */ {    			
	*Choose constant that minimises difference between calculated and target average MPC
	forvalues y = $mpcy_a_low($sensitivity_y)$mpcy_a_high {
		replace mpcy_uni_old = mpcy_uni // keep last period constant
		replace diff_old = mpcy_uni_old - ${mpcy_av} // detertmine difference between calculated average MPC from LAST period and the target average MPC
		replace diff_old = abs(diff_old) // takes absolute value for comparison
		global mpcy_a = `y'
		replace mpcy = ${mpcy_a} + ${mpcy_b}*hDTI // set new MPC with new constant
		replace mpcy = ${min_mpcy} if mpcy<${min_mpcy} 
		replace mpcy = ${max_mpcy} if mpcy>${max_mpcy} & mpcy!=.	
		
		summarize mpcy [aw=hwt]
		replace mpcy_uni = r(mean) // calculate new average MPC
		replace diff = mpcy_uni - ${mpcy_av} // determine difference between calculated average MPC from THIS period and the target average MPC
		replace diff = abs(diff) // takes absolute value for comparison
		replace mpcy_a_target = ${mpcy_a} if diff < diff_old // compare with last periods, and replace with THIS periods if it is closer to target average MPC 	
		}
	
	*Run desired model 
	replace mpcy = mpcy_a_target + ${mpcy_b}*hDTI
	replace mpcy = ${min_mpcy} if mpcy<${min_mpcy}
	replace mpcy = ${max_mpcy} if mpcy>${max_mpcy} & mpcy!=.		
	}		
if ${mpcy_type} == 4 /* use hTm */ {    			
	*Choose constant that minimises difference between calculated and target average MPC
	forvalues y = $mpcy_a_htm_low($sensitivity_y)$mpcy_a_htm_high {
		replace mpcy_uni_old = mpcy_uni // keep last period constant
		replace diff_old = mpcy_uni_old - ${mpcy_av} // detertmine difference between calculated average MPC from LAST period and the target average MPC
		replace diff_old = abs(diff_old) // takes absolute value for comparison
		global mpcy_htm_a = `y'
		replace mpcy = 1 if handtomouth_ ==1
		replace mpcy = ${mpcy_htm_a} if handtomouth_==0 
		replace mpcy = ${min_mpcy} if mpcy<${min_mpcy} 
		replace mpcy = ${max_mpcy} if mpcy>${max_mpcy} & mpcy!=.	
		
		summarize mpcy [aw=hwt]
		replace mpcy_uni = r(mean) // calculate new average MPC
		replace diff = mpcy_uni - ${mpcy_av} // determine difference between calculated average MPC from THIS period and the target average MPC
		replace diff = abs(diff) // takes absolute value for comparison
		replace mpcy_a_target = ${mpcy_htm_a} if diff < diff_old // compare with last periods, and replace with THIS periods if it is closer to target average MPC 	
		}
	
	*Run desired model 
	replace mpcy = 1 if handtomouth_ ==1
	replace mpcy = mpcy_a_target if handtomouth_ ==0						
	replace mpcy = ${min_mpcy} if mpcy<${min_mpcy}
	replace mpcy = ${max_mpcy} if mpcy>${max_mpcy} & mpcy!=.		
	}	
 drop diff diff_old mpcy_uni_old	
*************************************************
*	Weighted average MPCs of income and wealth  *
*************************************************
*Weighted constant mpcw and mpcy to check
summarize mpcw [aw=hwt]
replace mpcw_uni = r(mean)
summarize mpcy [aw=hwt]	
replace mpcy_uni = r(mean)

save "Results\HH stress test.dta", replace	
	
*****************************************************************************
*15. CARRY OUT MONTE CARLO SIMULATION FOR UNEMPLOYMENT
*****************************************************************************	

*Use Monte Carlo technique to simulate workers losing their job 
program stress_test
drop _all
use "Results\HH stress test.dta", clear

forvalues i=1/6 {
	*Rename post-stress probability of unemployment
	rename prunempl_`i' prunempl`i'_
	*INVESTMENT INCOME
	gen inv_change`i' = 0
	replace inv_change`i'  = -inv_inc`i'*${stress_invinc} if inv_inc`i'>0
	*OTHER INCOME
	gen oth_change`i' = 0		
	*EMPLOYMENT INCOME
	*Generate random numbers over the interval [0,1) and assign unemployment
	gen x`i' = runiform() if lfscp`i'==1
	*Calculate lost income if become unemployed
	gen unemp_change`i' = 0
	replace unemp_change`i' = -emp_inc`i' if prunempl`i'_ >= x`i' & lfscp`i'==1 & emp_inc`i'>0
	*Calculate regular unemployment benefits
	gen unben_change`i' = 0
	replace unben_change`i' = unben if prunempl`i'_ >= x`i' & lfscp`i'==1 & emp_inc`i'>0
	*Reduce bonus/overtime income for employed
	gen bonus_change`i' = 0
	replace bonus_change`i' = -emp_extras`i'*${bonuscut} if prunempl`i'_ < x`i' & lfscp`i'==1 & emp_extras`i'>0
	*Reduce hours income for employed
	gen hours_change`i' = 0
	replace hours_change`i' = -(emp_inc`i' - emp_extras`i') * ${hours_fall} if prunempl`i'_ < x`i' & lfscp`i'==1 & emp_inc`i'- emp_extras`i'>0 	
	*Update employment income (and unemployment benefits)
	gen emp_change`i' = 0
	replace emp_change`i' = unemp_change`i' + bonus_change`i' + hours_change`i' + unben_change`i'

	*Update taxable income
	gen tax_inc`i'_ = emp_inc`i' + inv_inc`i' + oth_inc`i' + emp_change`i' + inv_change`i' + oth_change`i'
	
	*Calculate household disposable income after stress
	*Apply income tax tables from https://www.ato.gov.au/Rates/Individual-income-tax-for-prior-years/	
	gen disp_inc`i'_ = disp_inc`i' 
	if ${year} == 2018 {	
		replace disp_inc`i'_ = tax_inc`i'_ 
		replace disp_inc`i'_ = tax_inc`i'_ - 0.19*(tax_inc`i'_-18200) if tax_inc`i'_ > 18200 & tax_inc`i'_!=.
		replace disp_inc`i'_ = tax_inc`i'_ - 3572 - 0.325*(tax_inc`i'_-37000) if tax_inc`i'_ > 37000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 19822 - 0.37*(tax_inc`i'_-87000) if tax_inc`i'_ > 87000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 54232 - 0.45*(tax_inc`i'_-180000) if tax_inc`i'_ > 180000 & tax_inc`i'_!=. 
		}	
	if ${year} == 2016 {	
		replace disp_inc`i'_ = tax_inc`i'_ 
		replace disp_inc`i'_ = tax_inc`i'_ - 0.19*(tax_inc`i'_-18200) if tax_inc`i'_ > 18200 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 3572 - 0.325*(tax_inc`i'_-37000) if tax_inc`i'_ > 37000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 17547 - 0.37*(tax_inc`i'_-80000) if tax_inc`i'_ > 80000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 54547 - 0.45*(tax_inc`i'_-180000) if tax_inc`i'_ > 180000 & tax_inc`i'_!=. 		
		}	
	if ${year} == 2010 {	
		replace disp_inc`i'_ = tax_inc`i'_ 
		replace disp_inc`i'_ = tax_inc`i'_ - 0.15*(tax_inc`i'_-6000) if tax_inc`i'_ > 6000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 4350 - 0.3*(tax_inc`i'_-35000) if tax_inc`i'_ > 35000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 17850 - 0.38*(tax_inc`i'_-80000) if tax_inc`i'_ > 80000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 55850 - 0.45*(tax_inc`i'_-180000) if tax_inc`i'_ > 180000 & tax_inc`i'_!=. 
		}	
	if ${year} == 2004 {	
		replace disp_inc`i'_ = tax_inc`i'_ 
		replace disp_inc`i'_ = tax_inc`i'_ - 0.17*(tax_inc`i'_-6000) if tax_inc`i'_ > 6000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 2652 - 0.3*(tax_inc`i'_-21600) if tax_inc`i'_ > 21600 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 11772 - 0.42*(tax_inc`i'_-52000) if tax_inc`i'_ > 52000 & tax_inc`i'_!=. 
		replace disp_inc`i'_ = tax_inc`i'_ - 16182 - 0.47*(tax_inc`i'_-62500) if tax_inc`i'_ > 62500 & tax_inc`i'_!=. 
		}

	*Unemployed
	gen unempl`i'_ = 0
	replace unempl`i'_ = 1 if lfscp`i' == 2
	replace unempl`i'_ = 1 if prunempl`i'_ >= x`i' & lfscp`i'==1 	
	}
		
* Calculate new disposable income as REPORTED pre-stress disposable income minus difference in person-level DERIVED disposable income pre- and post-stress. 
* Using this method means that any error in the DERIVED method of disposable income (due to incorrect tax treatment) is applied to both pre- and post-stress.
gen hhtotaldis_ = hhtotaldis
replace hhtotaldis_ = hhtotaldis - (disp_inc1-disp_inc1_) - (disp_inc2-disp_inc2_) - (disp_inc3-disp_inc3_) - (disp_inc4-disp_inc4_) - (disp_inc5-disp_inc5_) - (disp_inc6-disp_inc6_)

*Unemployment rate
gen unem_ = (unempl1_*pwt1 + unempl2_*pwt2 + unempl3_*pwt3 + unempl4_*pwt4 + unempl5_*pwt5 + unempl6_*pwt6) / hwt //unweighted HH
egen unempl_ = total(unem_*hwt)
gen unrate_ = unempl_/attlf
label variable unrate_ "unemployment rate, HH-weighted, post-stress"

gen unem_diff = unem_ - unem // unweighted

*****************************************************************************
*16. GENERATE PDs, POST-STRESS
*****************************************************************************	

*Generate  financial margin 
gen finmarg_ = hhtotaldis_ - bascon - debtserv_ - (wkrentch + netir)*52	
label variable finmarg_ "stressed financial margin"
	
* include asset measures into financial margins			
if ${fm_switch} == 1 {    // liquid assets
	replace finmarg_ = finmarg_ + (liquid_financial_assets_+redraw)/${stress_length}	
	}
	
*Calculate PD from financial margin
gen pd_ = 1 if finmarg_ < 0
replace pd_ = 0 if finmarg_ >= 0
replace pd_ =. if finmarg_==.

*Calculate population PD
egen pdp_numerator_ = mean(pd_*100*hwt)
egen pdp_denominator_ = mean(hwt)
gen pdp_ = pdp_numerator_ / pdp_denominator_
drop pdp_numerator_ pdp_denominator_

*****************************************************************************
*17. GENERATE EADs, POST-STRESS
*****************************************************************************	

*****************************
* INDIVIDUAL HOUSEHOLD LEVEL 
*****************************

*Generate exposure at default (EAD) for types of debt for each household
*Total secured EAD
gen secead_ = propertydebt*pd_ 
label variable secead_ "stressed individual EAD secured"
*First mortgage EAD	
gen mortead_ = hdebt*pd_ 
label variable mortead_ "stressed individual EAD first mortgage"
*Other secured EAD	
gen othsecead_ = nonhomepropertydebt*pd_ 
label variable othsecead_ "stressed individual EAD other secured"
*Credit card EAD	
gen ccead_ = ccdebt*pd_
label variable ccead_ "stressed individual EAD credit card"
*other unsecured EAD	
gen othead_ = nonpropbusccdebt*pd_
label variable othead_ "stressed individual EAD other unsecured"	

*****************************
* POPULATION LEVEL
*****************************
	
*sum together household EAD's to give system EAD (need to population weight the data) i.e. each household will have this population value for this variable. 
*total secured
egen popsecead_ = total(secead_*hwt), missing
label variable popsecead_ "stressed population EAD total secured"
*first mortgage
egen popmortead_ = total(mortead_*hwt), missing
label variable popmortead_ "stressed population EAD first mortgage"
*other secured
egen popothsecead_ = total(othsecead_*hwt), missing
label variable popsecead_ "stressed population EAD other secured"
*credit card
egen popccead_ = total(ccead_*hwt), missing
label variable popccead_ "stressed population EAD credit card"
*other unsecured
egen popothead_ = total(othead_*hwt), missing
label variable popothead_ "stressed population EAD other unsecured"

*Calculation EADs as a percentage of debt, i.e. the percentage of debt that belongs to vunerable households
gen poptoteadp_ = (popothead_ + popsecead_ + popccead_)/(popothdebt + popsecdebt + popccdebt)*100   
label variable poptoteadp_ "stressed population EAD - % of total debt"
gen popseceadp_ = popsecead_/popsecdebt*100
label variable popseceadp_ "stressed population total secured EAD, % of total secured debt"
gen popmorteadp_ = popmortead_/popmortdebt*100
label variable popmorteadp_ "stressed population mortgage EAD, % of total mortgage debt"
gen popothseceadp_ = popothsecead_/popothsecdebt*100
label variable popothseceadp_ "stressed population other unsecured EAD, % of total secured debt"
gen popcceadp_ = popccead_/popccdebt*100
label variable popcceadp_ "stressed population credit card EAD, % of total credit card debt"
gen popotheadp_ = popothead_/popothdebt*100
label variable popotheadp_ "stressed population other unsecured EAD, % of total other unsecured debt"

*****************************************************************************
*18. GENERATE NET DEBT VARIABLES POST-STRESS
*****************************************************************************		

*Net debt taking into account all assets
gen netdebt_ = hhdebt - asset_to_linsup_
replace netdebt_ = 0 if netdebt_<0
label variable netdebt_ "net debt, using total assets, post-stress"

*Net debt taking into account only property assets
gen netdebth_ = hhdebt - propasset_
replace netdebth_ = 0 if netdebth_<0
label variable netdebth_ "net debt, using property assets, post-stress"		

*****************************************************************************
*19. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL POST-STRESS, USING TOTAL ASSETS
*****************************************************************************	

*****************************
* CALCULATE NEW LVR FOR FORECLOSURE COSTS
*****************************

*Current LVR estimate (of dwelling)
gen LVR_=(hdebt)/hvaluech_*100
label variable LVR_ "Current LVR, first mortgage, post-stress"
replace LVR_ = 0  if hvaluech_ == 0 | LVR_==.

*Current LVR estimate for all property
gen LVRproperty_ = propertydebt/propasset_*100
label variable LVRproperty_ "Current LVR, all property, post-stress"
replace LVRproperty_ = 0  if propasset_ == 0 | LVRproperty_==.

*Current LVR estimate for all non-home property
gen nonhomeLVR_ = nonhomepropertydebt/(vrprch_+vnrprch_)*100
label variable nonhomeLVR_ "Current LVR, non-home property, post-stress"
replace nonhomeLVR_ = 0  if vrprch_+vnrprch_ == 0 | nonhomeLVR_==.

*****************************
* EXCLUDING LMI
*****************************

gen lgd_ = pd_*netdebt_
replace lgd_ = pd_*netdebt_ + (hdebt+nonhomepropertydebt)*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}
label variable lgd_ "household total LGD - using total assets, excl LMI, post-stress"

*****************************
* INCLUDING LMI
*****************************

*Generate first mortgage LGD
gen mortlgd_ = 0
replace mortlgd_ = (hdebt - asset_to_linsup_)*pd_
replace mortlgd_ = 0 if mortlgd_ < 0  // if assets>debt, LGD=0  
label variable mortlgd_ "stressed mortgage LGD, using total assets, excl LMI"
	
*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio
gen mortlgdlmi_ = mortlgd_
replace mortlgdlmi_ = mortlgd_*(1-poratio_)*pd_ if LVR>80 
label variable mortlgdlmi_ "stressed mortgage LGD, using total assets, including LMI"

*generate other secured LGD
gen othseclgd_ = 0
replace othseclgd_ = (nonhomepropertydebt - (asset_to_linsup_ - hdebt))*pd_
replace othseclgd_ = 0 if othseclgd_ < 0 // if assets>debt, LGD=0
*If already exhausted all assets paying back first mortgage, entire other unsecured debt is written off
replace othseclgd_ = nonhomepropertydebt*pd_ if mortlgd_> 0		
label variable othseclgd_ "stressed other secured LGD, using total assets"

*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio		
gen othseclgdlmi_ = 0
replace othseclgdlmi_ = nonhomepropertydebt*pd_ if mortlgdlmi_>0
label variable othseclgdlmi_ "stressed other secured LGD, using total assets, including LMI"	
replace othseclgdlmi_ = othseclgdlmi_*(1-poratio_) if nonhomeLVR>80

*LGDs for remaining assets will be the same regardless of whether LMI included, as we assume that the household would exhaust total assets before LMI is drawn down

* Personal loan LGD
gen othunseclgd_ = 0
replace othunseclgd_ = (nonpropbusccdebt - (asset_to_linsup_ - hdebt -nonhomepropertydebt))*pd_ 	
replace othunseclgd_ = 0 if othunseclgd_ < 0  // if assets>debt, LGD=0
* If already exhausted all assets paying back secured debts then entire personal loan amount is written off
replace othunseclgd_ = nonpropbusccdebt*pd_ if othseclgd_>0 | mortlgd_>0
label variable othunseclgd_ "Other unsecured LGD, using total assets"

*Credit card LGD
gen cclgd_ = 0
replace cclgd_ = (ccdebt - (asset_to_linsup_ - hdebt - nonhomepropertydebt - nonpropbusccdebt))*pd_ 
replace cclgd_ = 0 if cclgd_ < 0  // if assets>debt, LGD=0
*If already exhausted all assets paying back other debt types, entire outstanding credit card debt is written off
replace cclgd_ = (ccdebt)*pd_ if othunseclgd_>0 | othseclgd_>0 | mortlgd_>0
label variable cclgd_ "Credit card LGD, using total assets"

*Foreclosure costs
*Add foreclosure costs to estimates of LGD
*Add these after calculating other LGDs because these are paid by the bank so shouldn't affect other loans
replace mortlgd_ = mortlgd_ + hdebt*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}
replace mortlgdlmi_ = mortlgdlmi_ + hdebt*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}
replace othseclgd_ = othseclgd_ + nonhomepropertydebt*${foreclosure}*pd_ if nonhomeLVR_>${foreclosure_lvr}
replace othseclgdlmi_ = othseclgdlmi_ + nonhomepropertydebt*${foreclosure}*pd_ if nonhomeLVR_>${foreclosure_lvr}		

*All property LGD, excluding and including LMI
gen seclgd_ = othseclgd_ + mortlgd_
gen seclgdlmi_ = othseclgdlmi_ + mortlgdlmi_	

*Total lgd
gen totallgd_ = mortlgd_ + othseclgd_ + othunseclgd_ + cclgd_
label variable totallgd_ "Total LGD, using total assets, excluding LMI"

*Total lgd, taking into account LMI
gen totallgdlmi_ = mortlgdlmi_ + othseclgdlmi_ + othunseclgd_ + cclgd_
label variable totallgdlmi_ "Total LGD, using total assets, including LMI"

*****************************************************************************
*20. GENERATE LGD ESTIMATES AT INDIVIDUAL HOUSEHOLD LEVEL POST-STRESS, USING ONLY PROPERTY ASSETS
*****************************************************************************	
			
*****************************
* EXCLUDING LMI
*****************************

gen lgdh_ = pd_*netdebth_
replace lgdh_ = pd_*netdebth_ + (hdebt+nonhomepropertydebt)*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}
label variable lgdh_ "household total LGD - using property assets, excl LMI, post-stress"

*****************************
* INCLUDING LMI
*****************************

*Generate first mortgage LGD
gen mortlgdh_ = 0
replace mortlgdh_ = (hdebt - propasset_)*pd_
replace mortlgdh_ = 0 if mortlgdh_ < 0  // if assets>debt, LGD=0 
label variable mortlgdh_ "stressed mortgage LGD, using property assets, excl LMI"

*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio
gen mortlgdlmih_ = mortlgdh_
replace mortlgdlmih_ = mortlgdh_*(1-poratio_)*pd_ if LVR>80 
label variable mortlgdlmih_ "stressed mortgage LGD, using property assets, including LMI"

*Generate other secured LGD
gen othseclgdh_ = 0
replace othseclgdh_ = (nonhomepropertydebt - (propasset_ - hdebt))*pd_
replace othseclgdh_ = 0 if othseclgdh_ < 0 // if assets>debt, LGD=0 

*If already exhausted all assets paying back first mortgage, entire other unsecured debt is written off
replace othseclgdh_ = nonhomepropertydebt*pd_ if mortlgdh_> 0		
label variable othseclgdh_ "stressed other secured LGD, using property assets"

*For households with LMI, the insurance covers a proportion of the shortfall (i.e. the lender's loss) that is equal to the payout ratio		
gen othseclgdlmih_ = 0
replace othseclgdlmih_ = nonhomepropertydebt*pd_ if mortlgdlmih_>0
label variable othseclgdlmih_ "stressed other secured LGD, using property assets, including LMI"	
replace othseclgdlmih_ = othseclgdlmih_*(1-poratio_) if nonhomeLVR>80

*LGDs for remaining assets will be the same regardless of whether LMI included, as we assume that the household would exhaust total assets before LMI is drawn down

* Personal loan LGD
gen othunseclgdh_ = 0
replace othunseclgdh_ = (nonpropbusccdebt - (propasset_ - hdebt -nonhomepropertydebt))*pd_ 	
replace othunseclgdh_ = 0 if othunseclgdh_ < 0 // if assets>debt, LGD=0 
* If already exhausted all assets paying back secured debts then entire personal loan amount is written off
replace othunseclgdh_ = nonpropbusccdebt*pd_ if othseclgdh_>0 | mortlgdh_>0
label variable othunseclgdh_ "Other unsecured LGD, using property assets"

*Credit card LGD
gen cclgdh_ = 0
replace cclgdh_ = (ccdebt - (propasset_ - hdebt - nonhomepropertydebt - nonpropbusccdebt))*pd_ 
replace cclgdh_ = 0 if cclgdh_ < 0 // if assets>debt, LGD=0 
*If already exhausted all assets paying back other debt types, entire outstanding credit card debt is written off
replace cclgdh_ = (ccdebt)*pd_ if othunseclgdh_>0 | othseclgdh_>0 | mortlgdh_>0
label variable cclgdh_ "Credit card LGD, using property assets"

*Foreclosure costs
*Add foreclosure costs to estimates of LGD
*Add these after calculating other LGDs because these are paid by the bank so shouldn't affect other loans
replace mortlgdh_ = mortlgdh_ + hdebt*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}
replace mortlgdlmih_ = mortlgdlmih_ + hdebt*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}
replace othseclgdh_ = othseclgdh_ + nonhomepropertydebt*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}
replace othseclgdlmih_ = othseclgdlmih_ + nonhomepropertydebt*${foreclosure}*pd_ if LVR_>${foreclosure_lvr}		

*All property LGD, excluding and including LMI
gen seclgdh_ = othseclgdh_ + mortlgdh_
gen seclgdlmih_ = othseclgdlmih_ + mortlgdlmih_	

*Total lgd
gen totallgdh_ = mortlgdh_ + othseclgdh_ + othunseclgdh_ + cclgdh_
label variable totallgdh_ "Total LGD, using property assets, excluding LMI"

*Total lgd, taking into account LMI
gen totallgdlmih_ = mortlgdlmih_ + othseclgdlmih_ + othunseclgdh_ + cclgdh_
label variable totallgdlmih_ "Total LGD, using property assets, including LMI"

*****************************************************************************
*21. CALCULATE LGDs AT POPULATION LEVEL
*****************************************************************************			
	
*Calculate population LGDs (excluding LMI), total assets
*First mortgage
egen popmortlgd_ = total(mortlgd_*hwt)
label variable popmortlgd_ "population mortgage LGD, using total assets"
*in percentage terms
gen popmortlgdp_ = popmortlgd_/(popmortead_)*100
label variable popmortlgdp_ "population mortgage LGD, % of vulnerable debt, using total assets"
*Other secured
egen popothseclgd_ = total(othseclgd_*hwt)
label variable popothseclgd_ "population other secured LGD, using total assets"
*in percentage terms
gen popothseclgdp_ = popothseclgd_/(popothsecead_)*100
label variable popothseclgdp_ "population other secured LGD, % of vulnerable debt, using total assets"
*All secured
egen popseclgd_ = total(seclgd_*hwt)
label variable popseclgd_ "population mortgage LGD, using total assets"
*in percentage terms
gen popseclgdp_ = popseclgd_/(popsecead_)*100
label variable popmortlgdp_ "population mortgage LGD, % of vulnerable debt, using total assets"
*Oth unsecured
egen popothunseclgd_ = total(othunseclgd_*hwt)
label variable popothunseclgd_ "population other unsecured LGD, using total assets"
*in percentage terms
gen popothunseclgdp_ = popothunseclgd_/(popothead_)*100
label variable popothunseclgdp_ "population other unsecured LGD, % of vulnerable debt, using total assets"
*Credit card
egen popcclgd_ = total(cclgd_*hwt)
label variable popcclgd_ "population credit card LGD, using total assets"
*in percentage terms
gen popcclgdp_ = popcclgd_/(popccead_)*100
label variable popcclgdp_ "population credit card LGD, % of vulnerable debt, using total assets"
*Total LGD
egen poptotallgd_ = total(totallgd_*hwt)
label variable poptotallgd_ "population total LGD, using total assets"
*in percentage terms
gen poptotallgdp_ = poptotallgd_/(popmortead_ + popothsecead_ + popothead_ + popccead_)*100
label variable poptotallgdp_ "population total LGD, % of vulnerable debt, using total assets"

*Calculate population LGDs (excluding LMI), using only property assets
*First mortgage
egen popmortlgdh_ = total(mortlgdh_*hwt)
label variable popmortlgdh_ "population mortgage LGD, using property assets"
*in percentage terms
gen popmortlgdhp_ = popmortlgdh_/(popmortead_)*100
label variable popmortlgdhp_ "population mortgage LGD, % of vulnerable debt, using property assets"
*Other secured
egen popothseclgdh_ = total(othseclgdh_*hwt)
label variable popothseclgdh_ "population other secured LGD, using total assets"
*in percentage terms
gen popothseclgdhp_ = popothseclgdh_/(popothsecead_)*100
label variable popothseclgdhp_ "population other secured LGD, % of vulnerable debt, using property assets"
*All secured
egen popseclgdh_ = total(seclgdh_*hwt)
label variable popseclgd_ "population secured LGD, using total assets"
*in percentage terms
gen popseclgdhp_ = popseclgdh_/(popsecead_)*100
label variable popmortlgdp_ "population secured LGD, % of vulnerable debt, using total assets"
*Other unsecured
egen popothunseclgdh_ = total(othunseclgdh_*hwt)
label variable popothunseclgdh_ "population other unsecured LGD, using total assets"
*in percentage terms
gen popothunseclgdhp_ = popothunseclgdh_/(popothead_)*100
label variable popothunseclgdhp_ "population other unsecured LGD, % of vulnerable debt, using property assets"
*Credit card
egen popcclgdh_ = total(cclgdh_*hwt)
label variable popcclgdh_ "population credit card LGD, using total assets"
*in percentage terms
gen popcclgdhp_ = popcclgdh_/(popccead_)*100
label variable popcclgdhp_ "population credit card LGD, % of vulnerable debt, using property assets"
*Total LGD
egen poptotallgdh_ = total(totallgdh_*hwt)
label variable poptotallgdh_ "population total LGD, using property assets"
*in percentage terms
gen poptotallgdhp_ = poptotallgdh_/(popmortead_+ popothsecead_ + popothead_ + popccead_)*100
label variable poptotallgdhp_ "population total LGD, % of vulnerable debt, using property assets"

*Calculate population LGDs (including LMI), total assets 
*LGDs are only different for property loans
*mortgage
*Population first mortgage LGD taking into account all assets 
egen popmortlgdlmi_ = total(mortlgdlmi_*hwt)
label variable popmortlgdlmi_ "population mortgage LGD, using total assets, including LMI"
*in percentage terms
gen popmortlgdlmip_ = popmortlgdlmi_/(popmortead_)*100
label variable popmortlgdlmip_ "population mortgage LGD, % of vulnerable debt, using total assets, including LMI"
*other secured
egen popothseclgdlmi_ = total(othseclgdlmi_*hwt)
label variable popothseclgdlmi_ "population other secured LGD, using total assets, including LMI"
*in percentage terms
gen popothseclgdlmip_ = popothseclgdlmi_/(popothsecead_)*100
label variable popothseclgdlmip_ "population other secured LGD, % of vulnerable debt, using total assets, including LMI"
*secured 
egen popseclgdlmi_ = total(seclgdlmi_*hwt)
label variable popseclgdlmi_ "population mortgage LGD, using total assets"
*in percentage terms
gen popseclgdlmip_ = popseclgdlmi_/(popsecead_)*100
label variable popseclgdlmip_ "population mortgage LGD, % of vulnerable debt, using total assets"
*Total LGD
egen poplgdlmi_ = total(totallgdlmi_*hwt)
label variable poplgdlmi_ "population LGD, using total assets, including LMI"
*in percentage terms
gen poplgdlmip_ = poplgdlmi_/(popmortead_+ popothsecead_ + popothead_ + popccead_)*100
label variable poplgdlmip_ "population LGD, % of vulnerable debt, using total assets, including LMI"

*Calculate population LGDs (including LMI), property assets
*LGDs are only different for property loans
*First mortgage
*Population first mortgage LGD taking into account all assets 
egen popmortlgdlmih_ = total(mortlgdlmih_*hwt)
label variable popmortlgdlmih_ "population mortgage LGD, using property assets, including LMI"
*in percentage terms
gen popmortlgdlmihp_ = popmortlgdlmih_/(popmortead_)*100
label variable popmortlgdlmihp_ "population mortgage LGD, % of vulnerable debt, using property assets, including LMI"
*other secured
egen popothseclgdlmih_ = total(othseclgdlmih_*hwt)
label variable popothseclgdlmih_ "population other secured LGD, using property assets, including LMI"
*in percentage terms
gen popothseclgdlmihp_ = popothseclgdlmih_/(popothsecead_)*100
label variable popothseclgdhp_ "population other secured LGD, % of vulnerable debt, using property assets, including LMI"
*secured 
egen popseclgdlmih_ = total(seclgdlmih_*hwt)
label variable popseclgdlmih_ "population mortgage LGD, using property assets"
*in percentage terms
gen popseclgdlmihp_ = popseclgdlmih_/(popsecead_)*100
label variable popmortlgdhp_ "population mortgage LGD, % of vulnerable debt, using property assets"
*Total LGD
egen poplgdlmih_ = total(totallgdlmih_*hwt)
label variable poplgdlmih_ "population LGD, using property assets, including LMI"
*in percentage terms
gen poplgdlmihp_ = poplgdlmih_/(popmortead_+ popothsecead_ + popothead_ + popccead_)*100
label variable poplgdlmihp_ "population LGD, % of vulnerable debt, using property assets, including LMI"

*****************************************************************************
*22. CALCULATE SYSTEM LOSSES, POST-STRESS
*****************************************************************************					

*************
*TOTAL LOSSES
*************

*EXCLUDING LMI
*Calculate total post-stress system losses, in percentage terms, total assets, excluding LMI
gen systemloss_ = (poptoteadp_/100)*(poptotallgdp_/100)*100
label variable systemloss_ "post-stress losses, % total debt, using total assets, excl LMI"

*Calculate total post-stress system losses, in percentage terms, property assets only, excluding LMI
gen systemlossh_ = (poptoteadp_/100)*(poptotallgdhp_/100)*100
label variable systemlossh_ "post-stress losses, % total debt, using property assets, excl LMI"

*INCLUDING LMI
gen systemlosslmi_ = (poptoteadp_/100)*(poplgdlmip_/100)*100
label variable systemlosslmi_ "post-stress losses, % total debt, using total assets, incl LMI"
gen systemlosslmih_ = (poptoteadp_/100)*(poplgdlmihp_/100)*100
label variable systemlosslmih_ "post-stress losses, % total debt, using property assets, incl LMI"

****************
*LOSS COMPONENTS
****************

*EXLUDING LMI, TOTAL ASSETS

*first mortgage losses taking into account all assets 
gen mortloss_ = (popmorteadp_/100)*(popmortlgdp_/100)*100
label variable mortloss_ "post-stress mortgage losses, % total debt, using total assets, excl LMI"
gen othsecloss_ = (popothseceadp_/100)*(popothseclgdp_/100)*100
label variable othsecloss_ "post-stress other secured losses, % total debt, using total assets, excl LMI"
gen othunsecloss_ = (popotheadp_/100)*(popothunseclgdp_/100)*100
label variable othunsecloss_ "post-stress other unsecured losses, % total debt, using total assets, excl LMI"
gen ccloss_ = (popcceadp_/100)*(popcclgdp_/100)*100
label variable ccloss_ "post-stress credit card losses, % total debt, using total assets, excl LMI"
gen secloss_ = (popseceadp_/100)*(popseclgdp_/100)*100
label variable secloss_ "post-stress mortgage losses, % total debt, using total assets, excl LMI"

*EXCLUDING LMI, USING ONLY PROPERTY ASSETS
gen mortlossh_ = (popmorteadp_/100)*(popmortlgdhp_/100)*100
label variable mortlossh_ "post-stress mortgage losses, % total debt, using property assets, excl LMI"
gen othseclossh_ = (popothseceadp_/100)*(popothseclgdhp_/100)*100
label variable othseclossh_ "post-stress other secured losses, % total debt, using property assets, excl LMI"
gen othunseclossh_ = (popotheadp_/100)*(popothunseclgdhp_/100)*100
label variable othunseclossh_ "post-stress other unsecured losses, % total debt, using property assets, excl LMI"
gen cclossh_ = (popcceadp_/100)*(popcclgdhp_/100)*100
label variable cclossh_ "post-stress credit card losses, % total debt, using property assets, excl LMI"
gen seclossh_ = (popseceadp_/100)*(popseclgdhp_/100)*100
label variable seclossh_ "post-stress mortgage losses, % total debt, using total assets, excl LMI"

*INCLUDING LMI

gen mortlosslmi_ = (popmorteadp_/100)*(popmortlgdlmip_/100)*100
label variable mortlosslmi_ "post-stress mortgage losses, % total debt, using total assets, incl LMI"
gen mortlosslmih_ = (popmorteadp_/100)*(popmortlgdlmihp_/100)*100
label variable mortlosslmih_ "post-stress mortgage losses, % total debt, using property assets, incl LMI"
gen seclosslmi_ = (popseceadp_/100)*(popseclgdlmip_/100)*100
label variable seclosslmi_ "post-stress mortgage losses, % total debt, using total assets, incl LMI"
gen seclosslmih_ = (popseceadp_/100)*(popseclgdlmihp_/100)*100
label variable seclosslmih_ "post-stress mortgage losses, % total debt, using property assets, incl LMI"
	
*****************************************************************************
*23. EFFECT ON CONSUMPTION FROM WEALTH AND INCOME SHOCKS
*****************************************************************************			

*************
*INCOME SHOCK
*************

*Calculate change in post-tax household income
gen hhtotaldis_shock_abs = hhtotaldis_ - hhtotaldis
replace hhtotaldis_shock_abs = 0 if  hhtotaldis_shock_abs>-1 & hhtotaldis_shock_abs<1 // remove small rounding errors

*Aggregate individual to household
foreach x of newlist unemp_change bonus_change hours_change inv_change oth_change unben_change {
	gen `x' = `x'1 + `x'2 + `x'3 + `x'4 + `x'5 + `x'6
	replace `x' = 0 if `x' > -1 & `x' < 1 // remove small rounding errors
	}
gen pretax_tax_y_change = unemp_change + bonus_change + hours_change + inv_change + oth_change + unben_change
*Income shock shares (to decompose income shock into components)
foreach x of varlist unemp_change bonus_change hours_change inv_change oth_change unben_change {
	gen `x'_share = `x'/pretax_tax_y_change // pre-tax share
	gen `x'_posttax = `x'_share*hhtotaldis_shock_abs // apply pre-tax share to post-shock change
	replace `x'_posttax = 0 if hhtotaldis_shock_abs==0 		
	replace `x'_share = 0 if hhtotaldis_shock_abs==0 
	}

gen emp_change_share = unemp_change_share + bonus_change_share + hours_change_share + unben_change_share // employment income (memo-item)

*************
*WEALTH SHOCK
*************	

*Calculate change in wealth
gen wealth_shock_abs = wealth_ - wealth
replace wealth_shock_abs = 0 if  wealth_shock_abs>-1 & wealth_shock_abs<1 // remove small rounding errors

*Decomposing the wealth changes
gen sp_change = asset_sp_lin_ - asset_sp_lin // super
gen hs_change = asset_hs_ - asset_hs // housing
gen eq_change = asset_eq_ - asset_eq // equity
gen nres_change = asset_nres_ - asset_nres // nonres	
gen tr_change = asset_tr_ - asset_tr // trusts	
gen asset_oth_change = asset_oth_fin_ + asset_oth_ - (asset_oth_fin + asset_oth) // other assets

gen asset_to_change = asset_to_linsup_ - asset_to_linsup // all assets (same as wealth change)
gen non_hs_change = asset_to_change - hs_change // non-housing change (memo-item)

*Wealth shock shares (to decompose wealth shock into components)
gen sp_share = sp_change/wealth_shock_abs // super
gen hs_share = hs_change/wealth_shock_abs // housing
gen eq_share = eq_change/wealth_shock_abs // equity
gen nres_share = nres_change/wealth_shock_abs // nonres	
gen tr_share = tr_change/wealth_shock_abs // trusts	
gen asset_oth_share = asset_oth_change/wealth_shock_abs // other assets	

gen non_hs_share = non_hs_change/wealth_shock_abs // non-housing (memo-item)		
//replace with 0 if no change in wealth
foreach x of varlist sp_share hs_share eq_share nres_share tr_share asset_oth_share non_hs_share{
	replace `x' = 0 if wealth_shock_abs==0
	}		

*****************************************************	
*      Different MPCw from housing vs non-housing 	*
*****************************************************	
*Calculate multiplier for housing MPC compared to non-housing MPC while ensuring aggregate MPC is held constant
egen den = total(hwt)
egen num1 = total(hs_change * hwt) 
egen num2 = total(non_hs_change * hwt)
gen hs_change_mean = num1/den // weighted average change in housing wealth
gen non_hs_change_mean = num2/den // weighted average change in non-housing wealth	
drop den num1 num2
gen hs_share_mean = hs_change_mean/(hs_change_mean + non_hs_change_mean) // share of wealth loss that is housing
gen mpcw_hs_discount = 1 / (hs_share_mean +  (1-hs_share_mean)* ${mpc_nonhs_multiplier} ) // multiply aggregate MPCw by this discount to get desired MPCw for housing

*Assign MPC for housing and non-housing
gen mpcw_hs = mpcw * mpcw_hs_discount
replace mpcw_hs = mpcw if mpcw_hs_discount ==.
gen mpcw_nonhs = mpcw * mpcw_hs_discount * ${mpc_nonhs_multiplier}
replace mpcw_nonhs = mpcw if mpcw_hs_discount ==.

*****************************************************
*Effect on consumption     	
*****************************************************
gen con_shock_hs = hs_change * mpcw_hs
label variable con_shock_hs "effect on C from housing wealth"
gen con_shock_nonhs =  non_hs_change * mpcw_nonhs
label variable con_shock_nonhs "effect on C from non-housing wealth"	
gen con_shock_w = con_shock_hs+con_shock_nonhs
label variable con_shock_w "effect on C from all wealth"	
	
gen con_shock_y = hhtotaldis_shock_abs * mpcy	
label variable con_shock_y "effect on C from income"
gen con_shock = con_shock_w + con_shock_y
label variable con_shock "effect on C from income and wealth"	
gen old_con_shock = con_shock_w + con_shock_y // effect on C before moderating by subsistence consumption 

*Post-stress consumption (this will be updated after moderation of subsistence consumption)
gen gs_exp_ann_ = gs_exp_ann + con_shock
label variable gs_exp_ann_ "G&S expenditure, post-stress"	
gen gs_exp_ann_nonIR_ = gs_exp_ann_nonIR + con_shock 
label variable gs_exp_ann_ "G&S expenditure (excluding rent and imputed rent), post-stress"		
*Flag for households who consume at or below subsistence consumption
gen subsist_=0
replace subsist_ = 1 if gs_exp_ann_nonIR_ < bascon & con_shock!=. & bascon!=. // post-stress spending less than bascon

*Moderating consumption shock to consume at least at subsistence consumption
replace con_shock = 0 if gs_exp_ann_nonIR <= bascon // no further shock if initial spending is below bascon	
replace con_shock = -gs_exp_ann_nonIR + bascon if /*pre-spending is above basic*/ gs_exp_ann_nonIR > bascon & /*post-spending is below bascon*/ gs_exp_ann_nonIR_ < bascon & con_shock!=. & bascon!=.	

*Update consumption for moderation
replace gs_exp_ann_ = gs_exp_ann + con_shock	
replace gs_exp_ann_nonIR_ = gs_exp_ann_nonIR + con_shock 

*Effect of moderation
gen unmod_con_shock = old_con_shock - con_shock
label variable unmod_con_shock "Additional effect on C if no moderation"

*Adjust wealth and income decomposition for moderated consumption shock
replace con_shock_hs = con_shock / old_con_shock * con_shock_hs if con_shock_hs<0
replace con_shock_nonhs = con_shock / old_con_shock * con_shock_nonhs if con_shock_nonhs<0	
replace con_shock_w = con_shock / old_con_shock * con_shock_w if con_shock_w<0
replace con_shock_y = con_shock / old_con_shock * con_shock_y if con_shock_y<0

*Decomposition shares	
gen con_shock_y_unemp = unemp_change_share * con_shock_y 
	label variable con_shock_y_unemp "effect on C from unemployment"
gen	con_shock_y_bonus = bonus_change_share * con_shock_y
	label variable con_shock_y_bonus "effect on C from bonuses and overtime"		
gen	con_shock_y_hours = hours_change_share * con_shock_y
	label variable con_shock_y_hours "effect on C from hours"	
gen con_shock_y_unben = unben_change_share * con_shock_y 
	label variable con_shock_y_unben "effect on C from reg un ben"		
	
gen con_shock_y_emp = emp_change_share * con_shock_y 
	label variable con_shock_y_emp "effect on C from employment income"			
gen	con_shock_y_inv = inv_change_share * con_shock_y
	label variable con_shock_y_inv "effect on C from investment income"		
gen	con_shock_y_oth = oth_change_share * con_shock_y
	label variable con_shock_y_oth "effect on C from other income"		
gen con_shock_w_equity = eq_share / (1-hs_share) * con_shock_nonhs
	label variable con_shock_w_equity "effect on C from equity wealth"		
gen con_shock_w_trusts = tr_share / (1-hs_share) * con_shock_nonhs
	label variable con_shock_w_trusts "effect on C from trust wealth"			
gen con_shock_w_super = sp_share / (1-hs_share) * con_shock_nonhs
	label variable con_shock_w_super "effect on C from super wealth"			
gen con_shock_w_housing = con_shock_hs
	label variable con_shock_w_housing "effect on C from housing wealth"			
gen con_shock_w_nres = nres_share / (1-hs_share) * con_shock_nonhs	
	label variable con_shock_w_nres "effect on C from non-res property wealth"			
gen con_shock_w_other = asset_oth_share/ (1-hs_share) * con_shock_nonhs	
	label variable con_shock_w_other "effect on C from other asset wealth"			

replace con_shock_w_equity = 0 if hs_share==1
replace con_shock_w_trusts = 0 if hs_share==1
replace con_shock_w_super = 0 if hs_share==1
replace con_shock_w_nres = 0 if hs_share==1
replace con_shock_w_other = 0 if hs_share==1

*****************************************************************************
*24. GENERATE VARIABLES TO SUMMARIZE RESULTS
*****************************************************************************			

*****************************************************	
*      CONSUMPTION RESULTS
*****************************************************

*Variables of interest
local vlist con_shock_y_unemp con_shock_y_bonus con_shock_y_hours con_shock_y_emp con_shock_y_inv con_shock_y_oth con_shock_w_equity con_shock_w_trusts con_shock_w_super con_shock_w_housing con_shock_w_nres con_shock_w_other con_shock_y con_shock_w con_shock unmod_con_shock gs_exp_ann_ wealth hhtotaldis hhdebt asset_to_linsup mpcw mpcy age unem unem_ hh_attlf gs_exp_ann bascon subsist_ subsist

	*Totals
foreach x of  varlist `vlist'{
	tabstat `x' [aw=hwt], by(year) stats(mean) save
	mat temp = r(Stat1)
	gen `x'_t = temp[1,1]
	mat list temp
	mat drop temp
	}
	
if ${distribution_results} == 1 {	
	* Shocks by DTA
	foreach x of varlist `vlist' {
		tabstat `x' [aw=hwt], by (DTA_x) stats(mean) save
		forvalues l = 1(1)5 {
			mat temp = r(Stat`l')
			gen `x'_DTA_`l' = temp[1,1]
			mat list temp
			mat drop temp
			}
		}

	* Shocks by DTI
	foreach x of  varlist `vlist'{
		tabstat `x' [aw=hwt], by (DTI_x) stats(mean) save
		forvalues l = 1(1)5 {
			mat temp = r(Stat`l')
			gen `x'_DTI_`l' = temp[1,1]
			mat list temp
			mat drop temp
			}
		}

	* Shocks by income
	foreach x of  varlist `vlist'{
		tabstat `x' [aw=hwt], by (hhtotaldis_x) stats(mean) save
		forvalues l = 1(1)5 {
			mat temp = r(Stat`l')
			gen `x'_inc_`l' = temp[1,1]
			mat list temp
			mat drop temp
			}
		}

	* Shocks by age
	foreach x of varlist `vlist'{
		tabstat `x' [aw=hwt], by (age_cat) stats(mean) save
		forvalues l = 1(1)6 {
			mat temp = r(Stat`l')
			gen `x'_age_`l' = temp[1,1]
			mat list temp
			mat drop temp
			}
		}
}

end

*****************************************************************************
*25. RUN SIMULATION AND EXPORT RESULTS
*****************************************************************************	

*Bank stress test variables
local bank_stess_list ///
ccloss ccloss_ cclossh cclossh_ mortloss mortloss_ mortlossh mortlossh_ mortlosslmi mortlosslmi_ mortlosslmih mortlosslmih_ othsecloss othsecloss_ othseclossh othseclossh_ othunsecloss othunsecloss_ othunseclossh othunseclossh_ pdp pdp_ popcceadp_ popcclgdhp_ popcclgdp_ poplgdlmihp_ poplgdlmip_ popmorteadp_ popmortlgdhp_ popmortlgdlmihp_ popmortlgdlmip_ popmortlgdp_ popotheadp_ popothseceadp_ popothseclgdhp_ popothseclgdp_ popothunseclgdhp_ popothunseclgdp_ popseceadp_ poptotallgdhp_ poptotallgdp_ poptoteadp_ secloss_ seclossh_ seclosslmi_ seclosslmih_ systemloss systemloss_ systemlossh systemlossh_ systemlosslmi systemlosslmi_ systemlosslmih systemlosslmih_ 

*Consumption stress test aggregate variables
local cons_agg_list ///
unrate_ con_shock_y_unemp_t con_shock_y_bonus_t con_shock_y_hours_t con_shock_y_emp_t con_shock_y_inv_t con_shock_y_oth_t con_shock_w_equity_t con_shock_w_trusts_t con_shock_w_super_t con_shock_w_housing_t con_shock_w_nres_t con_shock_w_other_t con_shock_y_t con_shock_w_t con_shock_t unmod_con_shock_t gs_exp_ann__t wealth_t hhtotaldis_t hhdebt_t asset_to_linsup_t mpcw_t mpcy_t age_t unem_t unem__t hh_attlf_t gs_exp_ann_t subsist__t subsist_t bascon_t

*Consumption stress test distribution variables
local cons_dist_list ///
con_shock_y_unemp_DTA_1 con_shock_y_unemp_DTA_2 con_shock_y_unemp_DTA_3 con_shock_y_unemp_DTA_4 con_shock_y_unemp_DTA_5 con_shock_y_unemp_DTI_1 con_shock_y_unemp_DTI_2 con_shock_y_unemp_DTI_3 con_shock_y_unemp_DTI_4 con_shock_y_unemp_DTI_5 con_shock_y_unemp_age_1 con_shock_y_unemp_age_2 con_shock_y_unemp_age_3 con_shock_y_unemp_age_4 con_shock_y_unemp_age_5 con_shock_y_unemp_age_6 con_shock_y_unemp_inc_1 con_shock_y_unemp_inc_2 con_shock_y_unemp_inc_3 con_shock_y_unemp_inc_4 con_shock_y_unemp_inc_5 con_shock_y_bonus_DTA_1 con_shock_y_bonus_DTA_2 con_shock_y_bonus_DTA_3 con_shock_y_bonus_DTA_4 con_shock_y_bonus_DTA_5 con_shock_y_bonus_DTI_1 con_shock_y_bonus_DTI_2 con_shock_y_bonus_DTI_3 con_shock_y_bonus_DTI_4 con_shock_y_bonus_DTI_5 con_shock_y_bonus_age_1 con_shock_y_bonus_age_2 con_shock_y_bonus_age_3 con_shock_y_bonus_age_4 con_shock_y_bonus_age_5 con_shock_y_bonus_age_6 con_shock_y_bonus_inc_1 con_shock_y_bonus_inc_2 con_shock_y_bonus_inc_3 con_shock_y_bonus_inc_4 con_shock_y_bonus_inc_5 con_shock_y_hours_DTA_1 con_shock_y_hours_DTA_2 con_shock_y_hours_DTA_3 con_shock_y_hours_DTA_4 con_shock_y_hours_DTA_5 con_shock_y_hours_DTI_1 con_shock_y_hours_DTI_2 con_shock_y_hours_DTI_3 con_shock_y_hours_DTI_4 con_shock_y_hours_DTI_5 con_shock_y_hours_age_1 con_shock_y_hours_age_2 con_shock_y_hours_age_3 con_shock_y_hours_age_4 con_shock_y_hours_age_5 con_shock_y_hours_age_6 con_shock_y_hours_inc_1 con_shock_y_hours_inc_2 con_shock_y_hours_inc_3 con_shock_y_hours_inc_4 con_shock_y_hours_inc_5 con_shock_y_emp_DTA_1 con_shock_y_emp_DTA_2 con_shock_y_emp_DTA_3 con_shock_y_emp_DTA_4 con_shock_y_emp_DTA_5 con_shock_y_emp_DTI_1 con_shock_y_emp_DTI_2 con_shock_y_emp_DTI_3 con_shock_y_emp_DTI_4 con_shock_y_emp_DTI_5 con_shock_y_emp_age_1 con_shock_y_emp_age_2 con_shock_y_emp_age_3 con_shock_y_emp_age_4 con_shock_y_emp_age_5 con_shock_y_emp_age_6 con_shock_y_emp_inc_1 con_shock_y_emp_inc_2 con_shock_y_emp_inc_3 con_shock_y_emp_inc_4 con_shock_y_emp_inc_5 con_shock_y_inv_DTA_1 con_shock_y_inv_DTA_2 con_shock_y_inv_DTA_3 con_shock_y_inv_DTA_4 con_shock_y_inv_DTA_5 con_shock_y_inv_DTI_1 con_shock_y_inv_DTI_2 con_shock_y_inv_DTI_3 con_shock_y_inv_DTI_4 con_shock_y_inv_DTI_5 con_shock_y_inv_age_1 con_shock_y_inv_age_2 con_shock_y_inv_age_3 con_shock_y_inv_age_4 con_shock_y_inv_age_5 con_shock_y_inv_age_6 con_shock_y_inv_inc_1 con_shock_y_inv_inc_2 con_shock_y_inv_inc_3 con_shock_y_inv_inc_4 con_shock_y_inv_inc_5 con_shock_y_oth_DTA_1 con_shock_y_oth_DTA_2 con_shock_y_oth_DTA_3 con_shock_y_oth_DTA_4 con_shock_y_oth_DTA_5 con_shock_y_oth_DTI_1 con_shock_y_oth_DTI_2 con_shock_y_oth_DTI_3 con_shock_y_oth_DTI_4 con_shock_y_oth_DTI_5 con_shock_y_oth_age_1 con_shock_y_oth_age_2 con_shock_y_oth_age_3 con_shock_y_oth_age_4 con_shock_y_oth_age_5 con_shock_y_oth_age_6 con_shock_y_oth_inc_1 con_shock_y_oth_inc_2 con_shock_y_oth_inc_3 con_shock_y_oth_inc_4 con_shock_y_oth_inc_5 con_shock_w_equity_DTA_1 con_shock_w_equity_DTA_2 con_shock_w_equity_DTA_3 con_shock_w_equity_DTA_4 con_shock_w_equity_DTA_5 con_shock_w_equity_DTI_1 con_shock_w_equity_DTI_2 con_shock_w_equity_DTI_3 con_shock_w_equity_DTI_4 con_shock_w_equity_DTI_5 con_shock_w_equity_age_1 con_shock_w_equity_age_2 con_shock_w_equity_age_3 con_shock_w_equity_age_4 con_shock_w_equity_age_5 con_shock_w_equity_age_6 con_shock_w_equity_inc_1 con_shock_w_equity_inc_2 con_shock_w_equity_inc_3 con_shock_w_equity_inc_4 con_shock_w_equity_inc_5 con_shock_w_trusts_DTA_1 con_shock_w_trusts_DTA_2 con_shock_w_trusts_DTA_3 con_shock_w_trusts_DTA_4 con_shock_w_trusts_DTA_5 con_shock_w_trusts_DTI_1 con_shock_w_trusts_DTI_2 con_shock_w_trusts_DTI_3 con_shock_w_trusts_DTI_4 con_shock_w_trusts_DTI_5 con_shock_w_trusts_age_1 con_shock_w_trusts_age_2 con_shock_w_trusts_age_3 con_shock_w_trusts_age_4 con_shock_w_trusts_age_5 con_shock_w_trusts_age_6 con_shock_w_trusts_inc_1 con_shock_w_trusts_inc_2 con_shock_w_trusts_inc_3 con_shock_w_trusts_inc_4 con_shock_w_trusts_inc_5 con_shock_w_super_DTA_1 con_shock_w_super_DTA_2 con_shock_w_super_DTA_3 con_shock_w_super_DTA_4 con_shock_w_super_DTA_5 con_shock_w_super_DTI_1 con_shock_w_super_DTI_2 con_shock_w_super_DTI_3 con_shock_w_super_DTI_4 con_shock_w_super_DTI_5 con_shock_w_super_age_1 con_shock_w_super_age_2 con_shock_w_super_age_3 con_shock_w_super_age_4 con_shock_w_super_age_5 con_shock_w_super_age_6 con_shock_w_super_inc_1 con_shock_w_super_inc_2 con_shock_w_super_inc_3 con_shock_w_super_inc_4 con_shock_w_super_inc_5 con_shock_w_housing_DTA_1 con_shock_w_housing_DTA_2 con_shock_w_housing_DTA_3 con_shock_w_housing_DTA_4 con_shock_w_housing_DTA_5 con_shock_w_housing_DTI_1 con_shock_w_housing_DTI_2 con_shock_w_housing_DTI_3 con_shock_w_housing_DTI_4 con_shock_w_housing_DTI_5 con_shock_w_housing_age_1 con_shock_w_housing_age_2 con_shock_w_housing_age_3 con_shock_w_housing_age_4 con_shock_w_housing_age_5 con_shock_w_housing_age_6 con_shock_w_housing_inc_1 con_shock_w_housing_inc_2 con_shock_w_housing_inc_3 con_shock_w_housing_inc_4 con_shock_w_housing_inc_5 con_shock_w_nres_DTA_1 con_shock_w_nres_DTA_2 con_shock_w_nres_DTA_3 con_shock_w_nres_DTA_4 con_shock_w_nres_DTA_5 con_shock_w_nres_DTI_1 con_shock_w_nres_DTI_2 con_shock_w_nres_DTI_3 con_shock_w_nres_DTI_4 con_shock_w_nres_DTI_5 con_shock_w_nres_age_1 con_shock_w_nres_age_2 con_shock_w_nres_age_3 con_shock_w_nres_age_4 con_shock_w_nres_age_5 con_shock_w_nres_age_6 con_shock_w_nres_inc_1 con_shock_w_nres_inc_2 con_shock_w_nres_inc_3 con_shock_w_nres_inc_4 con_shock_w_nres_inc_5 con_shock_w_other_DTA_1 con_shock_w_other_DTA_2 con_shock_w_other_DTA_3 con_shock_w_other_DTA_4 con_shock_w_other_DTA_5 con_shock_w_other_DTI_1 con_shock_w_other_DTI_2 con_shock_w_other_DTI_3 con_shock_w_other_DTI_4 con_shock_w_other_DTI_5 con_shock_w_other_age_1 con_shock_w_other_age_2 con_shock_w_other_age_3 con_shock_w_other_age_4 con_shock_w_other_age_5 con_shock_w_other_age_6 con_shock_w_other_inc_1 con_shock_w_other_inc_2 con_shock_w_other_inc_3 con_shock_w_other_inc_4 con_shock_w_other_inc_5 con_shock_y_DTA_1 con_shock_y_DTA_2 con_shock_y_DTA_3 con_shock_y_DTA_4 con_shock_y_DTA_5 con_shock_y_DTI_1 con_shock_y_DTI_2 con_shock_y_DTI_3 con_shock_y_DTI_4 con_shock_y_DTI_5 con_shock_y_age_1 con_shock_y_age_2 con_shock_y_age_3 con_shock_y_age_4 con_shock_y_age_5 con_shock_y_age_6 con_shock_y_inc_1 con_shock_y_inc_2 con_shock_y_inc_3 con_shock_y_inc_4 con_shock_y_inc_5 con_shock_w_DTA_1 con_shock_w_DTA_2 con_shock_w_DTA_3 con_shock_w_DTA_4 con_shock_w_DTA_5 con_shock_w_DTI_1 con_shock_w_DTI_2 con_shock_w_DTI_3 con_shock_w_DTI_4 con_shock_w_DTI_5 con_shock_w_age_1 con_shock_w_age_2 con_shock_w_age_3 con_shock_w_age_4 con_shock_w_age_5 con_shock_w_age_6 con_shock_w_inc_1 con_shock_w_inc_2 con_shock_w_inc_3 con_shock_w_inc_4 con_shock_w_inc_5 con_shock_DTA_1 con_shock_DTA_2 con_shock_DTA_3 con_shock_DTA_4 con_shock_DTA_5 con_shock_DTI_1 con_shock_DTI_2 con_shock_DTI_3 con_shock_DTI_4 con_shock_DTI_5 con_shock_age_1 con_shock_age_2 con_shock_age_3 con_shock_age_4 con_shock_age_5 con_shock_age_6 con_shock_inc_1 con_shock_inc_2 con_shock_inc_3 con_shock_inc_4 con_shock_inc_5 unmod_con_shock_DTA_1 unmod_con_shock_DTA_2 unmod_con_shock_DTA_3 unmod_con_shock_DTA_4 unmod_con_shock_DTA_5 unmod_con_shock_DTI_1 unmod_con_shock_DTI_2 unmod_con_shock_DTI_3 unmod_con_shock_DTI_4 unmod_con_shock_DTI_5 unmod_con_shock_age_1 unmod_con_shock_age_2 unmod_con_shock_age_3 unmod_con_shock_age_4 unmod_con_shock_age_5 unmod_con_shock_age_6 unmod_con_shock_inc_1 unmod_con_shock_inc_2 unmod_con_shock_inc_3 unmod_con_shock_inc_4 unmod_con_shock_inc_5 gs_exp_ann__DTA_1 gs_exp_ann__DTA_2 gs_exp_ann__DTA_3 gs_exp_ann__DTA_4 gs_exp_ann__DTA_5 gs_exp_ann__DTI_1 gs_exp_ann__DTI_2 gs_exp_ann__DTI_3 gs_exp_ann__DTI_4 gs_exp_ann__DTI_5 gs_exp_ann__age_1 gs_exp_ann__age_2 gs_exp_ann__age_3 gs_exp_ann__age_4 gs_exp_ann__age_5 gs_exp_ann__age_6 gs_exp_ann__inc_1 gs_exp_ann__inc_2 gs_exp_ann__inc_3 gs_exp_ann__inc_4 gs_exp_ann__inc_5 wealth_DTA_1 wealth_DTA_2 wealth_DTA_3 wealth_DTA_4 wealth_DTA_5 wealth_DTI_1 wealth_DTI_2 wealth_DTI_3 wealth_DTI_4 wealth_DTI_5 wealth_age_1 wealth_age_2 wealth_age_3 wealth_age_4 wealth_age_5 wealth_age_6 wealth_inc_1 wealth_inc_2 wealth_inc_3 wealth_inc_4 wealth_inc_5 hhtotaldis_DTA_1 hhtotaldis_DTA_2 hhtotaldis_DTA_3 hhtotaldis_DTA_4 hhtotaldis_DTA_5 hhtotaldis_DTI_1 hhtotaldis_DTI_2 hhtotaldis_DTI_3 hhtotaldis_DTI_4 hhtotaldis_DTI_5 hhtotaldis_age_1 hhtotaldis_age_2 hhtotaldis_age_3 hhtotaldis_age_4 hhtotaldis_age_5 hhtotaldis_age_6 hhtotaldis_inc_1 hhtotaldis_inc_2 hhtotaldis_inc_3 hhtotaldis_inc_4 hhtotaldis_inc_5 hhdebt_DTA_1 hhdebt_DTA_2 hhdebt_DTA_3 hhdebt_DTA_4 hhdebt_DTA_5 hhdebt_DTI_1 hhdebt_DTI_2 hhdebt_DTI_3 hhdebt_DTI_4 hhdebt_DTI_5 hhdebt_age_1 hhdebt_age_2 hhdebt_age_3 hhdebt_age_4 hhdebt_age_5 hhdebt_age_6 hhdebt_inc_1 hhdebt_inc_2 hhdebt_inc_3 hhdebt_inc_4 hhdebt_inc_5 asset_to_linsup_DTA_1 asset_to_linsup_DTA_2 asset_to_linsup_DTA_3 asset_to_linsup_DTA_4 asset_to_linsup_DTA_5 asset_to_linsup_DTI_1 asset_to_linsup_DTI_2 asset_to_linsup_DTI_3 asset_to_linsup_DTI_4 asset_to_linsup_DTI_5 asset_to_linsup_age_1 asset_to_linsup_age_2 asset_to_linsup_age_3 asset_to_linsup_age_4 asset_to_linsup_age_5 asset_to_linsup_age_6 asset_to_linsup_inc_1 asset_to_linsup_inc_2 asset_to_linsup_inc_3 asset_to_linsup_inc_4 asset_to_linsup_inc_5 mpcw_DTA_1 mpcw_DTA_2 mpcw_DTA_3 mpcw_DTA_4 mpcw_DTA_5 mpcw_DTI_1 mpcw_DTI_2 mpcw_DTI_3 mpcw_DTI_4 mpcw_DTI_5 mpcw_age_1 mpcw_age_2 mpcw_age_3 mpcw_age_4 mpcw_age_5 mpcw_age_6 mpcw_inc_1 mpcw_inc_2 mpcw_inc_3 mpcw_inc_4 mpcw_inc_5 mpcy_DTA_1 mpcy_DTA_2 mpcy_DTA_3 mpcy_DTA_4 mpcy_DTA_5 mpcy_DTI_1 mpcy_DTI_2 mpcy_DTI_3 mpcy_DTI_4 mpcy_DTI_5 mpcy_age_1 mpcy_age_2 mpcy_age_3 mpcy_age_4 mpcy_age_5 mpcy_age_6 mpcy_inc_1 mpcy_inc_2 mpcy_inc_3 mpcy_inc_4 mpcy_inc_5 age_DTA_1 age_DTA_2 age_DTA_3 age_DTA_4 age_DTA_5 age_DTI_1 age_DTI_2 age_DTI_3 age_DTI_4 age_DTI_5 age_age_1 age_age_2 age_age_3 age_age_4 age_age_5 age_age_6 age_inc_1 age_inc_2 age_inc_3 age_inc_4 age_inc_5 unem_DTA_1 unem_DTA_2 unem_DTA_3 unem_DTA_4 unem_DTA_5 unem_DTI_1 unem_DTI_2 unem_DTI_3 unem_DTI_4 unem_DTI_5 unem_age_1 unem_age_2 unem_age_3 unem_age_4 unem_age_5 unem_age_6 unem_inc_1 unem_inc_2 unem_inc_3 unem_inc_4 unem_inc_5 unem__DTA_1 unem__DTA_2 unem__DTA_3 unem__DTA_4 unem__DTA_5 unem__DTI_1 unem__DTI_2 unem__DTI_3 unem__DTI_4 unem__DTI_5 unem__age_1 unem__age_2 unem__age_3 unem__age_4 unem__age_5 unem__age_6 unem__inc_1 unem__inc_2 unem__inc_3 unem__inc_4 unem__inc_5 hh_attlf_DTA_1 hh_attlf_DTA_2 hh_attlf_DTA_3 hh_attlf_DTA_4 hh_attlf_DTA_5 hh_attlf_DTI_1 hh_attlf_DTI_2 hh_attlf_DTI_3 hh_attlf_DTI_4 hh_attlf_DTI_5 hh_attlf_age_1 hh_attlf_age_2 hh_attlf_age_3 hh_attlf_age_4 hh_attlf_age_5 hh_attlf_age_6 hh_attlf_inc_1 hh_attlf_inc_2 hh_attlf_inc_3 hh_attlf_inc_4 hh_attlf_inc_5 gs_exp_ann_DTA_1 gs_exp_ann_DTA_2 gs_exp_ann_DTA_3 gs_exp_ann_DTA_4 gs_exp_ann_DTA_5 gs_exp_ann_DTI_1 gs_exp_ann_DTI_2 gs_exp_ann_DTI_3 gs_exp_ann_DTI_4 gs_exp_ann_DTI_5 gs_exp_ann_age_1 gs_exp_ann_age_2 gs_exp_ann_age_3 gs_exp_ann_age_4 gs_exp_ann_age_5 gs_exp_ann_age_6 gs_exp_ann_inc_1 gs_exp_ann_inc_2 gs_exp_ann_inc_3 gs_exp_ann_inc_4 gs_exp_ann_inc_5

*Select variables to simulate for each type of stress test
if ${hes} == 0 {
	local simulate_list `bank_stess_list'
	}

if ${hes} == 1 & ${distribution_results} == 0 {
	local simulate_list `bank_stess_list' `cons_agg_list'
	}
	
if ${hes} == 1 & ${distribution_results} == 1 {
	local simulate_list `bank_stess_list' `cons_agg_list' `cons_dist_list'
	}

*Simulate post stress 
n simulate `simulate_list', reps(${simtimes}): stress_test  		

*Calculate means of simulation variables 
local i=0
foreach x of newlist `simulate_list' {
	local i=`i'+1
	egen `x' = mean(_sim_`i')
	}

*Delete simulation variables
drop _sim*
 
*Export results
keep if _n ==1 // keep one row
gen batch = "${batch}" // labels 
gen case = "${case}" // scenario case ID

if ${iteration} == 1 {
	save "Results", replace
	}
if ${iteration} != 1 {
	append using "Results"
	save "Results", replace	
	}
	
global iteration = ${iteration} + 1	
erase "Results\HH stress test.dta"