# Title: Figures 2 &7 of RDP 2020-02
# Creator: Calvin He
# Date Created: 07 February 2020
# Description: this code runs the unrestricted model for each SA3 using different 
#               controls and monetary policy shocks

library(lfe)
library(rGertrude) # RBA package to access internal database
library(data.table)
library(zoo)
library(dummies)
library(rlang)
library(ggplot2)
library(broom)
library(lmtest)
library(sandwich)
library(lubridate)
library(tidyverse)
library(conflicted)

conflicted::conflict_prefer("select", "dplyr")
conflicted::conflict_prefer("filter", "dplyr")
conflicted::conflict_prefer("year", "lubridate")
conflicted::conflict_prefer("month", "lubridate")
conflicted::conflict_prefer("intersect", "dplyr")

# Step 0: Import functions and data ---------------------------------------

# Import Functions
user.defined.functions<- c(list.files(path = "./R", pattern = ".R",  full.names= T) )
if (length(user.defined.functions)>0){invisible(lapply(user.defined.functions, source))}

# Import Data
corelogic_data_raw <- readRDS("./data/corelogic_data_raw.RDS")
monetary_policy_shocks_raw <- readRDS("./data/monetary_policy_shocks.RDS")
grouping_variables <- readRDS("./data/grouping_variables.RDS")


# Step 1 - set options ----------------------------------------------------
opts <- set_options(figure = 2)


# Step 2 - Run Models: Aggregate controls and distribution of responses ---------
unrestricted_models <- map2(list(   c("gdp", "tot", "unemp" ), c(), c() ), 
                            list( "cash_rate_change_qtr" ,"cash_rate_change_qtr", "rr_shocks"),
                            function(x, y){
                              
                              # Set controls and monetary policy shock
                              opts$agg_controls <- x
                              opts$mp_shock_type <-  y
                              
                              # Build combined data
                              corelogic_sa3_data <- build_combined_data(corelogic_data_raw, monetary_policy_shocks_raw, grouping_variables,  opts)
                              
                              # Build regression data
                              reg_data_pre <- corelogic_sa3_data  %>%
                                arrange(region_name, Date) %>%
                                add_shock_lag(. , opts) %>% # add lag of mp shock
                                add_controls(., opts) %>%   # add in controls based on options
                                add_agg_controls(., opts ) %>% # add aggregate controls
                                add_ar_y(., opts) %>% # add AR terms if necessary
                                {if(opts$remove_gfc){ . %>% filter(Date < "2007-12-31"| Date > "2010-01-01") }else{.} } %>% # remove gfc
                                mutate(calculation_value = replace(calculation_value, calculation_value == 0 , 1)) %>%
                                arrange(region_name, Date) %>%
                                group_by(region_name) %>%
                                mutate(obs = sum(!is.na(calculation_value))) %>% # remove na
                                filter(obs > 30) # restrict to greater than 30 observations
                              
                              
                              # 3. Run Local Projections for each SA3 -------------------------------------------
                              output <- reg_data_pre  %>% 
                                group_by( STATE_NAME_2016, SA3_CODE_2016) %>% 
                                nest() %>% 
                                mutate(output = map(data, individual_reg, m = opts$horizon  )) 
                              
                              # 4. Extract Responses ----------------------------------------------------
                              
                              if(length(opts$agg_controls) > 1){
                                type <- paste0(opts$agg_controls, collapse = ",")
                                type<- "With controls"
                              } else if(length(opts$agg_controls) == 0 & opts$mp_shock_type == "cash_rate_change_qtr"){
                                type<- "Without controls"
                              } else if (length(opts$agg_controls) == 0 & opts$mp_shock_type == "rr_shocks"){
                                type <- "Romer and Romer"
                              }else{ type <- opts$agg_controls}
                              
                              
                              effects <- output %>% 
                                unnest(output) %>% 
                                filter(term == "mp_shock") %>% 
                                dplyr::select(-data) %>% 
                                mutate(mp_shock = type)
                              
                              
                              return(effects)
                            }
)

unrestricted_models <- unrestricted_models %>% bind_rows



# Step 3 ------------------------------------------------------------------

unrestricted_models <- readRDS("data/figure-data/figure-2-7.RDS") # import actual data used in the RDP


graph_data <- unrestricted_models %>% filter(horizon %in% c(8)) # select 8th qurater horizon

p <- ggplot(graph_data , aes(x= estimate, colour = mp_shock) )+
  geom_density( size = 0.75) +
  rba_theme(multipanel = T) + 
  labs(caption = "Sources: ABS; Authors' calculations; CoreLogic data; RBA") + 
  scale_x_continuous(name = "Estimate")+
  coord_cartesian(expand = FALSE) +
  scale_y_continuous(name = "Proportion", sec.axis = dup_axis()) +
  guides(colour = guide_legend( override.aes=list(size=6)))

p
