# Title: Rolling regressions
# Creator: Calvin He
# Date Created: 07 February 2020

library(lfe)
library(rGertrude) # RBA package to access internal database
library(data.table)
library(zoo)
library(dummies)
library(rlang)
library(ggplot2)
library(broom)
library(lubridate)
library(tidyverse)


# Step 0: Import functions and data ---------------------------------------

# Import Functions
user.defined.functions<- c(list.files(path = "./R", pattern = ".R",  full.names= T) )
if (length(user.defined.functions)>0){invisible(lapply(user.defined.functions, source))}

# Import Data
corelogic_data_raw <- readRDS("./data/corelogic_data_raw.RDS")
monetary_policy_shocks_raw <- readRDS("./data/monetary_policy_shocks.RDS")
grouping_variables <- readRDS("./data/grouping_variables.RDS")


# Step 1:  Choose your options --------------------------------------------
opts <- set_options(figure = 9)

# 1. Manipulate data ------------------------------------------------------------------------------------------------
# filter, group, join shock
corelogic_sa3_data <- build_combined_data(corelogic_data_raw, monetary_policy_shocks_raw, grouping_variables, opts)

# 2. prepare regression data -------------------------------------------------------------------
# add in decile interactions +  controls
reg_data_pre <- build_regression_data(corelogic_sa3_data , opts)


# 4.  Run regressions with a 15 year window -------------------------------------------------------
response_window <- map2( c(as.Date("2005-03-31"), as.Date("2019-06-30")), c(as.Date("1990-03-31"), as.Date("2005-03-31")), function(x, y){
  restricted_sample <- reg_data_pre %>% filter(Date <= x, Date >= y )
  
  output <- panel_reg_lfe(restricted_sample, 6, opts )$tidy_output %>% 
    filter(mp_shock_lag == 0) %>% 
    mutate(end_date = x)
  
  temp <-  output[1,]  %>% 
    mutate( interaction_var_level = 5 , estimate = 0)
  
  output <- bind_rows(output, temp)
  
  return(output)
} )

graph_data <- response_window %>% bind_rows() %>% 
  filter(!str_detect(term, "agg")) %>% 
  mutate(ci_upper = estimate + 1.96*std.error,
         ci_lower =  estimate - 1.96*std.error) %>%  
  mutate(significant = case_when(p.value > 0.05 & p.value <= 0.1 ~ "10% level", 
                                 p.value > 0.01 & p.value <= 0.05 ~ "5% level",
                                 p.value <= 0.01 ~ "1% level", 
                                 TRUE ~ "Insignificant")) 




# Prepare plot data --------------------------------------------------------------------
graph_data <- response_window %>% bind_rows() %>% 
  filter(!str_detect(term, "agg")) %>% 
  mutate(ci_upper = estimate + 1.96*std.error,
         ci_lower =  estimate - 1.96*std.error) %>%  
  mutate(significant = case_when(p.value > 0.05 & p.value <= 0.1 ~ "10% level", 
                                 p.value > 0.01 & p.value <= 0.05 ~ "5% level",
                                 p.value <= 0.01 ~ "1% level", 
                                 TRUE ~ "Insignificant")) 


graph_data <- readRDS("./data/figure-data/figure-9.RDS") # read in actual data


# Plot --------------------------------------------------------------------
# Map colours 
colour_map <- tibble(significant = c("1% level", "10% level", "5% level",  "Insignificant" ),
                     colour = c(   "dodgerblue", alpha("dodgerblue", 0.45),  alpha("dodgerblue", 0.75),   "orange")) %>% 
              filter(significant %in% unique(graph_data$significant))


ggplot(graph_data, aes(x=interaction_var_level, y = estimate, fill = significant ))+
  facet_wrap(~ end_date) + 
  geom_col() +
  rba_theme(multipanel = T) +
  scale_fill_manual(values = colour_map$colour  )+
  scale_y_continuous(name = "%", sec.axis = dup_axis())+
  scale_x_continuous(name = "Price Group", breaks = seq(1,10, by = 1)) +
  labs(caption = "Sources: ABS, Authors' calculations, CoreLogic data, RBA")
