##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Monetary Policy Effects from SVAR: Bishop and Tulip (2017) policy
#       shock vs BT shock purged of response to creadit spreads (BT-CS shock)
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates 2 SVARs to obtain the macroeconomic effects
#              of a cash rate change to create Figure 7 of the paper.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Figure 7 (for RBA users only, PNG-format)
#          2. Data for Figure 7 in Excel-file.
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Load functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer", "plyr",
                          'vars', "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
spec_seed <- 2304
horizon <- 20
cumirf <- FALSE
alpha <- 0.90               # Confidence interval
varp <- 4                     # VAR lag length 

## Endogenous Variables and Policy Shocks
endo_names <- c("cpii", "gdp", "ur")
endo_transform <- c("ln", "ln", "l")
endo_vars <- data.frame(endo_names, endo_transform, stringsAsFactors=FALSE)
colnames(endo_vars) <- c("variable", "transform")
endo_vars <- mutate(endo_vars, tf_variable = ifelse(transform=="l", variable,
                                                    paste(transform, variable, sep="_")))

shocks <- c("rr_fw", "rr_fw_csj")
# Order of variables (1 or length(endo_names)+1)
pos_shock <- length(endo_names)+1

### Sample
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")


##### Step 4: Load and clean data and construct missing variables #############
### Policy Shocks
load(paste0(results_dir, "rr_shocks.RData"))
if (any(str_detect(shocks, "unant"))) {
  load(paste0(results_dir, "rr_shocks_unant.RData"))
  rr_shocks <- left_join(rr_shocks, rr_shocks_unant, by="Date")
}
# Select shocks
rr_shocks <- select(rr_shocks, Date, !!shocks)
# Cumulate shocks
rr_shocks <- mutate_at(rr_shocks, vars(-Date), cumsum)


### VAR data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, !!endo_names, fc_mth)

# Keep forecast months only
Y <- filter(Y, fc_mth==1) %>% 
  select(-fc_mth)

# Transform endogenous variables
Y <- transform_df(Y, endo_vars)

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)

### Merge shocks and VAR data
Y <- left_join(Y, rr_shocks, by="Date")


##### Step 5: SVAR Regressions ################################################
for (j in 1:length(shocks)) {
  # Select variables
  mp_shk <- shocks[j]
  y <- select(Y, !!endo_vars$tf_variable, !!mp_shk)
  y[is.na(y), mp_shk] <- 0
  
  # Estimate VAR
  var_est <- VAR(y, varp, type = "const") # , exogen = x
  
  # Obtain IRF
  irf_est <- irf(var_est, impulse = mp_shk, cumulative = cumirf, boot = TRUE,
                 n.ahead = horizon, ortho = TRUE, ci = alpha, runs = 5000, seed = spec_seed)
  
  # Standardise and bind together
  IRFs <- as.data.frame(irf_est$irf[[1]]/irf_est$irf[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    mutate(horizon = seq(0,horizon))
  IRFs_lwr <- as.data.frame(irf_est$Lower[[1]]/irf_est$Lower[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    rename_all(function(x) paste0(x,"_lwr")) %>% 
    mutate(horizon = seq(0,horizon))
  IRFs_upr <- as.data.frame(irf_est$Upper[[1]]/irf_est$Upper[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    rename_all(function(x) paste0(x,"_upr")) %>% 
    mutate(horizon = seq(0,horizon))
  IRFs <- join_all(list(IRFs, IRFs_lwr, IRFs_upr), by = "horizon", type = "full")
  # Obtain percentage point/basis point changes
  IRFs <- mutate_at(IRFs, vars(contains("ln")), function(x) {100*x})
  
  # Transform to long format and re-order columns by variable names
  IRFs_lg <- pivot_longer(IRFs, -horizon, names_to = "variable", values_to = "value") %>% 
    arrange(variable, horizon) %>% 
    mutate(shock = mp_shk)
  
  if (j==1) {
    IRFs_all_lg <- IRFs_lg
  } else {
    IRFs_all_lg <- bind_rows(IRFs_all_lg, IRFs_lg)
  }
}
IRFs_all <- pivot_wider(IRFs_all_lg, names_from = c(variable, shock), names_sep = "_", 
                        values_from = value)

##### Step 6: Plot IRFs #######################################################
### CPI, Unemployment, GDP
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig7 <- arphitgg(aes = agg_aes(x = horizon, y = value, group = variable), layout = "3b2"
                   , dropxlabel = TRUE, showallxlabels = FALSE) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ln_cpii') & shock == "rr_fw"), panel = "1",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ln_cpii') & shock == "rr_fw_csj"), panel = "2",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ur') & shock == "rr_fw"), panel = "3",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ur') & shock == "rr_fw_csj"), panel = "4",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ln_gdp') & shock == "rr_fw"), panel = "5",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_line(data = IRFs_all_lg %>% filter(str_detect(variable, 'ln_gdp') & shock == "rr_fw_csj"), panel = "6",
             colour = c(RBA["Aqua8"], RBA["Red6"], RBA["Red6"]), lty = c(1,2,2), lwd = rep(3,3)) + 
    agg_yaxislabel("CPI", panel = "1") + 
    agg_yaxislabel("Unemployment", panel = "3") + 
    agg_yaxislabel("Real GDP", panel = "5") + 
    agg_title("BT Shock**", panel = "1") +
    agg_title("BT-CS Shock***", panel = "2") +
    agg_units("ppt", panel = c("3","4")) +
    agg_ylim(-2, 3, 6, panel = c("1","2")) +
    agg_ylim(-1, 1, 5, panel = c("3","4")) +
    agg_ylim(-3, 2, 6, panel = c("5","6")) +
    agg_title("Macroeconomic Effects of a Contractionary Monetary Policy Shock") +
    agg_subtitle("Cumulative quarterly responses to 100 bps cash rate shock*") +
    agg_source("ABS") +
    agg_source("Author's calculations") +
    agg_source("RBA") + 
    agg_footnote("Impulse responses from a recursive SVAR(4) including log CPI (underlying), log real GDP, the unemployment rate, and the cumulated policy shock; dashed lines show 90% confidence bands (bootstrapped); 1994:Q1-2018:Q4.") + 
    agg_footnote("Bishop and Tulip (2017, updated) policy shock; residual from column (BT) of Table 1.") +
    agg_footnote("BT Shock accounting for cash rate response to credit spreads; residual from column (BT-CS) of Table 1.")
  agg_draw(fig7, filename = paste0(results_dir, "Fig7_SVAR_BT-CS.png"))
}

### Export graph data
IRFs_all <- filter(IRFs_all_lg, str_detect(variable, 'rr_fw', negate = T)) %>% 
  mutate(label = case_when((variable=='ln_cpii' & shock == 'rr_fw') ~ "CPI (BT Shock)", 
                           (variable=='ln_cpii_lwr' & shock == 'rr_fw') ~ "CPI Lower (BT Shock)", 
                           (variable=='ln_cpii_upr' & shock == 'rr_fw') ~ "CPI Upper (BT Shock)",
                           (variable=='ln_cpii' & shock == 'rr_fw_csj') ~ "CPI (BT-CS Shock)", 
                           (variable=='ln_cpii_lwr' & shock == 'rr_fw_csj') ~ "CPI Lower (BT-CS Shock)", 
                           (variable=='ln_cpii_upr' & shock == 'rr_fw_csj') ~ "CPI Upper (BT-CS Shock)",
                           (variable=='ur' & shock == 'rr_fw') ~ "UR (BT Shock)", 
                           (variable=='ur_lwr' & shock == 'rr_fw') ~ "UR Lower (BT Shock)", 
                           (variable=='ur_upr' & shock == 'rr_fw') ~ "UR Upper (BT Shock)",
                           (variable=='ur' & shock == 'rr_fw_csj') ~ "UR (BT-CS Shock)", 
                           (variable=='ur_lwr' & shock == 'rr_fw_csj') ~ "UR Lower (BT-CS Shock)", 
                           (variable=='ur_upr' & shock == 'rr_fw_csj') ~ "UR Upper (BT-CS Shock)",
                           (variable=='ln_gdp' & shock == 'rr_fw') ~ "GDP (BT Shock)", 
                           (variable=='ln_gdp_lwr' & shock == 'rr_fw') ~ "GDP Lower (BT Shock)", 
                           (variable=='ln_gdp_upr' & shock == 'rr_fw') ~ "GDP Upper (BT Shock)",
                           (variable=='ln_gdp' & shock == 'rr_fw_csj') ~ "GDP (BT-CS Shock)", 
                           (variable=='ln_gdp_lwr' & shock == 'rr_fw_csj') ~ "GDP Lower (BT-CS Shock)", 
                           TRUE ~ "GDP Upper (BT-CS Shock)")) %>% 
  select(-c(variable, shock)) %>% 
  pivot_wider(names_from = label, values_from = value)
write_xlsx(IRFs_all, path = paste(results_dir, "Fig7_SVAR_BT-CS.xlsx", sep=""), col_names = TRUE)
