##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Romer and Romer Monetary Policy Shocks for Australia:
#       The Role of Credit Spreads - Subsample Regressions
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates Romer and Romer (2004, RR) type regressions 
#              for Australia for 4 different samples.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Table 7: Regression results from Romer and Romer Regression
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Source functions
source("Code/functions/funcs.R")
source("Code/functions/rr_regress.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer",
                          "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
dep_var <- "cr_chg"       # Dependent LHS variable (change in cash rate)
lag_dep_var <- "cr"       # Lagged dependent variable (from cr, cr_chg)
adlp <- 1                 # Number of lags of lagged dependent variable

## RBA forecasts and revisions
fcast_vars <- c("fccpiiq2", "fcgdpq2", "fcurq0", "fcrcpiiq2", "fcrgdpq2")
# "fc" for forecast, "fcr" for forecast revision, "q'h'" for quarterly horizon
# "cpii" for underlying inflation, "gdp" for GDP, "ur" for level of unemployment rate, 
# "dur" for change in unemployment rate

## Credit spreads and risk measures
cs_vars <- c("sprd_ir_bab_ois3", "sprd_ir_lbiz_bab3", "sprd_ir_us_cbb_gb10")

## Cash rate expectations (i.e. yield curve)
cr_exp <- "sprd_yc_ois3"
# Excepted change in cash rate over next 3 months


### Sample
# Frequency (m or q)
freq <- "q"
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
smpl_str <- paste(format(as.yearqtr(smpl_start), format = "%Y:Q%q"), "-", 
                 format(as.yearqtr(smpl_end), format = "%Y:Q%q"))
# Sample omission (specify start and end date)
smpl_omit <- c(as.Date("2008-06-01"), as.Date("2009-06-01"))  # e.g.: as.Date("2008-06-01"), as.Date("2009-06-01")


##### Step 4: Load and clean data and construct missing variables #############
# Load data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, fc_mth, !!dep_var, !!lag_dep_var, !!fcast_vars, !!cr_exp, !!cs_vars)

# Take end-of-PREVIOUS-month financial variables (Board Meeting is at beginning of month)
Y <- Y %>% 
  mutate_at(vars(cr_exp, cs_vars), lag)

# Retain adlp lags of *monthly* dependent variable
ldep_vars <- c()
if (adlp>0){
  for (p in 1:adlp) {
    var_name <- paste0("l", as.character(p), "_", lag_dep_var)
    Y <- mutate(Y, !!var_name := lag(!!as.name(lag_dep_var), n = p))
    ldep_vars <- c(ldep_vars, var_name)
  }
}
rm(var_name)

# Keep forecast months only
if (freq=="q") {
  Y <- filter(Y, fc_mth==1)
}

### Add GFC Dummy
Y <- mutate(Y, gfc_dum = 1*(Date>=smpl_omit[1] & Date<=smpl_omit[2]))

### Keep maximum permissible sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)

### Regression Specifications
for (spec in 1:6) {
  ## Credit spreads and risk measures
  cs_vars <- c("sprd_ir_bab_ois3", "sprd_ir_lbiz_bab3", "sprd_ir_us_cbb_gb10")
  ### Keep only selected sample
  if (spec==1) {
    # Full
    start <- smpl_start
    end <- smpl_end
    omit <- c()
    reg_name <- "rr_csj"
  } else if (spec==2) {
    # GFC Dummy
    start <- smpl_start
    end <- smpl_end
    omit <- c()
    cs_vars <- c(cs_vars, "gfc_dum")
    reg_name <- "rr_csj_gfc_dum"
  } else if (spec==3) {
    # Excl. GFC
    start <- smpl_start
    end <- smpl_end
    omit <- smpl_omit
    reg_name <- "rr_csj_gfc_ex"
  } else if (spec==4) {
    # Pre GFC
    start <- smpl_start
    end <- smpl_omit[1]
    omit <- c()
    reg_name <- "rr_csj_sub1"
  } else if (spec==5) {
    # Post GFC
    start <- smpl_omit[2]
    end <- smpl_end
    omit <- c()
    reg_name <- "rr_csj_sub2"
  } else if (spec==6) {
    # Excluding GFC Benchmark BT Regression
    start <- smpl_start
    end <- smpl_end
    omit <- smpl_omit
    cs_vars <- c()
    reg_name <- "rr_fw_gfc_ex"
  }
  y <- filter(Y, Date>=start & Date<=end)
  if (length(omit)>1) {
    y <- filter(y, !(Date>omit[1] & Date<omit[2]))
  }
  ##### Step 5: RR regressions ################################################
  assign(reg_name, rr_regress(y, dep_var, c(ldep_vars, fcast_vars, cs_vars)))
}

### Collect results
rr_list <- ls(pattern = "rr_csj|rr_fw")

# Collect fitted values and shocks series (residuals), and estimated models
y_fit <- select(Y, cr_chg) %>% 
  zoo(Y$Date)
rr_shocks <- select(Y, cr_chg) %>% 
  zoo(Y$Date)
rr_regs <- list()
for (j in 1:length(rr_list)) {
  y_fit <- cbind(y_fit, get(eval(rr_list[j]))$y_fit)
  rr_shocks <- cbind(rr_shocks, get(eval(rr_list[j]))$rr_shocks)
  colnames(y_fit)[ncol(y_fit)] <- rr_list[j]
  colnames(rr_shocks)[ncol(rr_shocks)] <- rr_list[j]
  assign(rr_list[j], get(eval(rr_list[j]))$rr_est)
  rr_regs[[j]] <- get(eval(rr_list[j]))
}


##### Step 6: Regression results ##############################################
# Name columns
collab <- c("GFC Dummy", "Excl. GFC", "Pre-GFC", "Post-GFC")
# Table 7: Estimated Policy Rules with Credit Spreads
stargazer(rr_regs[2:5], title = "Estimated Policy Rules with Credit Spreads - Subsample Evidence", 
          align = TRUE, type = "text", no.space = TRUE, dep_var.caption = "", 
          column.labels = collab, model.numbers = FALSE, omit.stat = c("ser", "f"), 
          digits = 2, dep_var.labels = smpl_str, intercept.bottom = FALSE, intercept.top = TRUE)
rr_shocks <- as_tibble(cbind(Y$Date, as.data.frame(rr_shocks)))
colnames(rr_shocks)[1] <- "Date"


##### Step 7: Remove anticipation effects #####################################
Y2 <- left_join(Y, rr_shocks, by="Date")
y.zoo <- select(Y2, -Date)
y.zoo <- zoo(y.zoo, Y$Date)

## Second stage regression
shocks_main <- "rr_csj_gfc_ex"
for (j in 1:length(shocks_main)) {
  # Second-stage regression of RR shocks on cash rate expectations
  eq_rr_ant <- paste0(shocks_main[j], " ~ L(", cr_exp, ", 0)")
  rr_regress_ant <- dynlm(as.formula(eq_rr_ant), y.zoo)
}
## Add to shock series
rr_shocks_unant <- rownames_to_column(as.data.frame(rr_regress_ant$residuals), "Date")
colnames(rr_shocks_unant)[2] <- paste0(shocks_main, "_unant")
rr_shocks_unant$Date <- as.Date(rr_shocks_unant$Date)
rr_shocks <- left_join(rr_shocks, rr_shocks_unant, by="Date")


##### Step 8: Save output #####################################################
save(list = c("Y", "rr_shocks"), file = paste0(results_dir, "rr_shocks_subsample.RData"))
write_xlsx(rr_shocks, path = paste(results_dir, "rr_shocks_subsample.xlsx", sep=""), col_names = TRUE)
