##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Romer and Romer Monetary Policy Shocks for Australia:
#       Are RR-Shocks Anticipated?
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file purges estimated Romer and Romer type policy shocks 
#              of market expectations of next quarter cash rate changes.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Table 3: Second-stage regression of RR-type shocks on expected
#             cash rate changes
#          2. Figure 5: Scatterplot of RR-type shocks and expected cash rate changes
#          3. Figure 8: Anticipated and Unanticipated Monetary Policy Shocks
#          4. R- and XLSX-files with 'unanticipated' RR-type policy shocks
###############################################################################


##### Step 1: Set Directories #################################################
### Get working directory
wd <- getwd()

### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Source functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer",
                          "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
shocks <- c("rr_fw", "rr_fw_csj")       # Romer and Romer shocks to be purged

## Cash rate expectations (i.e. yield curve)
cr_exp <- "sprd_yc_ois3"
# Excepted change in cash rate over next 3 months


##### Step 4: Load and clean data and construct missing variables #############
# Load data
load(paste0(results_dir, "rr_shocks.RData"))

# Select variables from dataset
Y <- select(Y, Date, !!cr_exp)

### Sample
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
smpl_str <- paste(format(as.yearqtr(smpl_start), format = "%Y:Q%q"), "-", 
                  format(as.yearqtr(smpl_end), format = "%Y:Q%q"))


### Merge data for second-stage regression
# Residuals from first-stage RR regression
Y <- left_join(Y, rr_shocks, by="Date")
y.zoo <- select(Y, -Date)
y.zoo <- zoo(y.zoo, Y$Date)


##### Step 5: (Un-)Anticipated Monetary Policy Shocks #########################
### Second-stage regression: Purge residuals of anticipated cash rate changes
rr_regress_ant <- list()
for (j in 1:length(shocks)) {
  # Second-stage regression of RR shocks on cash rate expectations
  eq_rr_ant <- paste0(shocks[j], " ~ L(", cr_exp, ", 0)")
  rr_regress_ant[[j]] <- dynlm(as.formula(eq_rr_ant), y.zoo)
}


### Table 3: Monetary Policy Shocks and Cash Rate Expectations
stargazer(rr_regress_ant, title = "Monetary Policy Shocks and Cash Rate Expectations", 
          align = TRUE, type = "text", no.space = TRUE, dep_var.caption = "", 
          column.labels = c("(BT)", "(BT-CS2)"),
          model.numbers = FALSE, omit.stat = c("ser", "f", "adj.rsq"), digits = 3, 
          dep_var.labels = smpl_str, intercept.bottom = FALSE, intercept.top = TRUE)


### Extract residuals from second-stage regression as unanticipated shocks
rr_shocks_unant <- as.data.frame(cbind(rr_regress_ant[[1]]$residuals, rr_regress_ant[[2]]$residuals))
rr_shocks_unant <- cbind(Y$Date, rr_shocks_unant)
colnames(rr_shocks_unant) <- c("Date", paste0(shocks[1], "_unant"), paste0(shocks[2], "_unant"))
rr_shocks_unant <- as_tibble(rr_shocks_unant)


##### Step 6: Plot Results ####################################################
### Figure 5: Monetary Policy Shocks and Expected Cash Rate Changes
data_fig5 <- as.data.frame(y.zoo)
# Anticipated policy shocks from first-stage regression
data_fig5 <- select(data_fig5, cr_exp, rr_fw, rr_fw_csj)
# Fitted values from second-stage regression
data_fig5 <- cbind(data_fig5, as.data.frame(rr_regress_ant[[1]]$fitted.values), 
                   as.data.frame(rr_regress_ant[[2]]$fitted.values))
colnames(data_fig5)[(ncol(data_fig5)-1):ncol(data_fig5)] <- c("rr_fw_ant_fit", "rr_fw_csj_ant_fit")
dl_fig5 <- data_fig5 %>%
  pivot_longer(-!!cr_exp, names_to = "rr_shock", values_to = "value") %>% 
  ungroup() %>% 
  arrange(rr_shock, sprd_yc_ois3, value)
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig5 <- arphitgg(aes = agg_aes(x = sprd_yc_ois3, y = value, group = rr_shock)) + 
    agg_point(data = dl_fig5 %>% filter(rr_shock == cr_exp | rr_shock == "rr_fw" | rr_shock == "rr_fw_csj"),
              panel = "1", colour = c(RBA["Blue7"], RBA["Orange3"])) +
    agg_line(data = dl_fig5 %>% filter(rr_shock == cr_exp | rr_shock == "rr_fw_ant_fit" | rr_shock == "rr_fw_csj_ant_fit"),
             panel = "1", colour = c(RBA["Blue7"], RBA["Orange3"])) +
    agg_units("ppt", panel = "1") +
    agg_xunits("ppt") +
    agg_yaxislabel("Residual") +
    agg_xaxislabel("Expected cash rate change") +
    agg_ylim(-2, 2, 5) +
    agg_xlim(-2, 2) +
    agg_label("Original (BT)**", x = -0.8, y = -0.8, colour = RBA["Blue7"], panel = "1") +
    agg_label("Augmented (BT-CS)***", x = 1.2, y = -0.5, colour = RBA["Orange3"], panel = "1") +
    agg_title("Monetary Policy Shocks and Expected Cash Rate Changes") +
    agg_subtitle("1994:Q1 - 2018:Q4*") +
    agg_footnote("Lines of best fit from equation (16).") +
    agg_footnote("Residuals from column BT of Table 1.") +
    agg_footnote("Residuals from column BT-CS of Table 1.") +
    agg_source("Author's calculations; Federal Reserve Bank of St. Louis; RBA")
  agg_draw(fig5, filename = paste0(results_dir, "Fig5_Shocks_Expectations.png"))
}
dl_fig5 <- dl_fig5 %>% 
  mutate(label = case_when(rr_shock=="rr_fw" ~ "Original (BT)", 
                           rr_shock=="rr_fw_csj" ~ "Augmented (BT-CS)", 
                           rr_shock=="rr_fw_ant_fit" ~ "Original (fitted line)", 
                           TRUE ~ "Augmented (fitted line)")) %>% 
  group_by(rr_shock) %>% 
  mutate(Observation = order(order(sprd_yc_ois3, value))) %>% 
  ungroup %>% 
  select(Observation, sprd_yc_ois3, value, label)
data_fig5 <- pivot_wider(dl_fig5, names_from = label, values_from = value)
colnames(data_fig5)[2] <- "Expected cash rate change"
write_xlsx(data_fig5, path = paste(results_dir, "Fig5_Shocks_Expectations.xlsx", sep=""), 
           col_names = TRUE)


### Figure 8: Unanticipated Monetary Policy Shocks
data_fig8 <- select(rr_shocks, Date, cr_chg, !!shocks) %>% 
  left_join(rr_shocks_unant, by="Date")
dl_fig8 <- data_fig8 %>%
  select(Date, cr_chg, rr_fw, rr_fw_unant, rr_fw_csj, rr_fw_csj_unant) %>% 
  pivot_longer(-Date, names_to = "variable", values_to = "value")
if ("arphit" %in% installed.packages()[,"Package"]) {
  fig8 <- arphitgg(layout = "2h", aes = agg_aes(x = Date, y = value, group = variable)) + 
    agg_line(panel = "1", data = dl_fig8 %>% filter(variable == "cr_chg" | variable == shocks[1] | variable == paste0(shocks[1], "_unant")), 
             colour = c(RBA["Grey10"], RBA["Blue7"], RBA["Purple2"]), lty = c(1,2,2)) +
    agg_line(panel = "3", data = dl_fig8 %>% filter(variable == "cr_chg" | variable == shocks[2] | variable == paste0(shocks[2], "_unant")), 
             colour = c(RBA["Grey10"], RBA["Orange3"], RBA["Brown3"]), lty = c(1,2,2)) +
    agg_units("ppt", panel = "1") +
    agg_units("ppt", panel = "3") +
    agg_ylim(-1, 1, 5) +
    agg_title("Unanticipated Monetary Policy Shocks") +
    agg_subtitle(paste0("Quarterly, ", smpl_str)) +
    agg_title("BT Shock", panel = "1") +
    agg_title("BT-CS Shock", panel = "3") +
    agg_label("Actual", x = 1996.5, y = 0.7, colour = RBA["Grey10"], 
              panel = "3") +
    agg_label("Original (BT)", x = 2005.5, y = -0.7, colour = RBA["Blue7"], 
              panel = "1") +
    agg_label("Unanticipated*", x = 2014, y = 0.4, colour = RBA["Purple2"], 
              panel = "1") +
    agg_label("Augmented (BT-CS)", x = 2004, y = -0.7, colour = RBA["Orange3"], 
              panel = "3") +
    agg_label("Unanticipated*", x = 2014, y = 0.45, colour = RBA["Brown3"], 
              panel = "3") +
    agg_source("Author's calculations") +
    agg_source("Federal Reserve Bank of St. Louis") +
    agg_source("RBA") + 
    agg_footnote("Monetary policy shocks purged of the expected cash rate change ahead of the Board meeting. See Figure 6 for further notes.")
  agg_draw(fig8, filename = paste0(results_dir, "Fig8_Shocks_Unanticipated.png")) # 
}
dl_fig8 <- dl_fig8 %>% 
  mutate(label = case_when(variable=="cr_chg" ~ "Actual", 
                           variable=="rr_fw" ~ "Original (BT)", 
                           variable=="rr_fw_unant" ~ "Unanticipated (BT)", 
                           variable=="rr_fw_csj" ~ "Augmented (BT-CS)",
                           TRUE ~ "Unanticipated (BT-CS)"))
data_fig8 <- select(dl_fig8, -c(variable)) %>% 
  pivot_wider(id_cols = Date, names_from = label, values_from = value)
write_xlsx(data_fig8, path = paste(results_dir, "Fig8_Shocks_Unanticipated.xlsx", sep=""), 
           col_names = TRUE)


##### Step 7: Save output #####################################################
save(list = c("rr_shocks_unant"), file = paste0(results_dir, "rr_shocks_unant.RData"))
colnames(rr_shocks_unant) <- c("Date", "BT (Unanticipated)", "BT-CS (Unanticipated)")
write_xlsx(rr_shocks_unant, path = paste(results_dir, "rr_shocks_unant.xlsx", sep=""), col_names = TRUE)
