##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Load Australian Bond Data
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file loads Australian government and corporate bond data
###############################################################################


##### Step 1: Load Australian Government Bond Data ############################
### Read the historical datasets
gov_bond <- read_excel(paste0(data_dir, "govt_bonds.xlsx"), skip = 9, col_names = T,
                       col_types = c("date", "numeric", "numeric", "numeric", "numeric"))
gov_bond <- filter(gov_bond, Date>=smplstart)

### Download the most recent data
tmp_file <- tempfile()
download.file('https://www.rba.gov.au/statistics/tables/xls/f02d.xls',
              tmp_file, mode = 'wb')
gov_bond_new <- read_excel(tmp_file, col_names = T, skip = 10)

### Clean the data
# Rename and select the variables of interest
gov_bond_new <- gov_bond_new %>%
  rename(Date = `Series ID`) %>% 
  select(matches("(Date.*)|(FCMYGBAG\\d)"))

colnames(gov_bond_new)[2:ncol(gov_bond_new)] <- 
  lookup_rename(gov_bond_new, metadata, "Mnemonic", "Series ID")[["Mnemonic"]]

gov_bond <- bind_rows(gov_bond, gov_bond_new)
rm(gov_bond_new)

# Remove rows where all the observations are NA
gov_bond <- filter_if(gov_bond, is.numeric, any_vars(!is.na(.)))

# Drop time
gov_bond <- mutate(gov_bond, Date = as_date(Date))

gov_bond_names <- names(gov_bond)


##### Step 2: Load Australian Lending Rates ###################################
### Download the most recent data
tmp_file <- tempfile(fileext = ".xls")
download.file('https://www.rba.gov.au/statistics/tables/xls/f05hist.xls',
              tmp_file, mode = 'wb')

ir_lend <- read_excel(tmp_file, col_names = T, skip = 10)

### Clean the Data
ir_lend <- ir_lend %>%
  rename(Date = `Series ID`) %>%
  filter(Date>=smplstart) %>%
  filter_if(is.numeric, any_vars(!is.na(.)))

### Keep only selected variables
ir_lend_names <- select(metadata, `Series ID`) %>% 
  filter(str_detect(`Series ID`, "FILR"))
ir_lend_names <- ir_lend_names[["Series ID"]]

ir_lend <- ir_lend %>% 
  select(c("Date", ir_lend_names))
colnames(ir_lend)[2:ncol(ir_lend)] <- 
  lookup_rename(ir_lend, metadata, "Mnemonic", "Series ID")[["Mnemonic"]]

# Drop time
ir_lend <- mutate(ir_lend, Date = as_date(Date))

ir_lend_names <- names(ir_lend)


##### Step 3: Load Australian Corporate Bond Rates ############################
### Read in the data
tmp_file <- tempfile()
download.file('https://www.rba.gov.au/statistics/tables/xls/f03hist.xls',
              tmp_file, mode = 'wb')

corp_bond <- read_excel(tmp_file, col_names = T, skip = 10) 

### Clean the data
corp_bond <- corp_bond %>%
  rename(Date = `Series ID`) %>%
  mutate(Date = as.Date(Date, format = "%d/%m/%Y")) %>%
  mutate_at(vars(starts_with("FN")), list(~ as.numeric(.)))

### Keep only selected variables and rename
corp_bond_names <- select(metadata, `Series ID`) %>% 
  filter(str_detect(`Series ID`, "FNF"))
corp_bond_names <- corp_bond_names[["Series ID"]]
corp_bond <- corp_bond %>%
  select(c("Date", corp_bond_names))
colnames(corp_bond)[2:ncol(corp_bond)] <- 
  lookup_rename(corp_bond, metadata, "Mnemonic", "Series ID")[["Mnemonic"]]

corp_bond_names <- names(corp_bond)


##### Step 5: Save data #######################################################
save(list = c("gov_bond", "ir_lend", "corp_bond"), 
     file = file.path(data_dir, "aus_ir.RData"))
