##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Monetary Policy Effects from SVARs: The role of individual credit 
#       spread measures for the estimated effects of a cash rate change
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file estimates the SVARs to obtain the results to create 
#              Figure D4 of the paper.
# User inputs: Set directories in Step 1
#              Specify regression in Step 3
# Outputs: 1. Figure D4 (for RBA users only, PNG-format)
#          2. Data for Figure D4 in Excel-file.
###############################################################################


##### Step 1: Set Directories #################################################
### Set the data directory
data_dir <- "Data/"
results_dir <- "Results/"


##### Step 2: Load functions and packages #####################################
# Load functions
source("Code/functions/funcs.R")

# Load necessary packages
rbaverse::update()
check_and_install_packs(c("arphit", "dynlm", "lubridate", "stargazer", "plyr",
                          'vars', "writexl", "tidyverse"))


##### Step 3: Set regression specifications ###################################
spec_seed <- 2304
horizon <- 20
cumirf <- FALSE
alpha <- 0.90               # Confidence interval
varp <- 4                     # VAR lag length 

## Endogenous Variables and Policy Shocks
endo_names <- c("cpii", "gdp", "ur")
endo_transform <- c("ln", "ln", "l")
endo_vars <- data.frame(endo_names, endo_transform, stringsAsFactors=FALSE)
colnames(endo_vars) <- c("variable", "transform")
endo_vars <- mutate(endo_vars, tf_variable = ifelse(transform=="l", variable,
                                                    paste(transform, variable, sep="_")))

shocks <- c("rr_fw", "rr_fw_cs1", "rr_fw_cs2", "rr_fw_cs3", "rr_fw_cs4", "rr_fw_csj")
# Order of variables (1 or length(endo_names)+1)
pos_shock <- length(endo_names)+1

### Sample
# Sample start and end
smpl_start <- as.Date("1994-01-01")
smpl_end <- as.Date("2018-12-31")
# Sample omission (specify start and end date)
smpl_omit <- c()  # e.g.: as.Date("2008-06-01"), as.Date("2009-06-01")


##### Step 4: Load and clean data and construct missing variables #############
### Policy Shocks
load(paste0(results_dir, "rr_shocks.RData"))
if (any(str_detect(shocks, "unant"))) {
  load(paste0(results_dir, "rr_shocks_unant.RData"))
  rr_shocks <- left_join(rr_shocks, rr_shocks_unant, by="Date")
}
# Select shocks
rr_shocks <- select(rr_shocks, Date, !!shocks)
# Cumulate shocks
rr_shocks <- mutate_at(rr_shocks, vars(-Date), cumsum)


### VAR data
load(paste0(data_dir, "data_final.RData"))

# Select variables from dataset
Y <- data_mthly %>% 
  select(Date, !!endo_names, fc_mth)

# Keep forecast months only
Y <- filter(Y, fc_mth==1) %>% 
  select(-fc_mth)

# Transform endogenous variables
Y <- transform_df(Y, endo_vars)

# Keep only selected sample
Y <- filter(Y, Date>=smpl_start & Date<=smpl_end)
if (length(smpl_omit)>1) {
  Y <- filter(Y, Date>=smpl_omit[1] & Date<=smpl_omit[2])
}

### Merge shocks and VAR data
Y <- left_join(Y, rr_shocks, by="Date")

##### Step 5: SVAR Regressions ################################################
for (j in 1:length(shocks)) {
  # Select variables
  mp_shk <- shocks[j]
  y <- select(Y, !!endo_vars$tf_variable, !!mp_shk)
  y[is.na(y), mp_shk] <- 0
  
  # Estimate VAR
  var_est <- VAR(y, varp, type = "const") # , exogen = x
  
  # Obtain IRF
  irf_est <- irf(var_est, impulse = mp_shk, cumulative = cumirf, boot = TRUE,
                 n.ahead = horizon, ortho = TRUE, ci = alpha, runs = 5000, seed = spec_seed)
  
  # Standardise and bind together
  IRFs <- as.data.frame(irf_est$irf[[1]]/irf_est$irf[[1]][1,mp_shk]) %>% 
    as_tibble() %>% 
    mutate(horizon = seq(0,horizon))
  if (str_detect(mp_shk, "[0-9]", negate=T)) {
    IRFs_lwr <- as.data.frame(irf_est$Lower[[1]]/irf_est$Lower[[1]][1,mp_shk]) %>% 
      as_tibble() %>% 
      rename_all(function(x) paste0(x,"_lwr")) %>% 
      mutate(horizon = seq(0,horizon))
    IRFs_upr <- as.data.frame(irf_est$Upper[[1]]/irf_est$Upper[[1]][1,mp_shk]) %>% 
      as_tibble() %>% 
      rename_all(function(x) paste0(x,"_upr")) %>% 
      mutate(horizon = seq(0,horizon))
    IRFs <- join_all(list(IRFs, IRFs_lwr, IRFs_upr), by = "horizon", type = "full")
  }
  # Obtain percentage point/basis point changes
  IRFs <- mutate_at(IRFs, vars(contains("ln")), function(x) {100*x})
  
  # Transform to long format and re-order columns by variable names
  IRFs_lg <- pivot_longer(IRFs, -horizon, names_to = "variable", values_to = "value") %>% 
    arrange(variable, horizon) %>% 
    mutate(shock = mp_shk)
  
  if (j==1) {
    IRFs_all_lg <- IRFs_lg
  } else {
    IRFs_all_lg <- bind_rows(IRFs_all_lg, IRFs_lg)
  }
}
IRFs_all <- pivot_wider(IRFs_all_lg, names_from = c(variable, shock), names_sep = "_", 
                        values_from = value)
IRF_plot_lg <- IRFs_all %>% 
  select(matches('horizon|cpii|ur')) %>% 
  pivot_longer(-horizon, names_to = "variable", values_to = "value") %>% 
  arrange(variable, horizon)

##### Step 6: Plot IRFs #######################################################
### CPI and UR
if ("arphit" %in% installed.packages()[,"Package"]) {
  figD4 <- arphitgg(aes = agg_aes(x = horizon, y = value, group = variable), layout = "2b2"
                   , dropxlabel = TRUE, showallxlabels = FALSE) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ln_cpii') & str_detect(variable, 'rr_fw_cs1|rr_fw_csj', negate=T)), 
             panel = "1", lty = c(2,1,1,1,1,2), lwd = rep(3,6), 
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Olive1"], RBA["Orange3"], RBA["Red9"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ln_cpii') & str_detect(variable, 'rr_fw_cs1|rr_fw_csj')), 
             panel = "2", lty = c(2,1,1,2), lwd = rep(3,4), 
             colour = c(RBA["Red6"], RBA["Aqua5"], RBA["Aqua8"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_cs1|rr_fw_csj', negate=T)), 
             panel = "3", lty = c(2,1,1,1,1,2), lwd = rep(3,6), 
             colour = c(RBA["Red6"], RBA["Aqua8"], RBA["Olive1"], RBA["Orange3"], RBA["Red9"], RBA["Red6"])) + 
    agg_line(data = IRF_plot_lg %>% filter(str_detect(variable, 'ur') & str_detect(variable, 'rr_fw_cs1|rr_fw_csj')), 
             panel = "4", lty = c(2,1,1,2), lwd = rep(3,4), 
             colour = c(RBA["Red6"], RBA["Aqua5"], RBA["Aqua8"], RBA["Red6"])) + 
    agg_yaxislabel("CPI (Underlying)", panel = "1") + 
    agg_yaxislabel("Unemployment", panel = "3") + 
    agg_title("No money market spread**", panel = "1") +
    agg_title("Incl. money market spread***", panel = "2") +
    agg_units("ppt", panel = c("3","4")) +
    agg_ylim(-3, 3, 7, panel = c("1","2")) +
    agg_ylim(-2, 2, 5, panel = c("3","4")) +
    agg_rename_series(list("BT" = "ln_cpii_rr_fw", "BT-CS2" = "ln_cpii_rr_fw_cs2",
                           "BT-CS3" = "ln_cpii_rr_fw_cs3", "BT-CS4" = "ln_cpii_rr_fw_cs4",
                           "BT-CS" = "ln_cpii_rr_fw_csj", "BT-CS1" = "ln_cpii_rr_fw_cs1", 
                           "<NA>" = "ln_cpii_lwr_rr_fw", "<NA>" = "ln_cpii_upr_rr_fw",
                           "<NA>" = "ln_cpii_lwr_rr_fw_csj", "<NA>" = "ln_cpii_upr_rr_fw_csj",
                           "<NA>" = "ur_rr_fw", "<NA>" = "ur_rr_fw_cs2",
                           "<NA>" = "ur_rr_fw_cs3", "<NA>" = "ur_rr_fw_cs4",
                           "<NA>" = "ur_rr_fw_cs1", "<NA>" = "ur_rr_fw_csj", 
                           "<NA>" = "ur_lwr_rr_fw", "<NA>" = "ur_upr_rr_fw",
                           "<NA>" = "ur_lwr_rr_fw_csj", "<NA>" = "ur_upr_rr_fw_csj")) + 
    agg_legend(x=0.4, y=0.61) + 
    agg_title("Responses to Cash Rate Tightening") +
    agg_subtitle("Cumulative quarterly response to 100 bps policy shock*") +
    agg_source("ABS") +
    agg_source("Author's calculations") +
    agg_source("RBA") + 
    agg_footnote("Impulse responses from the benchmark recursive SVAR with one of the policy shocks from Table 1. See Figure 7 for further notes.") + 
    agg_footnote("Bishop and Tulip (BT, 2017, updated) policy shock with confidence intervals or BT Shock accounting for cash rate response to domestic lending rate spread (BT-CS2), US corporate bond spread (BT-CS3), or US VIX (BT-CS4).") + 
    agg_footnote("BT Shock accounting for cash rate response to all credit spread measures (BT-CS) with confidence intervals, or BT Shock accounting for cash rate response to domestic money market spread (BT-CS1).")
  agg_draw(figD4, filename = paste0(results_dir, "FigD4_SVAR_all_BTCS_shocks.png"))
}

### Export graph data
IRFs_all_shks <- mutate(IRF_plot_lg, panel = case_when((variable=='ln_cpii_rr_fw') ~ 1, 
                                                       (variable=='ln_cpii_lwr_rr_fw') ~ 1, 
                                                       (variable=='ln_cpii_upr_rr_fw') ~ 1, 
                                                       (variable=='ln_cpii_rr_fw_cs2') ~ 1, 
                                                       (variable=='ln_cpii_rr_fw_cs3') ~ 1, 
                                                       (variable=='ln_cpii_rr_fw_cs4') ~ 1,  
                                                       (variable=='ln_cpii_rr_fw_csj') ~ 2, 
                                                       (variable=='ln_cpii_lwr_rr_fw_csj') ~ 2, 
                                                       (variable=='ln_cpii_upr_rr_fw_csj') ~ 2, 
                                                       (variable=='ln_cpii_rr_fw_cs1') ~ 2,  
                                                       (variable=='ur_rr_fw') ~ 3, 
                                                       (variable=='ur_lwr_rr_fw') ~ 3, 
                                                       (variable=='ur_upr_rr_fw') ~ 3, 
                                                       (variable=='ur_rr_fw_cs2') ~ 3, 
                                                       (variable=='ur_rr_fw_cs3') ~ 3, 
                                                       (variable=='ur_rr_fw_cs4') ~ 3,  
                                                       (variable=='ur_rr_fw_csj') ~ 4, 
                                                       (variable=='ur_lwr_rr_fw_csj') ~ 4, 
                                                       (variable=='ur_upr_rr_fw_csj') ~ 4, 
                                                       TRUE ~ 4)) %>% 
  arrange(panel, variable, horizon) %>% 
  mutate(label = case_when((variable=='ln_cpii_rr_fw') ~ "CPI (BT Shock)", 
                           (variable=='ln_cpii_lwr_rr_fw') ~ "CPI Lower (BT Shock)", 
                           (variable=='ln_cpii_upr_rr_fw') ~ "CPI Upper (BT Shock)", 
                           (variable=='ln_cpii_rr_fw_cs2') ~ "CPI (BT-CS2 Shock)", 
                           (variable=='ln_cpii_rr_fw_cs3') ~ "CPI (BT-CS3 Shock)", 
                           (variable=='ln_cpii_rr_fw_cs4') ~ "CPI (BT-CS4 Shock)",  
                           (variable=='ln_cpii_rr_fw_csj') ~ "CPI (BT-CS Shock)", 
                           (variable=='ln_cpii_lwr_rr_fw_csj') ~ "CPI Lower (BT-CS Shock)", 
                           (variable=='ln_cpii_upr_rr_fw_csj') ~ "CPI Upper (BT-CS Shock)", 
                           (variable=='ln_cpii_rr_fw_cs1') ~ "CPI (BT-CS1 Shock)",  
                           (variable=='ur_rr_fw') ~ "UR (BT Shock)", 
                           (variable=='ur_lwr_rr_fw') ~ "UR Lower (BT Shock)", 
                           (variable=='ur_upr_rr_fw') ~ "UR Upper (BT Shock)", 
                           (variable=='ur_rr_fw_cs2') ~ "UR (BT-CS2 Shock)", 
                           (variable=='ur_rr_fw_cs3') ~ "UR (BT-CS3 Shock)", 
                           (variable=='ur_rr_fw_cs4') ~ "UR (BT-CS4 Shock)",  
                           (variable=='ur_rr_fw_csj') ~ "UR (BT-CS Shock)", 
                           (variable=='ur_lwr_rr_fw_csj') ~ "UR Lower (BT-CS Shock)", 
                           (variable=='ur_upr_rr_fw_csj') ~ "UR Upper (BT-CS Shock)", 
                           TRUE ~ "UR (BT-CS1 Shock)")) %>% 
  select(-c(variable, panel)) %>% 
  pivot_wider(names_from = label, values_from = value)
write_xlsx(IRFs_all_shks, path = paste(results_dir, "FigD4_SVAR_all_BTCS_shocks.xlsx", sep=""), col_names = TRUE)
