##### Preamble ################################################################
# Project: Credit Spreads, Monetary Policy, and the Price Puzzle
# File: Load Australian Money Market Data
# Creator: Ben Beckers
# Date: 09 December 2019
# Description: This file reads and saves Australian money market data
###############################################################################


##### Step 1: Load Money Market Rates #########################################
### Download the historical data
tmp_file <- tempfile(fileext = ".xls")
download.file('https://www.rba.gov.au/statistics/tables/xls-hist/f01dhist.xls',
              tmp_file, mode = 'wb')

trash <- read_excel(tmp_file, col_names = T, skip = 10)
numcol <- ncol(trash)
mm_rates <- read_excel(tmp_file, col_names = T, skip = 10, 
                       col_types = c("date", rep("numeric", numcol-1)))

### Download the most recent data
tmp_file <- tempfile(fileext = ".xls")
download.file('https://www.rba.gov.au/statistics/tables/xls/f01d.xls',
              tmp_file, mode = 'wb')
trash <- read_excel(tmp_file, col_names = T, skip = 10)
numcol <- ncol(trash)
mm_rates_new <- read_excel(tmp_file, col_names = T, skip = 10, 
                           col_types = c("date", rep("numeric", numcol-1)))
rm(trash, numcol, tmp_file)

### Bind the data
mm_rates <- bind_rows(mm_rates, mm_rates_new)

### Clean the Data
mm_rates <- mm_rates %>%
  rename(Date = `Series ID`) %>%
  filter(Date>=smplstart) %>%
  filter_if(is.numeric, any_vars(!is.na(.))) %>% 
  mutate(Date = as_date(Date))
  

### Keep only selected variables and rename
mm_rates_names <- select(metadata, `Series ID`) %>% 
  filter(str_detect(`Series ID`, "FIRMM"))
mm_rates_names <- mm_rates_names[["Series ID"]]
mm_rates <- mm_rates %>%
  select(c("Date", mm_rates_names))
colnames(mm_rates)[2:ncol(mm_rates)] <- 
  lookup_rename(mm_rates, metadata, "Mnemonic", "Series ID")[["Mnemonic"]]



##### Step 2: Load Zero-Coupon Forward Rates ##################################
# Download the historical data
tmp_file <- tempfile()
download.file('https://www.rba.gov.au/statistics/tables/xls-hist/zcr-analytical-series-hist.xls',
              tmp_file, mode = 'wb')
zcf_rates <- read_excel(tmp_file, col_names = T, skip = 10)
zcf_rates <- zcf_rates %>% 
  rename(Date = Mnemonic) %>% 
  rename_if(is.numeric, list(~ str_replace(., "FZCY", "FZCF")))

### Download the most recent data
tmp_file <- tempfile()
download.file('https://www.rba.gov.au/statistics/tables/xls/f17hist.xls',
              tmp_file, mode = 'wb')
zcf_rates_new <- read_excel(tmp_file, col_names = T, skip = 10)
zcf_rates_new <- zcf_rates_new %>% 
  rename(Date = `Series ID`) %>% 
  rename_if(is.numeric, list(~ str_replace(., "FZCY", "FZCF")))

### Bind the data
zcf_rates <- bind_rows(zcf_rates, zcf_rates_new)
rm(zcf_rates_new)

### Clean the Data
zcf_rates <- zcf_rates %>%
  filter(Date>=smplstart) %>%
  filter_if(is.numeric, any_vars(!is.na(.))) %>% 
  mutate(Date = as_date(Date))

### Rename and keep selected variables
zcf_rates_names <- select(metadata, `Series ID`) %>% 
  filter(str_detect(`Series ID`, "FZCF"))
zcf_rates_names <- zcf_rates_names[["Series ID"]]
zcf_rates <- zcf_rates %>%
  select(c("Date", zcf_rates_names))
colnames(zcf_rates)[2:ncol(zcf_rates)] <- 
  lookup_rename(zcf_rates, metadata, "Mnemonic", "Series ID")[["Mnemonic"]]
rm(zcf_rates_names)


##### Step 3: Merge datasets ##################################################
mm_rates <- full_join(mm_rates, zcf_rates, by="Date")
mm_rates <- mm_rates %>% 
  arrange(Date) %>% 
  filter_if(is.numeric, any_vars(!is.na(.)))

# Fill in zero-coupon forward yield for missing OIS
mm_rates <- mm_rates %>% 
  mutate(ir_ois3 = ifelse(is.na(ir_ois3), ir_zcf3, ir_ois3)) %>% 
  mutate(ir_ois6 = ifelse(is.na(ir_ois6), ir_zcf6, ir_ois6))
rm(zcf_rates)

mm_rates_names <- names(mm_rates)


##### Step 2: Save data #######################################################
save(mm_rates, file = file.path(data_dir, "mm_rates.RData"))