/*******************************************************************************

	7_final_sample.do
	
	This file contains code to create the final set of variables required for
	the regression analysis in Section 4 of Bishop J and I Chan (2019), Is
	Declining Union Membership Contributing to Low Wages Growth?, RBA Research
	Discussion Paper No 2019-02. It also generates Figure 8 in the paper.
		
*******************************************************************************/

clear all
set more off   
set maxvar 10000

local dir  "<path for 'data' folder here>"

cd "`dir'"

/*********************************************************************************************
  This part of the program creates two new variables:
  
  1. Share of union-negotiated EBAs in each industry at each point in time

  2. Share of EBAs in each industry at each point in time with non-quantifiable AAWI
     [Excluded instrument in Heckman]
*********************************************************************************************/ 

use wad_extract_final, clear 

* Indicator for quantifiable AAWI

gen     quant = 0
replace quant = 1 if aawi!=.

gen first = min(cert_date, comm_date, firs_incr)
gen last  = min(term_date, max(expi_date, last_incr))

format first %td
format last  %td

* if start date is after finish date, set finish date to start date
replace last=first if (last-first)<0

keep union first last surveyq anzsic93 anzsic06 employees_combined quant eba_id

forvalues z = 1991(1)2017 {

foreach c in jun sep {
  gen byte active_`c'`z' = 0 
  replace  active_`c'`z' = 1 if first<=td(30`c'`z') & last>=td(30`c'`z') & first!=. & last!=.
 }
foreach p in mar dec {
  gen byte active_`p'`z' = 0 
  replace  active_`p'`z' = 1 if first<=td(31`p'`z') & last>=td(31`p'`z') & first!=. & last!=.
 }
}
*
compress

save wad_extract_final_working, replace 

foreach p in 93 06 {

use wad_extract_final_working, clear

drop if anzsic`p'==.

foreach i in mar jun sep dec {

forvalues z = 1992(1)2017 {

use wad_extract_final_working, clear

keep if active_`i'`z'==1 

collapse (mean) union_eba_ind`p'_rate=union quant_eba_ind`p'_rate=quant [weight=employees_combined], by(anzsic`p')

gen q = "`i'"
gen y = `z'

save un_anz`p'_`i'_`z', replace
}
}
*
use un_anz`p'_mar_1992, clear

foreach i in mar jun sep dec {
forvalues z = 1992(1)2017 {
append using un_anz`p'_`i'_`z'
erase un_anz`p'_`i'_`z'.dta
}
}
*
gen qtr = 1 if q=="mar"
replace qtr = 2 if q=="jun"
replace qtr = 3 if q=="sep"
replace qtr = 4 if q=="dec"

gen surveyq = yq(y, qtr)
format surveyq %tq

drop q y qtr

sort surveyq anzsic`p'

duplicates drop

drop if anzsic`p'==. | surveyq==.

xtset anzsic`p' surveyq

gen surveyq_l1 = surveyq-1
format surveyq_l1 %tq

drop surveyq

drop if surveyq_l1==.

save union_eba_ind_rate`p', replace

clear
}
*
erase wad_extract_final_working.dta

use wad_extract_final, clear

* merge based on lagged surveyq

gen surveyq_l1 = surveyq

foreach p in 93 06 {
merge m:1 anzsic`p' surveyq_l1 using union_eba_ind_rate`p'
drop if _merge==2
drop    _merge
}
*

foreach i in union quant {
gen `i'_eba_ind_rate = `i'_eba_ind06_rate
replace `i'_eba_ind_rate = `i'_eba_ind93_rate if `i'_eba_ind_rate==.
drop `i'_eba_ind06_rate `i'_eba_ind93_rate
}

/*********************************************************************************************
* Variables measuring agreement types
 *******************************************************************************************/ 
 
* Agreement types
gen AgreementType=.
label variable AgreementType "Agreement Type"

// Fair Work Act 2009
replace AgreementType=.  if eba_type=="DK (NOT YET ALLOCATED)"
replace AgreementType=11 if eba_type=="Single enterprise  non-greenfields"
replace AgreementType=12 if eba_type=="Single enterprise greenfield"
replace AgreementType=13 if eba_type=="Multi enterprise non-greenfield"
replace AgreementType=14 if eba_type=="Multi enterprise greenfield"

// Work Choices
replace AgreementType=21 if eba_type=="Union CA (Division 2)" // Post-WC WRA1996 ss 328[-9]
replace AgreementType=22 if eba_type=="Non-union CA (Division 2)" // Post-WC WRA1996 s 327
replace AgreementType=23 if eba_type=="Greenfield CA (Division 2)" // Post-WC WRA1996 ss 329
replace AgreementType=24 if eba_type=="Employer greenfields" // Post-WC WRA1996 s 330

// Workplace Relations Act 1996 (pre-Work Choices)
replace AgreementType=31 if eba_type=="LJ Union CA (Division 2)" // s170LJ WRA1996, before March 2006 amendments
replace AgreementType=32 if eba_type=="LK Non-union CA (Division 2)" // s170LK WRA1996, before March 2006 amendments
replace AgreementType=33 if eba_type=="LL Greenfield CA (Division 2)" // s170LL WRA1996, before March 2006 amendments
replace AgreementType=34 if eba_type=="Division 3 CA" // ss170LN-LS WRA1996, before March 2006 amendments, industrial dispute, 1997-2006

// Industrial Relations Act 1988
replace AgreementType=41 if eba_type=="Cert Agrt (s170Ma & s170Mb & s170Mc)" // IRA1988, 1991-98
replace AgreementType=42 if eba_type=="EFA (s170Na)" // IRA1988, 1994-1997
replace AgreementType=43 if eba_type=="Flexibility Clause" // IRA1988, 1992-1996
replace AgreementType=44 if eba_type=="Cert Agrt (Div 3A)" // IRA1988, 1992-95
replace AgreementType=45 if eba_type=="Cert Agrt (s115)" // IRA1988, 1991-95
replace AgreementType=46 if eba_type=="Consent Awd (s111(1)(b))" // IRA1988, 1992-95
replace AgreementType=47 if eba_type=="Consent Awd (s112)" // IRA1988, 1992-95

label define AgreementType ///
     11 "Single enterprise non-greenfield" ///
     12 "Single enterprise greenfield" ///
     13 "Multi-enterprise non-greenfield" ///
     14 "Multi-enterprise greenfield " ///
     ///
     21 "Union (WRA post-WC s328)" ///
     22 "Non-union (WRA post-WC s327)" ///
     23 "Union greenfields (WRA post-WC s329)" ///
     24 "Employer greenfield (WRA post-WC s330)" ///
     ///
     31 "Union (WRA pre-WC s170LJ)" ///
     32 "Non-union (WRA pre-WC s170LK)" ///
     33 "Greenfield (WRA pre-WC s170LL)" ///
     34 "Division 3 (WRA pre-WC ss170LN-LS)" ///
     ///
     41 "Cert agrt (IRA ss170MA-170MC)" ///
     42 "Enterprise Flexibility Agreements (IRA s170NA)" ///
     43 "Flexibility clause (IRA)" ///
     44 "Cert Agrt (IRA Div 3A)" ///
     45 "Cert Agrt (IRA s115)" ///
     46 "Consent Awd (IRA s111(1)(b))" ///
     47 "Consent Awd (IRA s112)"
label values AgreementType AgreementType
*drop AgreementTypeName

recode AgreementType ///
     (11/14 = 1) ///
     (21/24 = 2) ///
     (31/34 = 3) ///
     (41/47 = 4), gen(Law)
label variable Law "Legal frameowrk for agreement"
label define Law ///
     1 "Fair Work Act 2009" ///
     2 "Work Choices" ///
     3 "Workplace Relations Act 1996 (pre-Work Choices)" ///
     4 "Industrial Relations Act 1988"
label values Law Law

cap drop AgreementTypeGreenfields
recode AgreementType ///
	(12 14 23 24 33 = 1) ///
	(11 13 21 22 31 32 34 = 0) ///
	(41 42 43 44 45 46 47 = .), gen(AgreementTypeGreenfields)
label variable AgreementTypeGreenfields "Greenfields agreement"
label values AgreementTypeGreenfields yesno

tab Law, gen(law_)

rename law_1 fwa
rename law_2 wc
rename law_3 wra
rename law_4 ira

gen     ir_regime = "ira" if ira==1
replace ir_regime = "wra" if wra==1
replace ir_regime = "wc"  if wc==1
replace ir_regime = "fwa" if fwa==1

encode ir_regime , generate(ir_regime_cat)

rename AgreementTypeGreenfields gf

/*********************************************************************************************
* Define key variables
 *******************************************************************************************/
 
gen ln_emp = ln(employees_combined)

gen     multi_firm = 0 if countofemployer!=.
replace multi_firm = 1 if countofemployer>1 & countofemployer!=.

egen ind_state = group(anzsic_1dig state_code)
 
* Dummy = 1 if EBA is 'short' duration (<1 year)

gen short_duration = 1 if (max(expi_date, last_incr)-min(cert_date, comm_date, firs_incr))/365 < 1

* Dummy = 1 if AAWI quantifiable = 0 otherwise

gen     quant = 0
replace quant = 1 if aawi!=.
 
g double roa = db_ebit/db_asset 
g double lsales = ln(db_sales)

* Identify outliers in D&B data

g outlier = 0
replace outlier = 1 if roa > 50
replace outlier = 1 if roa < -50
replace outlier = 1 if db_assets <= 0

/*********************************************************************************************
* Indicators for being in matched sample for FE models 

  Requires being matched and having non-missing data for at least two periods, also some agreements transition from pub to pri
  
  Seperate indicators for full sample and public/private sample
 *******************************************************************************************/

gen id_nm_tot     = id if aawi_a_effect!=. & union!=. & ln_emp!=. & multi_firm!=. & ind_state!=. & public!=. & short_duration!=1 & anzsic06_2dig!=. & surveyq!=.
gen id_nm_pub     = id if aawi_a_effect!=. & union!=. & ln_emp!=. & multi_firm!=. & ind_state!=. & public==1 & short_duration!=1 & anzsic06_2dig!=. & surveyq!=.
gen id_nm_pri     = id if aawi_a_effect!=. & union!=. & ln_emp!=. & multi_firm!=. & ind_state!=. & public==0 & short_duration!=1 & anzsic06_2dig!=. & surveyq!=.
gen id_nm_priheck = id if                    union!=. & ln_emp!=. & multi_firm!=. & ind_state!=. & public==0 & short_duration!=1 & anzsic06_2dig!=. & surveyq!=.

foreach i in tot pub pri priheck {

bysort id_nm_`i': egen id_c_`i' = count(id_nm_`i')

gen in_panel_`i' = id_c_`i'>=2 & id_c_`i'!=.

drop id_nm_`i' id_c_`i'
}
*
save wad_extract_final_working, replace

/*********************************************************************************************
* Examine transitions in union status (Fig 8)
 *******************************************************************************************/

keep if in_panel_pri==1 
 
drop fam_count

sort family_id start_date
by   family_id: gen fam_count = _n 

xtset id fam_count

count
xtdes

gen     u_u  = 0 if                          fam_count!=1
replace u_u  = 1 if L1.union==1 & union==1 & fam_count!=1

gen     u_n  = 0 if                          fam_count!=1
replace u_n  = 1 if L1.union==1 & union==0 & fam_count!=1

gen     n_n  = 0 if                          fam_count!=1
replace n_n  = 1 if L1.union==0 & union==0 & fam_count!=1 

gen     n_u  = 0 if                          fam_count!=1
replace n_u  = 1 if L1.union==0 & union==1 & fam_count!=1 

gen year = yofd(dofq(surveyq))

* Figure 8

tabstat u_n n_u, by(year)

* Samples average

tabstat u_n n_u

* Averages before and after Work Choices 

tabstat u_n n_u if surveyq<=tq(2009q2)
tabstat u_n n_u if surveyq> tq(2009q2)
tabstat u_n n_u if surveyq<=tq(2006q2)
tabstat u_n n_u if surveyq> tq(2006q2)

keep eba_id u_n n_u u_u n_n 

compress

save union_transitions, replace

use wad_extract_final_working, clear

merge 1:1 eba_id using union_transitions, nogen

erase union_transitions.dta
erase wad_extract_final_working.dta

sort id fam_count

save wad_extract_final_RDP,replace

clear

* end of do file
