/*******************************************************************************

	14_paysetting.do
	
	This file contains code that extracts data from ABS, Employee Earnings and
	Hours, Cat. No. 6306.0 and Department of Jobs and Small Business
	Workplace (2018), 'Trends in Federal Enterprise Bargaining Report', September
	Quarter To produce Figure 2 in Bishop J and I Chan (2019), Is Declining
	Union Membership Contributing to Low Wages Growth?, RBA Research Discussion
	Paper No 2019-02.
	Output will be saved to fig5.xlsx.

	The package labutil is needed for this code to run.
	
*******************************************************************************/

clear all
set more off

local dir  "<path for 'data' folder here>"

cd "`dir'"

* Import data extracted from the ABS EEH
import excel using abs_paysetting.xlsx, clear sheet(juris) firstrow
	
* Import and append DoJSB jurisdiction data for period from 2010
preserve // 
	import excel using dojsb_fedstatecov.xlsx, clear firstrow
	egen double doj_col_sh = total(empl_sh) if paymeth == "Collective agreement", by(date) // Generate variable with total collective agreements according to DoJSB

	* Adjust 2016 fed/state registration breakdown (based on original 2016
	* ABS 6306.0 release) to the revised ('indicative comparable estimate') data 
	* for collective agreements (and awards) in 2016 from the 2018 ABS 6306.0
	* release. That revision attributed here to state jurisdiction given that most
	* of the revision was from award to state-registered collective agreement.
	gen eeh_col_adj = 38.5 if yofd(date)==2016 & paymeth == "Collective agreement"
	replace empl_sh = empl_sh + (eeh_col_adj - doj_col_sh) if ///
		yofd(date)==2016 & paymeth=="Collective agreement" & juris=="State" & regis=="Registered"
	drop eeh* doj*
	egen double doj_col_sh = total(empl_sh) if paymeth == "Collective agreement", by(date) 
	order date, first
	tempfile dojsb_fedstatecov
	save "`dojsb_fedstatecov'", replace
restore

* Generate consolidated file with federal/state registration information from 2000
* onwards
preserve
	* Import and append cleaned DoJSB data
	append using "`dojsb_fedstatecov'"

	* Generate column of total employee numbers by year from EEH release
	egen double eeh_empl_all = total(empl * (juris=="All" & regis=="All" & paymeth=="All")) ///
		if missing(empl_sh), by(date)
	
	* Calculate missing shares
	replace empl_sh = empl / eeh_empl_all * 100 if missing(empl_sh) & !missing(empl)
	
	* Generate column of collective bargain share from EEH releas
	egen double eeh_col_sh = total(empl_sh * (juris=="All" & regis=="All" & paymeth=="Collective agreement")), ///
		by(date) // 
	gen double emp_sh_adj = empl_sh/doj_col_sh * eeh_col_sh
	replace empl_sh = emp_sh_adj if !missing(emp_sh_adj) & paymeth == "Collective agreement" & regis != "All"

	keep if !(juris=="All") & regis=="Registered" & paymeth == "Collective agreement"
	keep date juris paymeth empl_sh
	reshape wide empl_sh, i(date paymeth) j(juris) string
	gen paymeth2 = 2 if paymeth == "Collective agreement"
	drop paymeth
	tempfile temp_regis
	save "`temp_regis'", replace
restore

keep if (juris=="All" & regis=="All")
drop juris regis

* Generate column of total employee numbers by year from EEH release
egen double eeh_empl_all = total(empl * (paymeth=="All")) ///
	if missing(empl_sh), by(date)
* Calculate missing shares
replace empl_sh = empl / eeh_empl_all * 100 if missing(empl_sh) & !missing(empl)

* Labelling
tokenize `""Award only" "Collective agreement" "Individual arrangement" "Individual arrangement - ex OMIE" "OMIE" "All" "All - ex OMIE""'
cap gen paymeth2 = .
forval i = 1/7 {
	replace paymeth2 = `i' if paymeth=="``i''"
	label define paymeth2 `i' "``i''", add
}
label values paymeth2 paymeth2
drop paymeth eeh*

* Merge with cleaned federal/state breakdown file
merge 1:1 date paymeth2 using "`temp_regis'"
drop _merge
reshape wide empl*, i(date) j(paymeth2)
drop *Federal1 *Federal3 *Federal4 *Federal5 *Federal6 *State1 *State3 *State4 *State5 *State6 
rename (*1 *2 *3 *4 *5 *6) (*_awd *_col *_ind *_indexomie *_omie *_all)
rename (*Federal* *State*) (**_fed **_state)

order *, alpha
order date, first

gen year = yofd(date)

replace empl_ind = empl_indexomie + empl_omie if missing(empl_ind)
foreach v in awd col ind indexomie omie all {
	replace empl_sh_`v' = empl_`v' / empl_all * 100 if missing(empl_sh_`v')
}

* Generate ex-OMIE series.
* Because data on the OMIE share is not available in 2000 and 2002, we estimate
* the share for these years by using the 2004 OMIE share as is.
gen empl_allexomie = empl_all - empl_omie 
gen empl_shexomie_all = empl_sh_awd + empl_sh_col + empl_sh_indexomie

cap drop omie*
egen double omie_sh_est04 = total(empl_sh_omie * (year==2004))
replace empl_sh_omie = omie_sh_est04 if inrange(year, 2000, 2002)
replace empl_sh_indexomie = empl_sh_ind - empl_sh_omie if inrange(year, 2000, 2002)
replace empl_shexomie_all = 100 - empl_sh_omie if inrange(year, 2000, 2002)

foreach v in awd col indexomie { 
	gen empl_shexomie_`v' = empl_`v' / empl_allexomie * 100 
	replace empl_shexomie_`v' = empl_sh_`v' / empl_shexomie_all * 100 if missing(empl_shexomie_`v')
}

gen double empl_shexomie_col_fed = empl_sh_col_fed / empl_shexomie_all * 100
gen double empl_shexomie_col_state = empl_sh_col_state / empl_shexomie_all * 100
gen double empl_sh_col_unreg = empl_sh_col - empl_sh_col_fed - empl_sh_col_state
gen double empl_shexomie_col_unreg = empl_shexomie_col - empl_shexomie_col_fed - empl_shexomie_col_state

foreach v of varlist * {
	label variable `v' ""
}

* Labelling
labvarch *all*, prefix("All")
labvarch *awd*, prefix("Award")
labvarch *col*, prefix("Collective agreement")
labvarch *ind*, prefix("Individual arrangement")
labvarch *_omie, prefix("OMIE")
labvarch *_fed, postfix(", Federal registered")
labvarch *_state, postfix(", state registered")
labvarch *_unreg, postfix(", other")
labvarch *exomie*, postfix(" (ex OMIE)")

keep year date empl_shexomie*
drop *all
order *, alpha
order year date, first

save paysetting, replace

* Export to Excel
export excel year empl* using fig2.xlsx, ///
	sheet("fig2") sheetreplace firstrow(varlabels)

* end of do file
	
